% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kfa.R
\name{kfa}
\alias{kfa}
\title{Conducts k-fold cross validation for factor analysis}
\usage{
kfa(
  data,
  variables = names(data),
  k = NULL,
  m = floor(length(variables)/4),
  seed = 101,
  cores = NULL,
  custom.cfas = NULL,
  power.args = list(rmsea0 = 0.05, rmseaA = 0.08),
  rotation = "oblimin",
  simple = TRUE,
  min.loading = NA,
  ordered = FALSE,
  estimator = NULL,
  missing = "listwise",
  ...
)
}
\arguments{
\item{data}{a \code{data.frame} containing the variables (i.e., items) to factor analyze}

\item{variables}{character vector of column names in \code{data} indicating the variables to factor analyze. Default is to use all columns.}

\item{k}{number of folds in which to split the data. Default is \code{NULL} which determines k via \code{\link[kfa]{find_k}}.}

\item{m}{integer; maximum number of factors to extract. Default is 4 items per factor.}

\item{seed}{integer passed to \code{set.seed} when randomly selecting cases for each fold.}

\item{cores}{integer; number of CPU cores to use for parallel processing. Default is \code{\link[parallel]{detectCores}} - 1.}

\item{custom.cfas}{a single object or named \code{list} of \code{lavaan} syntax specifying custom factor model(s).}

\item{power.args}{named \code{list} of arguments to pass to \code{\link[kfa]{find_k}} and \code{\link[semTools]{findRMSEAsamplesize}} when conducting power analysis to determine \code{k}.}

\item{rotation}{character (case-sensitive); any rotation method listed in
\code{\link[GPArotation]{rotations}} in the \code{GPArotation} package. Default is "oblimin".}

\item{simple}{logical; Should the simple structure be returned (default) when converting EFA results to CFA syntax?
If \code{FALSE}, items can cross-load on multiple factors.}

\item{min.loading}{numeric between 0 and 1 indicating the minimum (absolute) value of the loading for a variable on a factor
when converting EFA results to CFA syntax. Must be specified when \code{simple = FALSE}.}

\item{ordered}{logical; Should items be treated as ordinal and the
polychoric correlations used in the factor analysis? When \code{FALSE} (default)
the Pearson correlation matrix is used. A character vector of item names is
also accepted to prompt estimation of the polychoric correlation matrix.}

\item{estimator}{if \code{ordered = FALSE}, the default is "MLMVS". If
\code{ordered = TRUE}, the default is "WLSMV". See \code{\link[lavaan]{lavOptions}} for other options.}

\item{missing}{default is "listwise". See \code{\link[lavaan]{lavOptions}} for other options.}

\item{...}{other arguments passed to \code{lavaan} functions. See \code{\link[lavaan]{lavOptions}}.}
}
\value{
An object of class \code{"kfa"}, which is a four-element \code{list}:
\itemize{
\item \strong{cfas} \code{lavaan} CFA objects for each \emph{k} fold
\item \strong{cfa.syntax} syntax used to produce CFA objects
\item \strong{model.names} vector of names for CFA objects
\item \strong{efa.structures} all factor structures identified in the EFA
}
}
\description{
The function splits the data into \emph{k} folds where each fold contains training data and test data.
For each fold, exploratory factor analyses (EFAs) are run on the training data. The structure for each model
is transformed into \code{lavaan}-compatible confirmatory factor analysis (CFA) syntax.
The CFAs are then run on the test data.
}
\details{
In order for \code{custom.cfas} to be tested along with the EFA identified structures, each model supplied in \code{custom.cfas} must
include all \code{variables} in \code{lavaan}-compatible syntax.

Deciding an appropriate \emph{m} can be difficult, but is consequential for the possible factor structures to
examine, the power analysis to determine \emph{k}, and overall computation time.
The \code{n_factors} function in the \code{parameters} package can assist with this decision.

When converting EFA results to CFA syntax (via \code{\link[kfa]{efa_cfa_syntax}}), the simple structure is
defined as each variable loading onto a single factor. This is determined using the largest factor loading for each variable.
When \code{simple = FALSE}, variables are allowed to cross-load on multiple factors. In this case, all pathways with loadings
above the \code{min.loading} are retained. However, allowing cross-loading variables can result in model under-identification.
The \code{\link[kfa]{efa_cfa_syntax}}) function conducts an identification check (i.e., \code{identified = TRUE}) and
under-identified models are not run in the CFA portion of the analysis.
}
\examples{

# simulate data based on a 3-factor model with standardized loadings
sim.mod <- "f1 =~ .7*x1 + .8*x2 + .3*x3 + .7*x4 + .6*x5 + .8*x6 + .4*x7
                f2 =~ .8*x8 + .7*x9 + .6*x10 + .5*x11 + .5*x12 + .7*x13 + .6*x14
                f3 =~ .6*x15 + .5*x16 + .9*x17 + .4*x18 + .7*x19 + .5*x20
                f1 ~~ .2*f2
                f2 ~~ .2*f3
                f1 ~~ .2*f3
                x9 ~~ .2*x10"
set.seed(1161)
sim.data <- simstandard::sim_standardized(sim.mod, n = 900,
                                          latent = FALSE,
                                          errors = FALSE)[c(2:9,1,10:20)]

# include a custom 2-factor model
custom2f <- paste0("f1 =~ ", paste(colnames(sim.data)[1:10], collapse = " + "),
                   "\nf2 =~ ",paste(colnames(sim.data)[11:20], collapse = " + "))

\donttest{
mods <- kfa(data = sim.data,
            k = NULL, # prompts power analysis to determine number of folds
            cores = 2,
            custom.cfas = custom2f)
            }

}
