% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jamba-export.r
\name{writeOpenxlsx}
\alias{writeOpenxlsx}
\title{Export a data.frame to 'Excel' 'xlsx' format}
\usage{
writeOpenxlsx(
  x,
  file = NULL,
  wb = NULL,
  sheetName = "Sheet1",
  startRow = 1,
  startCol = 1,
  append = FALSE,
  headerColors = c("lightskyblue1", "lightskyblue2"),
  columnColors = c("aliceblue", "azure2"),
  highlightHeaderColors = c("tan1", "tan2"),
  highlightColors = c("moccasin", "navajowhite"),
  borderColor = "gray75",
  borderPosition = "BottomRight",
  highlightColumns = NULL,
  numColumns = NULL,
  fcColumns = NULL,
  lfcColumns = NULL,
  hitColumns = NULL,
  intColumns = NULL,
  pvalueColumns = NULL,
  numFormat = "#,##0.00",
  fcFormat = "#,##0.0",
  lfcFormat = "#,##0.0",
  hitFormat = "#,##0.0",
  intFormat = "#,##0",
  pvalueFormat = "[>0.01]0.00#;0.00E+00",
  numRule = c(1, 10, 20),
  fcRule = c(-6, 0, 6),
  lfcRule = c(-3, 0, 3),
  hitRule = c(-1.5, 0, 1.5),
  intRule = c(0, 100, 10000),
  pvalueRule = c(0, 0.01, 0.05),
  numStyle = c("#F2F0F7", "#B4B1D4", "#938EC2"),
  fcStyle = c("#4F81BD", "#EEECE1", "#C0504D"),
  lfcStyle = c("#4F81BD", "#EEECE1", "#C0504D"),
  hitStyle = c("#4F81BD", "#EEECE1", "#C0504D"),
  intStyle = c("#EEECE1", "#FDA560", "#F77F30"),
  pvalueStyle = c("#F77F30", "#FDC99B", "#EEECE1"),
  doConditional = TRUE,
  doCategorical = TRUE,
  colorSub = NULL,
  freezePaneColumn = 0,
  freezePaneRow = 2,
  doFilter = TRUE,
  fontName = "Arial",
  fontSize = 12,
  minWidth = getOption("openxlsx.minWidth", 8),
  maxWidth = getOption("openxlsx.maxWidth", 40),
  autoWidth = TRUE,
  colWidths = NULL,
  wrapCells = FALSE,
  wrapHeaders = TRUE,
  headerRowMultiplier = 5,
  keepRownames = FALSE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{\code{data.frame} to be saved to an 'Excel' 'xlsx' file.}

\item{file}{\code{character} valid path to save an 'Excel' 'xlsx' file. If the file
exists, and \code{append=TRUE} the new data will be added to the existing
file withthe defined \code{sheetName}.
\itemize{
\item Note when \code{file=NULL} the output is not saved to a file,
instead the \code{Workbook} object is returned by this function.
The \code{Workbook} object can be passed as argument \code{wb} in order
to add multiple sheets to the same Workbook prior to saving
them together. This operation is intended to provide a
substantial improvement in speed.
}}

\item{wb}{\code{Workbook} object as defined in R package \code{openxlsx}.
When this argument is defined, data is not imported from \code{file},
and instead the workbook data is used from \code{wb}. This option is
intended to improve speed of writing several sheets to the same
output file, by preventing the slow read/write steps each time
a new sheet is added.}

\item{sheetName}{\code{character} value less with a valid
'Excel' 'xlsx' worksheet name. At this time (version 0.0.29.900) the
sheetName is restricted to 31 characters, with no puntuation except
"-" and "_".}

\item{startRow, startCol}{\code{integer} indicating the row and column number
to start with the top,left cell written to the worksheet,
default are 1.}

\item{append}{\code{logical} default FALSE, whether to append to file (TRUE),
or to write over an existing file. The \code{append=TRUE} is useful when
adding a worksheet to an existing file.}

\item{headerColors, columnColors, highlightHeaderColors, highlightColors, borderColor, borderPosition}{default values for the 'Excel' worksheet background and border
colors. As of version 0.0.29.900, colors must use valid 'Excel'
color names.}

\item{highlightColumns, numColumns, fcColumns, lfcColumns, hitColumns, intColumns, pvalueColumns}{\code{integer} vector referring the column number in the input \code{data.frame}
\code{x} to define as each column type, as relevant.}

\item{numFormat, fcFormat, lfcFormat, hitFormat, intFormat, pvalueFormat}{\code{character} string with valid 'Excel' cell formatting, for example
\code{"#,##0.00"} defines a column to use comma-delimited numbers above
one thousand, and display two decimal places in all numeric cells.
See \verb{[https://support.microsoft.com]} topic
\code{"Excel Create and apply a custom number format."} or
\code{"Excel Number format codes"} for more details. Some examples below:
\itemize{
\item \code{"#,##0"} : display only integer values, using comma as delimiter
for every thousands place.
The number \code{2142.12} would be represented: \code{"2,142"}
\item \code{"###0.0"} : display numeric values rounded to the \code{0.1} place,
using no comma delimiter for values above one thousand.
The number \code{2142.12} would be represented: \code{"2142.1"}
\item \code{"[>0.01]0.00#;0.00E+00"} : this rule is a conditional format,
values above \code{0.01} are represented as numbers rounded to the
thousandths position \code{0.001}; values below \code{0.01} are represented
with scientific notation with three digits.
The number \code{0.1256} would be represented: \code{"0.126"}
The number \code{0.001256} would be represented: \code{"1.26E-03"}
\item \code{"[Red]#,###.00_);[Blue](#,###.00);[Black]0.00_)"} : this format applies
to positive values, negative values, and zero, in order delimited
by semicolons. Positive values are colored red.
The string \code{"_)"} adds whitespace (defined by \code{"_"})
equale to the width of the character \code{")"} to the end
of positive values.
Negative values are surrounded by parentheses \code{"()"}
and are colored blue.
Values equal to zero are represented with two trailing digits,
and whitespace (\code{"_"}) equal to width \code{")"}.
The whitespace at the end of positive values and zero are used
to align all values at the same decimal position.
}}

\item{numRule, fcRule, lfcRule, hitRule, intRule, pvalueRule}{\code{numeric} vector
\code{length=3} indicating the breakpoints for 'Excel' to apply conditional
color formatting, using the corresponding style.
Note that all conditional formatting applied by this function uses
the \code{"3-Color Scale"}, therefore there should be three values,
and three corresponding colors in the corresponding Style arguments.}

\item{numStyle, fcStyle, lfcStyle, intStyle, hitStyle, pvalueStyle}{\code{character}
vector \code{length=3} containing three valid R colors. Note that alpha
transparency will be removed prior to use in 'Excel', as required.
Note that all conditional formatting applied by this function uses
the \code{"3-Color Scale"}, therefore there should be three colors,
which match three values in the corresponding Rule arguments.}

\item{doConditional}{\code{logical} indicating whether to apply conditional
formatting of cells, with this function only the background cell
color (and contrasting text color) is affected.}

\item{doCategorical}{\code{logical} indicating whether to apply categorical
color formatting, of only the background cell colors and contrasting
text color. This argument requires \code{colorSub} be defined.}

\item{colorSub}{\code{character} vector of R colors, whose names refer to
cell values in the input \code{x} data.frame.}

\item{freezePaneColumn, freezePaneRow}{\code{integer} value of the row or
column before which the 'Excel' "freeze panes" is applied.
Note that these values are adjusted relative by \code{startRow} and
\code{startCol} in the 'Excel' worksheet, so that the values are applied
relative to the \code{data.frame} argument \code{x}.}

\item{doFilter}{\code{logical} indicating whether to enable column
filtering by default.}

\item{fontName}{\code{character} default font configuration, containing
a valid 'Excel' font name.}

\item{fontSize}{\code{numeric} default font size in 'Excel' point units.}

\item{minWidth, maxWidth, autoWidth}{\code{numeric} minimum, maximum size
for each 'Excel' cell, in character units as defined by 'Excel',
used when \code{autoWidth=TRUE} to restrict cell widths to this range.
Note that the argument \code{colWidths} is generally preferred, if the
numeric widths can be reasonable calculated or anticipated upfront.
When \code{autoWidth=FALSE} 'Excel' typically auto-sizes cells to the width
of the largest value in each column, which may not be ideal when
values are extremely large.}

\item{colWidths}{\code{numeric} width of each column in \code{x}, recycled
to the total number of columns required. Note that when
\code{keepRownames=TRUE}, the first column will contain \code{rownames(x)},
therefore the length of \code{colWidths} in that case will be
\code{ncol(x) + 1}.}

\item{wrapCells}{\code{logical} default FALSE, indicating whether to
enable word-wrap within cells.}

\item{wrapHeaders}{\code{logical} indicating whether to enable word wrap
for column headers, which is helpful when \code{autoWidth=TRUE} since
it fixed the cell width while allowing the column header to be seen.}

\item{headerRowMultiplier}{\code{numeric} value to define the row height of
the first header row in 'Excel'. This value is defined as a multiple
of subsequent rows, and should usually represent the maximum number
of lines after word-wrapping, as relevant. This argument
is helpful when \code{wrapHeaders=TRUE} and \code{autoWidth=TRUE}.}

\item{keepRownames}{\code{logical} indicating whether to include
\code{rownames(x)} in its own column in 'Excel'.}

\item{verbose}{\code{logical} indicating whether to print verbose output.}

\item{...}{additional arguments are passed to \code{applyXlsxConditionalFormat()}
and \code{applyXlsxCategoricalFormat()} as relevant.}
}
\value{
\code{Workbook} object as defined by the \code{openxlsx} package
is returned invisibly with \code{invisible()}. This \code{Workbook}
can be used in argument \code{wb} to provide a speed boost when
saving multiple sheets to the same file.
}
\description{
Export a data.frame to 'Excel' 'xlsx' format
}
\details{
This function is a minor but useful customization of the
\code{openxlsx::saveWorkbook()} and associated functions, intended
to provide some pre-configured formatting of known column
types, typically relevant to statistical values, and
in some cases, gene or transcript expression values.

There are numerous configurable options when saving an 'Excel'
worksheet, most of the defaults in this function are intended
not to require changes, but are listed as formal function
arguments to make each option visibly obvious.

If \code{colorSub} is supplied as a named vector of colors, then
by default text values will be colorized accordingly, which
can be especially helpful when including data with categorical
text values.

This function pre-configures formatting options for the following
column data types, each of which has conditional
color-formatting, defined numeric ranges, and color scales.

\describe{
\item{int}{integer values, where numeric values are formatted
without visible decimal places, and the \code{big.mark=","} standard
is used to help visually distinguish large integers. The color
scale is by default c(0, 100, 10000).}
\item{num}{numeric values, with fixed number of visible decimal
places, which helps visibly align values along each row.}
\item{hit}{numeric type, a subset of "int" intended when data
is flagged with something like a "+1" or "-1" to indicate a
statistical increase or decrease.}
\item{pvalue}{P-value, where numeric values range from 1 down
near zero, and values are formatted consistently with scientific
notation.}
\item{fc}{numeric fold change, whose values are expected to range
from 1 and higher, and -1 and lower. Decimal places are by default
configured to show one decimal place, to simplify the 'Excel' visual
summary.}
\item{lfc}{numeric log fold change, whose values are expected to be
centered at zero. Decimal places are by default
configured to show one decimal place, to simplify the 'Excel' visual
summary.}
\item{highlight}{character and undefined columns to be highlighted
with a brighter background color, and bold text.}
}

For each column data type, a color scale and default numeric range
is defined, which allows conditional formatting of cells based upon
expected ranges of values.

A screenshot of the file produced by the example is shown below.

\if{html}{\figure{writeopenxlsx.png}{xlsx screenshot}}

\if{latex}{\figure{writeopenxlsx.png}{options: width=0.5in}}
}
\examples{
# set up a test data.frame
set.seed(123);
lfc <- -3:3 + stats::rnorm(7)/3;
colorSub <- nameVector(
   rainbow2(7),
   LETTERS[1:7])
df <- data.frame(name=LETTERS[1:7],
   int=round(4^(1:7)),
   num=(1:7)*4-2 + stats::rnorm(7),
   fold=2^abs(lfc)*sign(lfc),
   lfc=lfc,
   pvalue=10^(-1:-7 + stats::rnorm(7)),
   hit=sample(c(-1,0,0,1,1), replace=TRUE, size=7));
df;
# write to tempfile for examples
if (check_pkg_installed("openxlsx")) {
   out_xlsx <- tempfile(pattern="writeOpenxlsx_", fileext=".xlsx")
   writeOpenxlsx(x=df,
      file=out_xlsx,
      sheetName="jamba_test",
      colorSub=colorSub,
      intColumns=2,
      numColumns=3,
      fcColumns=4,
      lfcColumns=5,
      pvalueColumns=6,
      hitColumn=7,
      freezePaneRow=2,
      freezePaneColumn=2,
      append=FALSE);
   # now read it back
   df_list <- readOpenxlsx(xlsx=out_xlsx);
   sdim(df_list)
}

}
\seealso{
Other jam export functions: 
\code{\link{applyXlsxCategoricalFormat}()},
\code{\link{applyXlsxConditionalFormat}()},
\code{\link{readOpenxlsx}()},
\code{\link{set_xlsx_colwidths}()},
\code{\link{set_xlsx_rowheights}()}
}
\concept{jam export functions}
