% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsss.R
\name{prob_sum_score}
\alias{prob_sum_score}
\title{Calculate summed-score probabilities}
\usage{
prob_sum_score(ip, theta, theta_pdf = NULL)
}
\arguments{
\item{ip}{An \code{\link{Itempool-class}} object. Item pool parameters can
be composed of any combination of unidimensional dichotomous or polytomous
items.}

\item{theta}{A numeric vector representing the theta values at which the sum
score probabilities will be calculated.}

\item{theta_pdf}{A numeric vector with the same length of \code{theta}
argument representing the density values of each theta value. The
resulting probabilities will be weighted by these values. The default
value is \code{NULL} where the resulting probabilities will not be
weighted.}
}
\value{
A matrix containing the probabilities of each possible sum score.
  Each row represent a sum score and each column represent the theta value
  provided by \code{theta} argument.
}
\description{
This function calculates all summed-score probabilities
  of a given theta value(s) using recursive algorithm described in
  Thissen, Pommerich, Billeaud and Williams (1995). This function is the
  extension of the recursive algorithm proposed by Lord and Wingersky (1984)
  to polytomous items.
}
\examples{
### Example with weighting ###
ip <- generate_ip(model = sample(c("GPCM", "2PL"), 10, TRUE))
theta <- c(-3, -1.2, 0.5, 3)
prob_sum_score(ip, theta = theta)
# Most probable sum scores:
apply(prob_sum_score(ip, theta = theta), MARGIN = 2, which.max) - 1
\dontrun{
   plot(ip, type = "tcc", suppress_plot = TRUE) +
     ggplot2::geom_vline(xintercept = theta, lty = "dashed")
}
### Example from Kolen and Brennan (2014) ###
# Item parameters from Kolen and Brennan (2014), p.175, Table 6.1.
ip <- itempool(a = c(1.30, .6, 1.7),
               b = c(-1.30, -.10, .9),
               c = c(.1, .17, .18),
               D = 1.7)
prob(ip, theta = c(-2, 1))
# IRT observed score distribution using recursive formula from
# Kolen and Brennan (2014), p.200, Table 6.4.
# Numbers are not exactly the same as Kolen and Brennan since due to
# rounding applied to the numbers in the book.
prob_sum_score(ip, theta = -2)


### Example from Thissen, Pommerich, Billeaud and Williams (1995) ###
# Replicating Thissen et al. (1995) example, p.43-44, Table 1.
i1 <- item(a = .5, b = -1)
i2 <- item(a = 1, b = 0)
i3 <- item(a = 1.5, b = 1)
ip <- c(i1, i2, i3) # combine items to form an item pool
theta <- -3:3 # Quadrature points

prob_sum_score(ip, theta)

# Item parameters in Table 2
i1 <- item(a = 1.87, b = c(.65, 1.97, 3.14), model = "GRM")
i2 <- item(a = 2.66, b = c(.12, 1.57, 2.69), model = "GRM")
i3 <- item(a = 1.24, b = c(.08, 2.03, 4.30), model = "GRM")
ip <- c(i1, i2, i3)
delta <- 0.01
theta <- seq(-3, 3, delta)

x <- prob_sum_score(ip = ip, theta = theta, theta_pdf = dnorm(theta))

# Figure 1
plot(x = theta, y = x[2, ], type = "l", ylab = "Posterior Density",
     xlab = "Theta",
     main = paste0("Posterior Distribution for all Examinees Obtaining ",
                   "a Summed Score of 1"))

# Table 3, column "Modeled Score Group Proportion"
rowSums(x)/sum(rowSums(x))

}
\references{
Kolen, M. J., & Brennan, R. L. (2014). Test equating, scaling, and linking:
Methods and practices. Springer Science & Business Media.

Lord, F. M., & Wingersky, M. S. (1984). Comparison of IRT true-score and
equipercentile observed-score" equatings". Applied Psychological Measurement,
8(4), 453-461.

Thissen, D., Pommerich, M., Billeaud, K., & Williams, V. S. (1995). Item
response theory for scores on tests including polytomous items with ordered
responses. Applied Psychological Measurement, 19(1), 39-49.
}
\author{
Emre Gonulates
}
