% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/micro_read.r
\name{read_ipums_micro}
\alias{read_ipums_micro}
\alias{read_ipums_micro_list}
\title{Read data from an IPUMS extract}
\usage{
read_ipums_micro(
  ddi,
  vars = NULL,
  n_max = Inf,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)

read_ipums_micro_list(
  ddi,
  vars = NULL,
  n_max = Inf,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)
}
\arguments{
\item{ddi}{Either a filepath to a DDI xml file downloaded from
the website, or a \code{ipums_ddi} object parsed by \code{\link{read_ipums_ddi}}}

\item{vars}{Names of variables to load. Accepts a character vector of names, or
\code{\link{dplyr_select_style}} conventions. For hierarchical data, the
rectype id variable will be added even if it is not specified.}

\item{n_max}{The maximum number of records to load.}

\item{data_file}{Specify a directory to look for the data file.
If left empty, it will look in the same directory as the DDI file.}

\item{verbose}{Logical, indicating whether to print progress information
to console.}

\item{var_attrs}{Variable attributes to add from the DDI, defaults to
adding all (val_labels, var_label and var_desc). See
\code{\link{set_ipums_var_attributes}} for more details.}

\item{lower_vars}{Only if reading a DDI from a file, a logical indicating
whether to convert variable names to lowercase (default is FALSE, in line
with IPUMS conventions). Note that this argument will be ignored if
argument \code{ddi} is an \code{ipums_ddi} object rather than a file path.
See \code{\link{read_ipums_ddi}} for converting variable names to lowercase
when reading in the DDI.}
}
\value{
\code{read_ipums_micro} returns a single tbl_df data frame, and
  \code{read_ipums_micro_list} returns a list of data frames, named by
  the Record Type. See 'Details' for more
  information.
}
\description{
Reads a dataset downloaded from the IPUMS extract system.
For IPUMS projects with microdata, it relies on a downloaded
DDI codebook and a fixed-width file. Loads the data with
value labels (using \code{\link[haven]{labelled}} format)
and variable labels. See 'Details' for more information on
how record types are handled by the ipumsr package.
}
\details{
Some IPUMS projects have data for multiple types of records
(eg Household and Person). When downloading data from many of these
projects you have the option for the IPUMS extract system
to "rectangularize" the data, meaning that the data is
transformed so that each row of data represents only one
type of record.

There also is the option to download "hierarchical" extracts,
which are a single file with record types mixed in the rows.
The ipumsr package offers two methods for importing this data.

\code{read_ipums_micro} loads this data into a "long" format
where the record types are mixed in the rows, but the variables
are \code{NA} for the record types that they do not apply to.

\code{read_ipums_micro_list} loads the data into a list of
data frames objects, where each data frame contains only
one record type. The names of the data frames in the list
are the text from the record type labels without 'Record'
(often 'HOUSEHOLD' for Household and 'PERSON' for Person).
}
\examples{
  # Rectangular example file
  cps_rect_ddi_file <- ipums_example("cps_00006.xml")

  cps <- read_ipums_micro(cps_rect_ddi_file)
  # Or load DDI separately to keep the metadata
  ddi <- read_ipums_ddi(cps_rect_ddi_file)
  cps <- read_ipums_micro(ddi)

  # Hierarchical example file
  cps_hier_ddi_file <- ipums_example("cps_00010.xml")

  # Read in "long" format and you get 1 data frame
  cps_long <- read_ipums_micro(cps_hier_ddi_file)
  head(cps_long)

  # Read in "list" format and you get a list of multiple data frames
  cps_list <- read_ipums_micro_list(cps_hier_ddi_file)
  head(cps_list$PERSON)
  head(cps_list$HOUSEHOLD)

  # Or you can use the \code{\%<-\%} operator from zeallot to unpack
  c(household, person) \%<-\% read_ipums_micro_list(cps_hier_ddi_file)
  head(person)
  head(household)

}
\seealso{
Other ipums_read: 
\code{\link{read_ipums_micro_chunked}()},
\code{\link{read_ipums_micro_yield}()},
\code{\link{read_ipums_sf}()},
\code{\link{read_nhgis}()},
\code{\link{read_terra_area}()},
\code{\link{read_terra_micro}()},
\code{\link{read_terra_raster}()}
}
\concept{ipums_read}
