#' Plot Conditional Coefficients of a Variable in an Interaction Term
#' 
#' \code{interplot} is a generic function to produce a plot of the coefficient estimates of one variable in a two-way interaction conditional on the values of the other variable in the interaction term. The function invokes particular \code{methods} which depend on the \code{\link{class}} of the first argument. 
#' 
#' @param m A model object including an interaction term, or, alternately, a data frame generated by an earlier call to interplot using the argument plot = FALSE.
#' @param var1 The name (as a string) of the variable of interest in the interaction term; its conditional coefficient estimates will be plotted.
#' @param var2 The name (as a string) of the other variable in the interaction term
#' @param plot A logical value indicating whether the output is a plot or a dataframe including the conditional coefficient estimates of var1, their upper and lower bounds, and the corresponding values of var2.
#' @param point A logical value determining the format of plot. By default, the function produces a line plot when var2 takes on ten or more distinct values and a point (dot-and-whisker) plot otherwise; option TRUE forces a point plot.
#' @param sims Number of independent simulation draws used to calculate upper and lower bounds of coefficient estimates: lower values run faster; higher values produce smoother curves.
#' @param xmin A numerical value indicating the minimum value shown of x shown in the graph. Rarely used.
#' @param xmax A numerical value indicating the maximum value shown of x shown in the graph. Rarely used.
#' 
#' @details \code{interplot} visualizes the changes in the coefficient of one term in a two-way interaction conditioned by the other term. In the current version, the function works with interactions in the following classes of models:
#' \itemize{
#'   \item Ordinary linear models (object class: \code{lm});
#'   \item Generalized linear models (object class: \code{glm});
#'   \item Linear mixed-effects models (object class: \code{lmerMod});
#'   \item Generalized linear mixed-effects models (object class: \code{glmerMod});
#'   \item Ordinary linear models with imputed data (object class: \code{list});
#'   \item Generalized linear models with imputed data (object class: \code{list})
#'   \item Linear mixed-effects models with imputed data (object class: \code{list});
#'   \item Generalized linear mixed-effects models with imputed data (object class: \code{list}).
#' }
#' 
#' The examples below illustrate how methods invoked by this generic deal with different type of objects.
#' 
#' Because the output function is based on \code{\link[ggplot2]{ggplot}}, any additional arguments and layers supported by \code{ggplot2} can be added with the \code{+}. 
#' 
#' @return The function returns a \code{ggplot} object.
#'
#' @import  abind 
#' @import  arm
#' @import  ggplot2
#' 
#' @examples
#' data(mtcars)
#' m_cyl <- lm(mpg ~ wt * cyl, data = mtcars)
#' library(interplot)
#' 
#' # Plot interactions with a continous conditioning variable
#' interplot(m = m_cyl, var1 = "cyl", var2 = "wt") +
#' xlab("Automobile Weight (thousands lbs)") +
#' ylab("Estimated Coefficient for Number of Cylinders") +
#' ggtitle("Estimated Coefficient of Engine Cylinders\non Mileage by Automobile Weight") +
#' theme(plot.title = element_text(face="bold"))
#' 
#' 
#' # Plot interactions with a categorical conditioning variable
#' interplot(m = m_cyl, var1 = "wt", var2 = "cyl") +
#' xlab("Number of Cylinders") +
#' ylab("Estimated Coefficient for Automobile Weight (thousands lbs)") +
#' ggtitle("Estimated Coefficient of Automobile Weight \non Mileage by Engine Cylinders") +
#' theme(plot.title = element_text(face="bold"))
#' 
#' @export




interplot <- function(m, var1, var2, plot = TRUE, point = FALSE, sims = 5000, xmin = NA, xmax = NA) {
    
    
    if (class(m) == "list") {
        if (class(m[[1]]) == "lmerMod") {
            class(m) <- "mlmmi"
        }
        if (class(m[[1]]) == "glmerMod") {
            class(m) <- "gmlmmi"
        }
        if (class(m[[1]]) == "lm") {
            class(m) <- "lmmi"
        }
        if (class(m[[1]]) == "glm") {
            class(m) <- "glmmi"
        }
    }
  
    if (class(m) == "data.frame") 
        class(m) <- "plot"
    
    
    UseMethod("interplot", m)
} 
