#' Interval level intensity analysis
#' @details Gets the list of crosstabulation tables, time points vector and returns a list of interval level metrics accompanied with relevant bar graphs.
#' @param crosstabulation List of crosstabulation tables generated by \code{multicrosstab} function.
#' @param time.points a charachter vector showing the time point of each raster layer in chronological order.
#' @return The output is a list of lists. Elements of the list include: change amount, uniform intensity of change and Uniform change all in all intervals.
#' @import ggplot2
#' @importFrom graphics plot
#' @export
#' @examples
#' raster_2005 <- raster::raster(system.file("external/RASTER_2005.RST", package="intensity.analysis"))
#' raster_2010 <- raster::raster(system.file("external/RASTER_2010.RST", package="intensity.analysis"))
#' raster_2012 <- raster::raster(system.file("external/RASTER_2012.RST", package="intensity.analysis"))
#' raster.layers <- list(raster_2005, raster_2010, raster_2012)
#' time.points <- c("2005","2010","2012")
#' categories <- c("Water","Trees","Impervious")
#' crosstabulation <- multicrosstab(raster.layers, time.points, categories)
#' IIA.output <- IIA(crosstabulation, time.points)



IIA <- function(crosstabulation, time.points){

  parameters <- reqpar(time.points)

  # Amount of change in each interval.
  change.amount <- vector(mode = "numeric", length = as.integer(parameters$number.of.intervals))

  # vector saving the annual intensity of change for each time interval.
  uniform.intensity.of.change <- vector(mode = "numeric", length = as.integer(parameters$number.of.intervals))

  # Comparing the change intensity of each interval and uniform change intensity.
  # If change intensity for an interval is equal to the uniform change intensity of all intervals it labels as "uniform".
  # If change intensity for an interval is bigger in value than the uniform change intensity it labels as "Fast".
  # If change intensity for an interval is less than the uniform change intensity it labels as "Slow".
  interval.behavior <- list()

  # vector saving the intensity of change for each time interval without compensating the length of the interval.
  intensity.of.change <- matrix(nrow = as.integer(parameters$number.of.intervals), ncol = 1)

  # sum of all elements in the crosstab matrix of each interval
  study.area <- vector(mode = "integer", length = length(time.points) - 1)

  # Uniform change through all time intervals
  Uniform.change.all.intervals <- vector(mode = "numeric", length = as.integer(parameters$number.of.intervals))
  # Total duration of all time intervals
  total.duration <- sum(parameters$duration)

  for (i in 1: as.integer(parameters$number.of.intervals)){
    study.area[[i]] <- sum(crosstabulation[[i]])
    change.amount[[i]] <- study.area[[i]] - sum(diag(crosstabulation[[i]]))
    intensity.of.change[[i]] <- (study.area[[i]] - sum(diag(crosstabulation[[i]]))) / study.area[[i]]
    uniform.intensity.of.change[[i]] <- intensity.of.change[[i]] / parameters$duration[[i]]
  }

  Uniform.change.all.intervals <- sum(change.amount) / (study.area[1] * total.duration)

  Epsilon <- 0.0000001
  for (i in 1: as.integer(parameters$number.of.intervals)){
    interval.behavior[[i]] <- ifelse(abs(uniform.intensity.of.change[[i]] - Uniform.change.all.intervals) < Epsilon, "Uniform", ifelse (uniform.intensity.of.change[[i]] < Uniform.change.all.intervals, "Slow", "Fast"))
  }

  # plot of Annual Change in pixels
  Interval.change.df <- as.data.frame(cbind(parameters$interval.name, intensity.of.change))
  colnames(Interval.change.df) <- c("Interval", "Change size")
  plot(ggplot(Interval.change.df, aes(x= parameters$interval.name ,y= intensity.of.change * 100, fill = Interval.change.df$Interval)) +
         geom_bar(width = 0.5, position = position_dodge(width=0.6), stat="identity") + theme_bw() + coord_flip() +
         labs(title= "Time Interval Change Size", x= "Time Interval", y= "Change Size (% of domain)") +
         scale_fill_manual(values = rep("#010101", as.integer(parameters$number.of.intervals))) +
         theme(plot.title = element_text(family = "Times", color="#353535", face="bold", size=14, hjust=0.5)) +
         theme(legend.position="none", legend.title = element_blank()))

  Interval.change.intensity.df <- as.data.frame(cbind(parameters$interval.name, uniform.intensity.of.change, Uniform.change.all.intervals, interval.behavior))
  colnames(Interval.change.intensity.df) <- c("Interval", "Change.intensity", "Uniform.annual.change", "Interval.behavior")
  Interval.change.intensity.df$Interval.behavior <- as.vector(unlist(Interval.change.intensity.df$Interval.behavior))

  text.df <- as.data.frame(Uniform.change.all.intervals, stringsAsFactors = FALSE)

  plot(ggplot(Interval.change.intensity.df, aes(x= parameters$interval.name ,y = as.vector(uniform.intensity.of.change * 100) , fill = Interval.change.df$Interval)) +
         geom_bar(width = 0.5, position = position_dodge(width=0.6), stat="identity") + theme_bw() + coord_flip() +
         labs(title= "Time Interval Change Intensity", x= "Time Interval", y= "Annual Change (% of domain)") +
         scale_fill_manual(values = rep("#8d8d8d", as.integer(parameters$number.of.intervals))) +
         geom_hline(aes(yintercept = Uniform.change.all.intervals[[1]] * 100), linetype= "dashed", colour = "blue", size = 1, show.legend = FALSE) + coord_flip() +
         geom_text(x = length(parameters$interval.name), aes(y = Uniform.change.all.intervals * 100, label = "Fast"), stat = "identity", family = "Times", colour = "black", angle = 90, vjust = 1.2, show.legend = NA) +
         geom_text(aes(x = length(parameters$interval.name), y = Uniform.change.all.intervals * 100, label = "Slow"), stat = "identity", family = "Times", colour = "black", angle = 90, vjust = -1, show.legend = NA) +
         theme(plot.title = element_text(family = "Times", color="#353535", face="bold", size=14, hjust=0.5)) +
         theme(legend.position="none", legend.title = element_blank()))

  return(list(Change.Size = change.amount, Uniform.Intensity.of.Change = uniform.intensity.of.change, Uniform.Change.all.Intervals = Uniform.change.all.intervals, Interval.Behavior = interval.behavior))

}
