\name{optimizeNetwork}
\alias{optimizeNetwork}
\title{Optimization of networks}
\description{
This function allows to choose several options for optimizing a network of 
observation locations
}
\usage{
optimizeNetwork(observations, predGrid, candidates, method, action,
nDiff, model, criterion = "MUKV", plotOptim = TRUE, nGridCells, nTry, nr_iterations,
formulaString, ...)}

\arguments{
\item{observations}{object of class \code{\link[sp:Spatial-class]{Spatial}}*
               with coordinates and possible covariates }
\item{predGrid}{object of class \code{\link[sp:Spatial-class]{Spatial}}*
               with coordinates of prediction locations for optimization
               when \code{method = "ssa"} }
\item{candidates}{when \code{method = "manual"} or \code{method = "ssa"}, candidates is the study area of class 
\code{\link[sp:SpatialPolygons]{SpatialPolygonsDataFrame}}; for other methods, 
when \code{action = add}, candidates are points or polygons of class 
              \code{\link[sp:Spatial-class]{Spatial}}*.}
\item{method}{ character string: \code{"spcov"} for spatial coverage, 
           \code{"ssa"} for spatial simulated annealing or 
           \code{"manual"} for manual processing of the network}
\item{action}{ character string indicating which action to do with network:
                 \code{"add"} to add new measurement stations to the existing 
                 network or \code{"del"} to turn off existing stations }
\item{nDiff}{ number of stations to add or delete }
\item{model}{ variogram model to consider when \code{criterion = "mukv"}; 
         object of class \code{variogramModel},
         as generated by \code{\link[gstat:vgm]{vgm}}}
\item{criterion}{ Only in use for method \code{"ssa"}:  character string like \code{"mukv"} }
\item{plotOptim}{ plot or not the result; TRUE by default }
\item{nGridCells}{ when method is \code{"spcov"} and action is \code{"add"}:  string forcing the number gridcells to explore for candidate locations of new measurements}
\item{nTry}{ the method will try \code{nTry} initial configurations and will keep the best solution in order to reduce the risk of ending up with an unfavorable solution  }
\item{nr_iterations}{ number of iterations before stoping }
\item{formulaString}{formula that defines the dependent variable as a linear model 
      of independent variables; suppose the dependent variable has name \code{z}, 
      for ordinary and simple kriging use the formula \code{z~1}; 
      for universal kriging, suppose \code{z} is linearly dependent on 
      \code{x} and \code{y}, use the formula \code{z~x+y}. Functions of 
      dependent or independent variables are not possible.}
\item{...}{ other arguments to be passed on at lower level functions.  }
}

\details{
This function contains different methods for optimization of measurement networks,
either adding or removing observation locations.
The methods can be validated using the function \code{\link{calculateMukv}} which
return the mean universal kriging variance for a measurement network. This function is
also used for the optimization criterion for \code{method = "ssa"} with 
\code{criterion = "mukv"}.
                                                        
                                                         
The user should be aware of the following limitations:
\enumerate{
\item{\code{method = "ssa"} is only implemented for \code{criterion = "mukv"}.}
\item{Input \code{candidates} should preferably be a continuous domain such 
             as \code{\link[sp:SpatialPolygons]{SpatialPolygons}}}
\item{\code{method = "ssa"} makes it possible to assume a linear relationship between 
      independent and dependent variables (observations or spatial locations), as in 
      \code{\link[gstat:krige]{krige}}.
      However, a correct estimate of the mean universal kriging variance in the 
      prediction grid in this package requires that the independent variables are known 
      at the candidate locations. This is not possible (except for the spatial
      locations themselves), these are instead estimated
      from the prediction locations.
      This is an approximation that might influence the criterion used for optimization,
      with increasing number of points added. This method can also slow down optimization.}
\item{Note that it is not recommended to use independent variables which differ strongly 
      in magnitude (as for traditional universal kriging).}
\item{If no \code{formulaString} is supplied, it is assumed that the user will
      optimize according to the mean ordinary kriging variance.}
}
}

\value{
Object of class \code{\link[sp:SpatialPoints]{SpatialPoints}}* with coordinates 
of optimized locations (including the former observation locations when \code{action = "add"}).
}

\references{

[1] \url{http://www.intamap.org/}

[2] D. J. Brus, G. B. M. Heuvelink (2007). Optimization of sample patterns for universal 
kriging of environmental variables, Geoderma, 138: 86-95 (2007).

[3] D. J. Brus, J. de Gruijter, J. van Groenigen (2006). Designing spatial coverage samples using
the k-means clustering algorithm. In A. McBratney M. Voltz and P. Lagacherie,
editor, Digital Soil Mapping: An Introductory Perspective, Developments in Soil
Science, vol. 3., Elsevier, Amsterdam.

}

\author{ Olivier Baume }

\seealso{\code{\link{ssaOptim}}, \code{\link{spCovDel}}, \code{\link{spCovAdd}}, \code{\link{calculateMukv}}  }
\examples{
\dontrun{
library(gstat)
library(intamapInteractive)
options(error = recover)
# load data:
data(meuse)
coordinates(meuse) = ~x+y
data(meuse.grid)
coordinates(meuse.grid) = ~x+y
gridded(meuse.grid) = TRUE
predGrid = meuse.grid

# estimate variograms (OK/UK):
vfitOK = fit.variogram(variogram(zinc~1, meuse), vgm(1, "Exp", 300, 1))
vfitUK = fit.variogram(variogram(zinc~x+y, meuse), vgm(1, "Exp", 300, 1))
vfitRK = fit.variogram(variogram(zinc~dist+ffreq+soil, meuse), vgm(1, "Exp", 300, 1))

# study area of interest:
bb = bbox(predGrid)
boun = SpatialPoints(data.frame(x=c(bb[1,1],bb[1,2],bb[1,2],bb[1,1],bb[1,1]),
                                y=c(bb[2,1],bb[2,1],bb[2,2],bb[2,2],bb[2,1])))
Srl = Polygons(list(Polygon(boun)),ID = as.character(1))
candidates = SpatialPolygonsDataFrame(SpatialPolygons(list(Srl)),
                                      data = data.frame(ID=1))

# add 20 more points assuming OK model (SSA method):
 optimOK <- optimizeNetwork(meuse, meuse.grid, candidates = candidates,
  method= "ssa", action= "add", nDiff = 20, model = vfitOK, criterion="MUKV",
  nr_iterations=10000, nmax=40)


# add 20 more points assuming UK model (SSA method):
optimUK <- optimizeNetwork(meuse, meuse.grid, candidates = candidates,
   method = "ssa", action = "add", nDiff = 20, model=vfitUK, criterion="MUKV",
   nr_iterations = 10000, nmax = 40, formulaString = zinc~x+y)

# add 20 more points with auxiliary variables (SSA method):
optimRK <- optimizeNetwork(meuse, meuse.grid, candidates=candidates,
   method="ssa", action="add", nDiff=4, model=vfitRK, criterion="MUKV",
   nr_iterations=10000, formula=zinc~dist+ffreq+soil, nmax=200)

# add optimally 20 stations from current network with method "spcov"
# (spatial coverage method)
optimSC = optimizeNetwork(meuse, meuse.grid, candidates, method = "spcov",
            action = "add", nDiff = 10, model = model, criterion = "MUKV", plotOptim = TRUE,
            nGridCells = 10000,nTry = 100 )

# delete optimally 10 stations from current network with method "manual"
optimMAN = optimizeNetwork(meuse, meuse.grid, candidates, method = "manual",
            action = "del", nDiff = 10, model = model, criterion = "MUKV", plotOptim = TRUE )


# comparison of results with ordinary kriging variogram, otherwise add formulaString
calculateMukv(optimOK, predGrid, vfitOK) # ssa method, assuming ordinary kriging
calculateMukv(optimUK, predGrid, vfitOK) # ssa method, using spatial location as covariates
calculateMukv(optimRK, predGrid, vfitOK) # ssa method, using other variables as covariates
calculateMukv(optimSC, predGrid, vfitOK) # spcov method
calculateMukv(optimMAN, predGrid, vfitOK) # 10 stations manually deleted

}
}

\keyword{spatial}


