% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_predicted_ci.R
\name{get_predicted_ci}
\alias{get_predicted_ci}
\alias{get_predicted_ci.default}
\title{Confidence intervals around predicted values}
\usage{
get_predicted_ci(x, ...)

\method{get_predicted_ci}{default}(
  x,
  predictions = NULL,
  data = NULL,
  se = NULL,
  ci = 0.95,
  ci_type = "confidence",
  ci_method = NULL,
  dispersion_method = "sd",
  vcov = NULL,
  vcov_args = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{A statistical model (can also be a data.frame, in which case the
second argument has to be a model).}

\item{...}{Other argument to be passed, for instance to \code{get_predicted_ci()}.}

\item{predictions}{A vector of predicted values (as obtained by
\code{stats::fitted()}, \code{stats::predict()} or
\code{\link[=get_predicted]{get_predicted()}}).}

\item{data}{An optional data frame in which to look for variables with which
to predict. If omitted, the data used to fit the model is used. Visualization matrices can be generated using \code{\link[=get_datagrid]{get_datagrid()}}.}

\item{se}{Numeric vector of standard error of predicted values. If \code{NULL},
standard errors are calculated based on the variance-covariance matrix.}

\item{ci}{The interval level. Default is \code{NULL}, to be fast even for larger
models. Set the interval level to an explicit value, e.g. \code{0.95}, for \verb{95\%}
CI).}

\item{ci_type}{Can be \code{"prediction"} or \code{"confidence"}. Prediction
intervals show the range that likely contains the value of a new
observation (in what range it would fall), whereas confidence intervals
reflect the uncertainty around the estimated parameters (and gives the
range of the link; for instance of the regression line in a linear
regressions). Prediction intervals account for both the uncertainty in the
model's parameters, plus the random variation of the individual values.
Thus, prediction intervals are always wider than confidence intervals.
Moreover, prediction intervals will not necessarily become narrower as the
sample size increases (as they do not reflect only the quality of the fit).
This applies mostly for "simple" linear models (like \code{lm}), as for
other models (e.g., \code{glm}), prediction intervals are somewhat useless
(for instance, for a binomial model for which the dependent variable is a
vector of 1s and 0s, the prediction interval is... \verb{[0, 1]}).}

\item{ci_method}{The method for computing p values and confidence intervals. Possible values depend on model type.
\itemize{
\item \code{NULL} uses the default method, which varies based on the model type.
\item Most frequentist models: \code{"normal"} (default).
\item Bayesian models:  \code{"quantile"}  (default), \code{"hdi"}, \code{"eti"}, and \code{"spi"}.
\item Mixed effects \strong{lme4} models: \code{"normal"} (default), \code{"satterthwaite"}, and \code{"kenward"}.
}}

\item{dispersion_method}{Bootstrap dispersion and Bayesian posterior summary: \code{"sd"} or \code{"mad"}.}

\item{vcov}{Variance-covariance matrix used to compute uncertainty estimates (e.g., for robust standard errors). This argument accepts a covariance matrix, a function which returns a covariance matrix, or a string which identifies the function to be used to compute the covariance matrix.
\itemize{
\item A covariance matrix
\item A function which returns a covariance matrix (e.g., \code{stats::vcov()})
\item A string which indicates the kind of uncertainty estimates to return.
\itemize{
\item Heteroskedasticity-consistent: \code{"vcovHC"}, \code{"HC"}, \code{"HC0"}, \code{"HC1"}, \code{"HC2"}, \code{"HC3"}, \code{"HC4"}, \code{"HC4m"}, \code{"HC5"}. See \code{?sandwich::vcovHC}
\item Cluster-robust: \code{"vcovCR"}, \code{"CR0"}, \code{"CR1"}, \code{"CR1p"}, \code{"CR1S"}, \code{"CR2"}, \code{"CR3"}. See \code{?clubSandwich::vcovCR()}
\item Bootstrap: \code{"vcovBS"}, \code{"xy"}, \code{"residual"}, \code{"wild"}, \code{"mammen"}, \code{"webb"}. See \code{?sandwich::vcovBS}
\item Other \code{sandwich} package functions: \code{"vcovHAC"}, \code{"vcovPC"}, \code{"vcovCL"}, \code{"vcovPL"}.
}
}}

\item{vcov_args}{List of arguments to be passed to the function identified by
the \code{vcov} argument. This function is typically supplied by the \strong{sandwich}
or \strong{clubSandwich} packages. Please refer to their documentation (e.g.,
\code{?sandwich::vcovHAC}) to see the list of available arguments. If no estimation
type (argument \code{type}) is given, the default type for \code{"HC"} (or \code{"vcovHC"})
equals the default from the \strong{sandwich} package; for type \code{"CR"} (or
\code{"vcoCR"}), the default is set to \code{"CR3"}.}

\item{verbose}{Toggle warnings.}
}
\description{
Confidence intervals around predicted values
}
\examples{
# Confidence Intervals for Model Predictions
# ------------------------------------------

data(mtcars)

# Linear model
# ------------
x <- lm(mpg ~ cyl + hp, data = mtcars)
predictions <- predict(x)
ci_vals <- get_predicted_ci(x, predictions, ci_type = "prediction")
head(ci_vals)
ci_vals <- get_predicted_ci(x, predictions, ci_type = "confidence")
head(ci_vals)
ci_vals <- get_predicted_ci(x, predictions, ci = c(0.8, 0.9, 0.95))
head(ci_vals)

# Bootstrapped
# ------------
if (require("boot")) {
  predictions <- get_predicted(x, iterations = 500)
  get_predicted_ci(x, predictions)
}

if (require("datawizard") && require("bayestestR")) {
  ci_vals <- get_predicted_ci(x, predictions, ci = c(0.80, 0.95))
  head(ci_vals)
  datawizard::reshape_ci(ci_vals)

  ci_vals <- get_predicted_ci(x,
    predictions,
    dispersion_method = "MAD",
    ci_method = "HDI"
  )
  head(ci_vals)
}


# Logistic model
# --------------
x <- glm(vs ~ wt, data = mtcars, family = "binomial")
predictions <- predict(x, type = "link")
ci_vals <- get_predicted_ci(x, predictions, ci_type = "prediction")
head(ci_vals)
ci_vals <- get_predicted_ci(x, predictions, ci_type = "confidence")
head(ci_vals)
}
