## ----eval=FALSE---------------------------------------------------------------
# # install.packages("remotes")
# remotes::install_github("EuropeanIFCBGroup/iRfcb")

## ----eval=FALSE---------------------------------------------------------------
# library(iRfcb)

## ----include=FALSE------------------------------------------------------------
library(iRfcb)

## ----eval=FALSE---------------------------------------------------------------
# # Define data directory
# data_dir <- "data"
# 
# # Download and extract test data in the data folder
# ifcb_download_test_data(dest_dir = data_dir,
#                         max_retries = 10,
#                         sleep_time = 30,
#                         verbose = FALSE)

## ----include=FALSE------------------------------------------------------------
# Define data directory
data_dir <- "data"

# Download and extract test data in the data folder
if (!dir.exists(data_dir)) {
  # Download and extract test data if the folder does not exist
  ifcb_download_test_data(dest_dir = data_dir,
                          max_retries = 10,
                          sleep_time = 30,
                          verbose = FALSE)
}

## -----------------------------------------------------------------------------
# Extract all classified images from a sample
ifcb_extract_classified_images(
  sample = "D20230314T001205_IFCB134",
  classified_folder = "data/classified",
  roi_folder = "data/data",
  out_folder = "data/classified_images",
  taxa = "Tripos_lineatus", # A specific taxa or "All"
  threshold = "opt") # or specify another threshold

## -----------------------------------------------------------------------------
# Read a MATLAB summary file generated by `countcells_allTBnew_user_training`
summary_data <- ifcb_read_summary("data/classified/2023/summary/summary_allTB_2023.mat",
                                  biovolume = FALSE,
                                  threshold = "opt")

# Print output
head(summary_data)

## -----------------------------------------------------------------------------
# Summarize biovolume data using IFCB data from classified data folder
biovolume_data <- ifcb_summarize_biovolumes(
  feature_folder = "data/features/2023",
  mat_folder = "data/classified",
  hdr_folder = "data/data/2023",
  micron_factor = 1/3.4,
  diatom_class = "Bacillariophyceae",
  threshold = "opt",
  verbose = FALSE) # Do not print progress bars

# Print output
head(biovolume_data)

## -----------------------------------------------------------------------------
# Summarize biovolume data using IFCB data from manual data folder
manual_biovolume_data <- ifcb_summarize_biovolumes(
  feature_folder = "data/features",
  mat_folder = "data/manual",
  class2use_file = "data/config/class2use.mat",
  hdr_folder = "data/data",
  micron_factor = 1/3.4,
  diatom_class = "Bacillariophyceae",
  verbose = FALSE) # Do not print progress bars

# Print output
head(manual_biovolume_data)

## -----------------------------------------------------------------------------
# Summarise counts on sample level
png_per_sample <- ifcb_summarize_png_counts(png_folder = "data/png",
                                            hdr_folder = "data/data",
                                            sum_level = "sample")

# Print output
head(png_per_sample)

# Summarise counts on class level
png_per_class <- ifcb_summarize_png_counts(png_folder = "data/png",
                                           sum_level = "class")

# Print output
head(png_per_class)

## -----------------------------------------------------------------------------
# Summarize counts from MATLAB files
mat_count <- ifcb_count_mat_annotations(
  manual_files = "data/manual",
  class2use_file = "data/config/class2use.mat",
  skip_class = "unclassified", # Or class ID
  sum_level = "class") # Or per "sample"

# Print output
head(mat_count)

## ----eval=FALSE---------------------------------------------------------------
# # Run Shiny app
# ifcb_run_image_gallery()

## ----include=FALSE------------------------------------------------------------
library(reticulate)

# Define path to virtual environment
env_path <- file.path(tempdir(), "iRfcb") # Or your preferred venv path

# Install python virtual environment
tryCatch({
  ifcb_py_install(envname = env_path)
}, error = function(e) {
  warning("Python environment could not be installed.")
})

## ----echo=FALSE---------------------------------------------------------------
# Check if Python is available
if (!py_available(initialize = TRUE)) {
  knitr::opts_chunk$set(eval = FALSE)
  warning("Python is not available. Skipping the rest of the vignette evaluation.")
} else {
  # List available packages
  available_packages <- py_list_packages(python = reticulate::py_discover_config()$python)
  
  # Check if scipy is available
  if (!"scipy" %in% available_packages$package) {
    knitr::opts_chunk$set(eval = FALSE)
    warning("Required python modules are not available. Skipping the rest of the vignette evaluation.")
  }
}

## ----eval=FALSE---------------------------------------------------------------
# # Get class2use
# class_name <- ifcb_get_mat_names("data/config/class2use.mat")
# class2use <- ifcb_get_mat_variable("data/config/class2use.mat",
#                                    variable_name = class_name)
# 
# # Find the class id of unclassified
# unclassified_id <- which(grepl("unclassified",
#                          class2use))
# 
# # Initialize the python session if not already set up
# env_path <- file.path(tempdir(), "iRfcb") # Or your preferred venv path
# ifcb_py_install(envname = env_path)
# 
# # Correct the annotation with the output from the image gallery
# ifcb_correct_annotation(
#   manual_folder = "data/manual",
#   out_folder = "data/manual",
#   correction = "data/manual/correction/Alexandrium_pseudogonyaulax_selected_images.txt",
#   correct_classid = unclassified_id)

## ----include=FALSE------------------------------------------------------------
# Get class2use
class_name <- ifcb_get_mat_names("data/config/class2use.mat")
class2use <- ifcb_get_mat_variable("data/config/class2use.mat",
                                   variable_name = class_name)

# Find the class id of unclassified
unclassified_id <- which(grepl("unclassified",
                         class2use))

# Correct the annotation with the output from the image gallery
ifcb_correct_annotation(
  manual_folder = "data/manual",
  out_folder = "data/manual",
  correction = "data/manual/correction/Alexandrium_pseudogonyaulax_selected_images.txt",
  correct_classid = unclassified_id)

## -----------------------------------------------------------------------------
# Get class2use
class_name <- ifcb_get_mat_names("data/config/class2use.mat")
class2use <- ifcb_get_mat_variable("data/config/class2use.mat",
                                   variable_name = class_name)

# Find the class id of Alexandrium_pseudogonyaulax
ap_id <- which(grepl("Alexandrium_pseudogonyaulax",
                     class2use))

# Find the class id of unclassified
unclassified_id <- which(grepl("unclassified",
                         class2use))

# Move all Alexandrium_pseudogonyaulax images to unclassified
ifcb_replace_mat_values(manual_folder = "data/manual",
                        out_folder = "data/manual",
                        target_id = ap_id,
                        new_id = unclassified_id)

## -----------------------------------------------------------------------------
# Summarize new counts after correction
mat_count <- ifcb_count_mat_annotations(
  manual_files = "data/manual",
  class2use_file = "data/config/class2use.mat",
  skip_class = "unclassified", # Or class ID
  sum_level = "class") # Or per "sample"

# Print output
head(mat_count)

## -----------------------------------------------------------------------------
# Read a file with selected images, generated by the image gallery app
correction <- read.table(
  "data/manual/correction/Alexandrium_pseudogonyaulax_selected_images.txt", 
  header = TRUE)

# Print image names to be annotated
print(correction$image_filename)

# Re-annotate the images that were moved to unclassified earlier in the tutorial
ifcb_annotate_batch(png_images = correction$image_filename,
                    class = "Alexandrium_pseudogonyaulax",
                    manual_folder = "data/manual",
                    adc_folder = "data/data",
                    class2use_file = "data/config/class2use.mat")

# Summarize new counts after re-annotation
mat_count <- ifcb_count_mat_annotations(
  manual_files = "data/manual",
  class2use_file = "data/config/class2use.mat",
  skip_class = "unclassified",
  sum_level = "class")

# Print output and check if Alexandrium pseudogonyaulax is back
head(mat_count)

## ----echo=FALSE---------------------------------------------------------------
# Print citation
citation("iRfcb")

## ----include=FALSE------------------------------------------------------------
# Clean up
unlink(file.path(data_dir, "classified_images"), recursive = TRUE)
unlink(file.path(data_dir, "zip"), recursive = TRUE)

