% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hardy-cross.R
\name{hardycross}
\alias{hardycross}
\title{Applies the Hardy-Cross method to solve for pipe flows in a network.}
\usage{
hardycross(
  dfpipes = dfpipes,
  loops = loops,
  Qs = Qs,
  n_iter = 1,
  units = c("SI", "Eng"),
  ret_units = FALSE
)
}
\arguments{
\item{dfpipes}{data frame with the pipe data. Format is described above, but must contain a column named _ID_.}

\item{loops}{integer list defining pipes in each loop of the network.}

\item{Qs}{numeric list of initial flows for each pipe in each loop [\eqn{m^3 s^{-1}}{m^3/s} or \eqn{ft^3 s^{-1}}{ft^3/s}]}

\item{n_iter}{integer identifying the number of iterations to perform.}

\item{units}{character vector that contains the system of units [options are
\code{SI} for International System of Units and \code{Eng} for English (US customary)
units.}

\item{ret_units}{If set to TRUE the value(s) returned for pipe flows are of 
class \code{units} with units attached to the value. [Default is FALSE]}
}
\value{
Returns a list of two data frames:
\itemize{
  \item dfloops - the final flow magnitude and direction (clockwise positive) for each loop and pipe
  \item dfpipes - the input pipe data frame, with additional columns including final Q
}
}
\description{
This function uses the Hardy-Cross method to iteratively solve the
equations for conservation of mass and energy in a water pipe network.
The input consists of a data frame with the pipe characteristics and lists 
of the pipes in each loop (listed in a clockwise direction) and the initial
guesses of flows in each pipe (positive flows are in a clockwise direction).
}
\details{
The input data frame with the pipe data must contain a pipe ID column with 
the pipe numbers used in the loops input list. There are three options for 
input column of the pipe roughness data frame:
\tabular{ll}{
  \strong{Column Name} \tab \strong{Approach for Determining K} \cr
  ks \tab {f calculated using Colebrook equation, K using Darcy-Weisbach} \cr
  f \tab {f treated as fixed, K calculated using Darcy-Weisbach} \cr
  K \tab {K treated as fixed}
  }
  In the case where absolute pipe roughness, \eqn{ks} (in m or ft), is input, 
  the input pipe data frame must also include columns for the length, \eqn{L} and
  diameter, \eqn{D}, (both in m or ft) so \eqn{K} can be calculated. In this case, 
  a new \eqn{f} and \eqn{K} are calculated at each iteration, the final values of 
  which are included in the output. If input \eqn{K} or \eqn{f} columns are provided, values 
  for \eqn{ks} are ignored. If an input \eqn{K} column is provided, \eqn{ks} and \eqn{f} are 
  ignored. If the Colebrook equation is used to determine \eqn{f}, a water 
  temperature of \eqn{20^{o}C} or \eqn{68^{o}F} is used.
  
  The number of iterations to perform may be specified with the n_iter input
  value, but execution stops if the average flow adjustment becomes smaller 
  than 1 percent of the average flow in all pipes.

The Darcy-Weisbach equation is used to estimate the head loss in each
pipe segment, expressed in a condensed form as \eqn{h_f = KQ^{2}}
where: \deqn{K = \frac{8fL}{\pi^{2}gD^{5}}}  
If needed, the friction factor \eqn{f} is calculated using the Colebrook 
equation. The flow adjustment in each loop is calculated at each iteration as: 
\deqn{\Delta{Q_i} = -\frac{\sum_{j=1}^{p_i} K_{ij}Q_j|Q_j|}{\sum_{j=1}^{p_i} 2K_{ij}Q_j^2}}
where \eqn{i} is the loop number, \eqn{j} is the pipe number, \eqn{p_i} is the number of 
pipes in loop \eqn{i} and \eqn{\Delta{Q_i}} is the flow adjustment to be applied 
to each pipe in loop \eqn{i} for the next iteration.
}
\examples{

#              A----------B --> 0.5m^3/s
#              |\   (4)   |
#              | \        |
#              |  \       |
#              |   \(2)   |
#              |    \     |(5)
#              |(1)  \    |
#              |      \   |
#              |       \  |
#              |        \ |
#              |   (3)   \|
# 0.5m^3/s --> C----------D

#Input pipe characteristics data frame. With K given other columns not needed
dfpipes <- data.frame(
ID = c(1,2,3,4,5),                     #pipe ID
K = c(200,2500,500,800,300)            #resistance used in hf=KQ^2
)
loops <- list(c(1,2,3),c(2,4,5))
Qs <- list(c(0.3,0.1,-0.2),c(-0.1,0.2,-0.3))
hardycross(dfpipes = dfpipes, loops = loops, Qs = Qs, n_iter = 1, units = "SI")

}
\seealso{
\code{\link{colebrook}}, \code{\link{darcyweisbach}}
}
