% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ionization_functions.R
\name{calc_ionization}
\alias{calc_ionization}
\title{Calculate the ionization.}
\usage{
calc_ionization(
  chem.cas = NULL,
  chem.name = NULL,
  dtxsid = NULL,
  parameters = NULL,
  pH = NULL,
  pKa_Donor = NULL,
  pKa_Accept = NULL
)
}
\arguments{
\item{chem.cas}{Either the chemical name or the CAS number must be
specified.}

\item{chem.name}{Either the chemical name or the CAS number must be
specified.}

\item{dtxsid}{EPA's 'DSSTox Structure ID (https://comptox.epa.gov/dashboard)
the chemical must be identified by either CAS, name, or DTXSIDs}

\item{parameters}{Chemical parameters from a parameterize_MODEL function,
overrides chem.name and chem.cas.}

\item{pH}{pH where ionization is evaluated.}

\item{pKa_Donor}{Compound H dissociation equilibirum constant(s).
Overwrites chem.name and chem.cas.}

\item{pKa_Accept}{Compound H association equilibirum constant(s).
Overwrites chem.name and chem.cas.}
}
\value{
\item{fraction_neutral}{fraction of compound neutral}
\item{fraction_charged}{fraction of compound charged}
\item{fraction_negative}{fraction of compound negative}
\item{fraction_positive}{fraction of compound positive}
\item{fraction_zwitter}{fraction of compound zwitterionic}
}
\description{
This function calculates the ionization of a compound at a given pH. The 
pKa's are either entered as parameters or taken from a specific compound in
the package.
}
\details{
The arguments pKa_Donor and pKa_Accept may be single numbers, characters, or 
vectors. We support characters because there are many instances with multiple 
predicted values and all those values can be included by concatenating with 
commas (for example, pKa_Donor = "8.1,8.6". Finally, pka_Donor and pKa_Accept 
may be vectors of characters representing different chemicals or instances of
chemical parameters to allow for uncertainty analysis.

The fractions are calculated by determining the coefficients for each
species and dividing the particular species by the sum of all three.  The
positive, negative and zwitterionic/neutral coefficients are given by:
\deqn{zwitter/netural = 1} \deqn{for(i in 1:pkabove) negative = negative +
10^(i * pH - pKa1 - ... - pKai)} \deqn{for(i in 1:pkbelow) positive =
positive + 10^(pKa1 + ... + pKai - i * pH)} where i begins at 1 and ends at
the number of points above(for negative) or below(for positive) the
neutral/zwitterionic range.  The neutral/zwitterionic range is either the pH
range between 2 pKa's where the number of acceptors above is equal to the
number of donors below, everything above the pKa acceptors if there are no
donors, or everything below the pKa donors if there are no acceptors.  Each
of the terms in the sums represent a different ionization.
}
\examples{
# Donor pKa's 9.78,10.39 -- Should be almost all neutral at plasma pH:
out <- calc_ionization(chem.name='bisphenola',pH=7.4)
print(out)
out[["fraction_neutral"]]==max(unlist(out))

# Donor pKa's 9.78,10.39 -- Should be almost all negative (anion) at higher pH:
out <- calc_ionization(chem.name='bisphenola',pH=11)
print(out)
out[["fraction_negative"]]==max(unlist(out))

# Fictious compound, should be almost all all negative (anion):
out <- calc_ionization(pKa_Donor=8,pKa_Accept="1,4",pH=9)
print(out)
out[["fraction_negative"]]>0.9

# Donor pKa 6.54 -- Should be mostly negative (anion):
out <- calc_ionization(chem.name='Acephate',pH=7)
print(out)
out[["fraction_negative"]]==max(unlist(out))

#Acceptor pKa's "9.04,6.04"  -- Should be almost all positive (cation) at plasma pH:
out <- calc_ionization(chem.cas="145742-28-5",pH=7.4)
print(out)
out[["fraction_positive"]]==max(unlist(out))

#Fictious Zwitteron:
out <- calc_ionization(pKa_Donor=6,pKa_Accept="8",pH=7.4)
print(out)
out[["fraction_zwitter"]]==max(unlist(out))

}
\references{
Pearce, Robert G., et al. "Evaluation and calibration of
high-throughput predictions of chemical distribution to tissues." Journal of
Pharmacokinetics and Pharmacodynamics 44.6 (2017): 549-565.

Strope, Cory L., et al. "High-throughput in-silico prediction of ionization 
equilibria for pharmacokinetic modeling." Science of The Total Environment 
615 (2018): 150-160.
}
\author{
Robert Pearce and John Wambaugh
}
\keyword{Parameter}
