% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/category.R
\name{category}
\alias{category}
\title{Calculate the categories of events.}
\usage{
category(
  data,
  y = temp,
  S = TRUE,
  name = "Event",
  climatology = FALSE,
  season = "range",
  roundVal = 4
)
}
\arguments{
\item{data}{The function receives the full (list) output from the
\code{\link{detect_event}} function.}

\item{y}{The column containing the measurement variable. If the column
name differs from the default (i.e. \code{temp}), specify the name here.}

\item{S}{This argument informs the function if the data were collected in the
southern hemisphere (TRUE, default) or the northern hemisphere (FALSE) so that it may correctly
output the \code{season} column (see below).}

\item{name}{If a character string (e.g. "Bohai Sea") is provided here it will be used
to name the events in the \code{event_name} column (see below) of the output.
If no value is provided the default output is "Event".}

\item{climatology}{The default setting of \code{FALSE} will tell this function to output only
the summary (wide) results for the individual events as seen in Hobday et al. (2018). If set
to \code{TRUE}, this function will return a list of two dataframes, same
as \code{\link{detect_event}}. The first dataframe \code{climatology}, contains the same
information as found in \code{\link{detect_event}}, but with the addition of the daily
intensity (anomaly above seasonal doy threshold) and category values. The second dataframe,
\code{event}, is the summary results that this function produces by default.}

\item{season}{This argument allows the user to decide how the season(s) of occurrence for
the MHWs are labelled. The default setting of \code{"range"} will return the range of seasons
over which the MHW occurred, as seen in Hobday et al. (2018). One may chose to rather have
this function return only the season during the "start", "peak", or "end" of the MHW by giving
the corresponding character vector.}

\item{roundVal}{This argument allows the user to choose how many decimal places
the outputs will be rounded to. Default is 4. To
prevent rounding set \code{roundClm = FALSE}. This argument may only be given
numeric values or FALSE.}
}
\value{
The function will return a tibble with results similar to those seen in
Table 2 of Hobday et al. (2018). This provides the information necessary to
appraise the extent of the events in the output of \code{\link{detect_event}} based on the
category ranking scale. The category thresholds are calculated based on the difference
between the given seasonal climatology and threshold climatology. The four category levels
are then the difference multiplied by the category level.

The definitions for the default output columns are as follows:
  \item{event_no}{The number of the event as determined by \code{\link{detect_event}}
  to allow for joining between the outputs.}
  \item{event_name}{The name of the event. Generated from the \code{\link{name}}
  value provided and the year of the \code{peak_date} (see following) of
  the event. If no \code{\link{name}} value is provided the default "Event" is used.
  As proposed in Hobday et al. (2018), \code{Moderate} events are not given a name
  so as to prevent multiple repeat names within the same year. If two or more events
  ranked greater than Moderate are reported within the same year, they will be
  differentiated with the addition of a trailing letter
  (e.g. Event 2001a, Event 2001b).}
  \item{peak_date}{The date (day) on which the maximum intensity of the event
  was recorded.}
  \item{category}{The maximum category threshold reached/exceeded by the event.}
  \item{i_max}{The maximum intensity of the event above the threshold value.}
  \item{duration}{The total duration (days) of the event. Note that this includes
  any possible days when the measurement value \code{\link{y}}) may have dropped below the
  threshold value. Therefore, the proportion of the event duration (days) spent above
  certain thresholds may not add up to 100\% (see following four items).}
  \item{p_moderate}{The proportion of the total duration (days) spent at or above
  the first threshold, but below any further thresholds.}
  \item{p_strong}{The proportion of the total duration (days) spent at or above
  the second threshold, but below any further thresholds.}
  \item{p_severe}{The proportion of the total duration (days) spent at or above
  the third threshold, but below the fourth threshold.}
  \item{p_extreme}{The proportion of the total duration (days) spent at or above
  the fourth and final threshold.}
  \item{season}{The season(s) during which the event occurred. If the event
  occurred across two seasons this will be displayed as e.g. "Winter/Spring".
  Across three seasons as e.g. "Winter-Summer". Events lasting across four or more
  seasons are listed as "Year-round". December (June) is used here as the start of
  Austral (Boreal) summer. If "start", "peak", or "end" was given to the \code{season}
  argument then only the one season during that chosen period will be given.}

If \code{climatology = TRUE}, this function will output a list of two dataframes.
The first dataframe, \code{climatology}, will contain the following columns:
  \item{t}{The column containing the daily date values.}
  \item{event_no}{The numeric event number label.}
  \item{intensity}{The daily exceedance (default is degrees C) above the
  seasonal climatology.}
  \item{category}{The category classification per day.}

The second dataframe, \code{event}, contains the default output of this function,
as detailed above.
}
\description{
Calculates the categories of MHWs or MCSs produced by \code{\link{detect_event}} in
accordance with the naming scheme proposed in Hobday et al. (2018).
}
\details{
An explanation for the categories is as follows:
\enumerate{
  \item{I Moderate-}{Events that have been detected, but with a maximum intensity that does not
  double the distance between the seasonal climatology and the threshold value.}
  \item{II Strong-}{Events with a maximum intensity that doubles the distance from the seasonal
  climatology and the threshold, but do not triple it.}
  \item{III Severe-}{Events that triple the aforementioned distance, but do not quadruple it.}
  \item{IV Extreme-}{Events with a maximum intensity that is four times or greater than the
  aforementioned distance. Scary stuff...}
  }
}
\examples{
res_WA <- detect_event(ts2clm(sst_WA,
                       climatologyPeriod = c("1983-01-01", "2012-12-31")))
# Note that the name argument expects a character vector
cat_WA <- category(res_WA, name = "WA")
tail(cat_WA)

# If the data were collected in the northern hemisphere
# we must let the function know this, as seen below
res_Med <- detect_event(ts2clm(sst_Med,
                        climatologyPeriod = c("1983-01-01", "2012-12-31")))
cat_Med <- category(res_Med, S = FALSE, name = "Med")
tail(cat_Med)

# One may also choose to have this function output the daily
# category classifications as well by setting: climatology = TRUE
cat_WA_daily <- category(res_WA, name = "WA", climatology = TRUE)
head(cat_WA_daily$climatology)

# Note that this will not return the complete time series, only the
# days during which events were detected.
# This was done to reduce the size of the output for those working
# with gridded data.
# Should one want a complete time series, the daily category results
# may simply be left_join() with the detect_event() results
cat_WA_ts <- dplyr::left_join(res_WA$climatology,
                              cat_WA_daily$climatology)
head(cat_WA_ts)

}
\references{
Hobday et al. (2018). Categorizing and Naming
Marine Heatwaves. Oceanography 31(2).
}
\author{
Robert W. Schlegel
}
