/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5PLmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5PLpkg.h"    

herr_t
H5PLset_loading_state(unsigned int plugin_control_mask)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5PL__set_plugin_control_mask(plugin_control_mask) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "error setting plugin control mask");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLget_loading_state(unsigned *plugin_control_mask )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (NULL == plugin_control_mask)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_control_mask parameter cannot be NULL");

    
    if (H5PL__get_plugin_control_mask(plugin_control_mask) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTGET, FAIL, "error getting plugin control mask");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLappend(const char *search_path)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == search_path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot be NULL");
    if (0 == strlen(search_path))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot have length zero");

    
    if (H5PL__append_path(search_path) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTAPPEND, FAIL, "unable to append search path");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLprepend(const char *search_path)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == search_path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot be NULL");
    if (0 == strlen(search_path))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot have length zero");

    
    if (H5PL__prepend_path(search_path) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to prepend search path");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLreplace(const char *search_path, unsigned int idx)
{
    unsigned num_paths;           
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == search_path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot be NULL");
    if (0 == strlen(search_path))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot have length zero");

    
    num_paths = H5PL__get_num_paths();
    if (0 == num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "path table is empty");
    else if (idx >= num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL,
                    "index path out of bounds for table - can't be more than %u", (num_paths - 1));

    
    if (H5PL__replace_path(search_path, idx) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to replace search path");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLinsert(const char *search_path, unsigned int idx)
{
    unsigned num_paths;           
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == search_path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot be NULL");
    if (0 == strlen(search_path))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "plugin_path parameter cannot have length zero");

    
    num_paths = H5PL__get_num_paths();
    if ((0 != num_paths) && (idx >= num_paths))
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL,
                    "index path out of bounds for table - can't be more than %u", (num_paths - 1));

    
    if (H5PL__insert_path(search_path, idx) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTINSERT, FAIL, "unable to insert search path");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLremove(unsigned int idx)
{
    unsigned num_paths;           
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    num_paths = H5PL__get_num_paths();
    if (0 == num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "path table is empty");
    else if (idx >= num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL,
                    "index path out of bounds for table - can't be more than %u", (num_paths - 1));

    
    if (H5PL__remove_path(idx) < 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTDELETE, FAIL, "unable to remove search path");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5PLget(unsigned int idx, char *path_buf, size_t buf_size)
{
    unsigned    num_paths;        
    const char *path      = NULL; 
    size_t      path_len  = 0;    
    ssize_t     ret_value = 0;    

    FUNC_ENTER_API(FAIL)

    
    num_paths = H5PL__get_num_paths();
    if (0 == num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "path table is empty");
    else if (idx >= num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL,
                    "index path out of bounds for table - can't be more than %u", (num_paths - 1));

    
    if (H5PL__get_num_paths() == 0)
        HGOTO_ERROR(H5E_PLUGIN, H5E_NOSPACE, (-1), "plugin search path table is empty");

    
    if (NULL == (path = H5PL__get_path(idx)))
        HGOTO_ERROR(H5E_PLUGIN, H5E_BADVALUE, (-1), "no path stored at that index");
    path_len = strlen(path);

    
    if (path_buf) {
        strncpy(path_buf, path, buf_size);
        if ((size_t)path_len >= buf_size)
            path_buf[buf_size - 1] = '\0';
    } 

    
    ret_value = (ssize_t)path_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5PLsize(unsigned int *num_paths)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!num_paths)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "num_paths parameter cannot be NULL");

    
    *num_paths = H5PL__get_num_paths();

done:
    FUNC_LEAVE_API(ret_value)
} 
