/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5Pprivate.h"  
#include "H5Sprivate.h"  

static void  *H5O__fill_old_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                   size_t p_size, const uint8_t *p);
static herr_t H5O__fill_old_encode(H5F_t *f, uint8_t *p, const void *_mesg);
static size_t H5O__fill_old_size(const H5F_t *f, const void *_mesg);
static void  *H5O__fill_new_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                   size_t p_size, const uint8_t *p);
static herr_t H5O__fill_new_encode(H5F_t *f, uint8_t *p, const void *_mesg);
static size_t H5O__fill_new_size(const H5F_t *f, const void *_mesg);
static void  *H5O__fill_copy(const void *_mesg, void *_dest);
static herr_t H5O__fill_reset(void *_mesg);
static herr_t H5O__fill_free(void *_mesg);
static herr_t H5O__fill_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                      const H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__fill_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

#define H5O_SHARED_TYPE        H5O_MSG_FILL
#define H5O_SHARED_DECODE      H5O__fill_shared_decode
#define H5O_SHARED_DECODE_REAL H5O__fill_old_decode
#define H5O_SHARED_ENCODE      H5O__fill_shared_encode
#define H5O_SHARED_ENCODE_REAL H5O__fill_old_encode
#define H5O_SHARED_SIZE        H5O__fill_shared_size
#define H5O_SHARED_SIZE_REAL   H5O__fill_old_size
#define H5O_SHARED_DELETE      H5O__fill_shared_delete
#undef H5O_SHARED_DELETE_REAL
#define H5O_SHARED_LINK H5O__fill_shared_link
#undef H5O_SHARED_LINK_REAL
#define H5O_SHARED_COPY_FILE H5O__fill_shared_copy_file
#undef H5O_SHARED_COPY_FILE_REAL
#define H5O_SHARED_POST_COPY_FILE H5O__fill_shared_post_copy_file
#undef H5O_SHARED_POST_COPY_FILE_REAL
#undef H5O_SHARED_POST_COPY_FILE_UPD
#define H5O_SHARED_DEBUG      H5O__fill_shared_debug
#define H5O_SHARED_DEBUG_REAL H5O__fill_debug
#include "H5Oshared.h" 

#undef H5O_SHARED_TYPE
#define H5O_SHARED_TYPE H5O_MSG_FILL_NEW
#undef H5O_SHARED_DECODE
#define H5O_SHARED_DECODE H5O__fill_new_shared_decode
#undef H5O_SHARED_DECODE_REAL
#define H5O_SHARED_DECODE_REAL H5O__fill_new_decode
#undef H5O_SHARED_ENCODE
#define H5O_SHARED_ENCODE H5O__fill_new_shared_encode
#undef H5O_SHARED_ENCODE_REAL
#define H5O_SHARED_ENCODE_REAL H5O__fill_new_encode
#undef H5O_SHARED_SIZE
#define H5O_SHARED_SIZE H5O__fill_new_shared_size
#undef H5O_SHARED_SIZE_REAL
#define H5O_SHARED_SIZE_REAL H5O__fill_new_size
#undef H5O_SHARED_DELETE
#define H5O_SHARED_DELETE H5O__fill_new_shared_delete
#undef H5O_SHARED_DELETE_REAL
#undef H5O_SHARED_LINK
#define H5O_SHARED_LINK H5O__fill_new_shared_link
#undef H5O_SHARED_LINK_REAL
#undef H5O_SHARED_COPY_FILE
#define H5O_SHARED_COPY_FILE H5O__fill_new_shared_copy_file
#undef H5O_SHARED_COPY_FILE_REAL
#undef H5O_SHARED_POST_COPY_FILE
#define H5O_SHARED_POST_COPY_FILE H5O__fill_new_shared_post_copy_file
#undef H5O_SHARED_POST_COPY_FILE_REAL
#undef H5O_SHARED_POST_COPY_FILE_UPD
#undef H5O_SHARED_DEBUG
#define H5O_SHARED_DEBUG H5O__fill_new_shared_debug
#undef H5O_SHARED_DEBUG_REAL
#define H5O_SHARED_DEBUG_REAL H5O__fill_debug
#undef H5Oshared_H
#include "H5Oshared.h" 

const H5O_msg_class_t H5O_MSG_FILL[1] = {{
    H5O_FILL_ID,                               
    "fill",                                    
    sizeof(H5O_fill_t),                        
    H5O_SHARE_IS_SHARABLE | H5O_SHARE_IN_OHDR, 
    H5O__fill_shared_decode,                   
    H5O__fill_shared_encode,                   
    H5O__fill_copy,                            
    H5O__fill_shared_size,                     
    H5O__fill_reset,                           
    H5O__fill_free,                            
    H5O__fill_shared_delete,                   
    H5O__fill_shared_link,                     
    NULL,                                      
    NULL,                                      
    H5O__fill_pre_copy_file,                   
    H5O__fill_shared_copy_file,                
    H5O__fill_shared_post_copy_file,           
    NULL,                                      
    NULL,                                      
    H5O__fill_shared_debug                     
}};

const H5O_msg_class_t H5O_MSG_FILL_NEW[1] = {{
    H5O_FILL_NEW_ID,                           
    "fill_new",                                
    sizeof(H5O_fill_t),                        
    H5O_SHARE_IS_SHARABLE | H5O_SHARE_IN_OHDR, 
    H5O__fill_new_shared_decode,               
    H5O__fill_new_shared_encode,               
    H5O__fill_copy,                            
    H5O__fill_new_shared_size,                 
    H5O__fill_reset,                           
    H5O__fill_free,                            
    H5O__fill_new_shared_delete,               
    H5O__fill_new_shared_link,                 
    NULL,                                      
    NULL,                                      
    H5O__fill_pre_copy_file,                   
    H5O__fill_new_shared_copy_file,            
    H5O__fill_new_shared_post_copy_file,       
    NULL,                                      
    NULL,                                      
    H5O__fill_new_shared_debug                 
}};

const unsigned H5O_fill_ver_bounds[] = {
    H5O_FILL_VERSION_1,     
    H5O_FILL_VERSION_3,     
    H5O_FILL_VERSION_3,     
    H5O_FILL_VERSION_3,     
    H5O_FILL_VERSION_3,     
    H5O_FILL_VERSION_3,     
    H5O_FILL_VERSION_LATEST 
};

#define H5O_FILL_MASK_ALLOC_TIME      0x03
#define H5O_FILL_SHIFT_ALLOC_TIME     0
#define H5O_FILL_MASK_FILL_TIME       0x03
#define H5O_FILL_SHIFT_FILL_TIME      2
#define H5O_FILL_FLAG_UNDEFINED_VALUE 0x10
#define H5O_FILL_FLAG_HAVE_VALUE      0x20
#define H5O_FILL_FLAGS_ALL                                                                                   \
    (H5O_FILL_MASK_ALLOC_TIME | (H5O_FILL_MASK_FILL_TIME << H5O_FILL_SHIFT_FILL_TIME) |                      \
     H5O_FILL_FLAG_UNDEFINED_VALUE | H5O_FILL_FLAG_HAVE_VALUE)

H5FL_DEFINE(H5O_fill_t);

H5FL_BLK_EXTERN(type_conv);

static void *
H5O__fill_new_decode(H5F_t H5_ATTR_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                     unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                     const uint8_t *p)
{
    H5O_fill_t    *fill      = NULL;
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;           

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (fill = H5FL_CALLOC(H5O_fill_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill value message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    fill->version = *p++;
    if (fill->version < H5O_FILL_VERSION_1 || fill->version > H5O_FILL_VERSION_LATEST)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for fill value message");

    
    if (fill->version < H5O_FILL_VERSION_3) {

        

        
        if (H5_IS_BUFFER_OVERFLOW(p, 3, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

        
        fill->alloc_time = (H5D_alloc_time_t)*p++;

        
        fill->fill_time = (H5D_fill_time_t)*p++;

        
        fill->fill_defined = *p++;

        
        if (fill->fill_defined) {

            if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            INT32DECODE(p, fill->size);

            if (fill->size > 0) {
                H5_CHECK_OVERFLOW(fill->size, ssize_t, size_t);

                if (H5_IS_BUFFER_OVERFLOW(p, fill->size, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

                if (NULL == (fill->buf = H5MM_malloc((size_t)fill->size)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill value");
                H5MM_memcpy(fill->buf, p, (size_t)fill->size);
            }
        }
        else
            fill->size = -1;
    }
    else {

        

        unsigned flags; 

        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        flags = *p++;

        
        if (flags & (unsigned)~H5O_FILL_FLAGS_ALL)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "unknown flag for fill value message");

        
        fill->alloc_time =
            (H5D_alloc_time_t)((flags >> H5O_FILL_SHIFT_ALLOC_TIME) & H5O_FILL_MASK_ALLOC_TIME);

        
        fill->fill_time = (H5D_fill_time_t)((flags >> H5O_FILL_SHIFT_FILL_TIME) & H5O_FILL_MASK_FILL_TIME);

        
        if (flags & H5O_FILL_FLAG_UNDEFINED_VALUE) {

            if (flags & H5O_FILL_FLAG_HAVE_VALUE)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "have value and undefined value flags both set");

            
            fill->size = -1;
        }
        else if (flags & H5O_FILL_FLAG_HAVE_VALUE) {
            
            if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT32DECODE(p, fill->size);

            
            H5_CHECK_OVERFLOW(fill->size, ssize_t, size_t);

            if (H5_IS_BUFFER_OVERFLOW(p, fill->size, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

            if (NULL == (fill->buf = H5MM_malloc((size_t)fill->size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill value");
            H5MM_memcpy(fill->buf, p, (size_t)fill->size);

            
            fill->fill_defined = true;
        }
        else
            
            fill->fill_defined = true;
    }

    
    ret_value = (void *)fill;

done:
    if (!ret_value && fill) {
        H5MM_xfree(fill->buf);
        fill = H5FL_FREE(H5O_fill_t, fill);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__fill_old_decode(H5F_t *f, H5O_t *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                     unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_fill_t    *fill      = NULL; 
    htri_t         exists    = false;
    H5T_t         *dt        = NULL;
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;           

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (fill = H5FL_CALLOC(H5O_fill_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill value message");

    
    fill->version    = H5O_FILL_VERSION_2;
    fill->alloc_time = H5D_ALLOC_TIME_LATE;
    fill->fill_time  = H5D_FILL_TIME_IFSET;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT32DECODE(p, fill->size);

    
    if (fill->size > 0) {
        H5_CHECK_OVERFLOW(fill->size, ssize_t, size_t);

        
        if (H5_IS_BUFFER_OVERFLOW(p, fill->size, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

        
        if ((exists = H5O_msg_exists_oh(open_oh, H5O_DTYPE_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, NULL, "unable to read object header");
        if (exists) {
            if (NULL == (dt = (H5T_t *)H5O_msg_read_oh(f, open_oh, H5O_DTYPE_ID, NULL)))
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, NULL, "can't read DTYPE message");
            
            if (fill->size != (ssize_t)H5T_GET_SIZE(dt))
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, NULL, "inconsistent fill value size");
        }

        if (NULL == (fill->buf = H5MM_malloc((size_t)fill->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill value");
        H5MM_memcpy(fill->buf, p, (size_t)fill->size);
        fill->fill_defined = true;
    }
    else
        fill->size = -1;

    
    ret_value = (void *)fill;

done:
    if (dt)
        H5O_msg_free(H5O_DTYPE_ID, dt);

    if (!ret_value && fill) {
        H5MM_xfree(fill->buf);
        H5FL_FREE(H5O_fill_t, fill);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__fill_new_encode(H5F_t H5_ATTR_UNUSED *f, uint8_t *p, const void *_fill)
{
    const H5O_fill_t *fill = (const H5O_fill_t *)_fill;

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(p);
    assert(fill && NULL == fill->type);

    
    *p++ = (uint8_t)fill->version;

    if (fill->version < H5O_FILL_VERSION_3) {
        
        *p++ = (uint8_t)fill->alloc_time;

        
        *p++ = (uint8_t)fill->fill_time;

        
        *p++ = (uint8_t)fill->fill_defined;

        
        if (fill->fill_defined) {
            UINT32ENCODE(p, fill->size);
            if (fill->size > 0)
                if (fill->buf) {
                    H5_CHECK_OVERFLOW(fill->size, ssize_t, size_t);
                    H5MM_memcpy(p, fill->buf, (size_t)fill->size);
                } 
        }         
    }             
    else {
        uint8_t flags = 0; 

        
        assert(fill->alloc_time == (H5O_FILL_MASK_ALLOC_TIME & fill->alloc_time));
        flags =
            (uint8_t)(flags | ((H5O_FILL_MASK_ALLOC_TIME & fill->alloc_time) << H5O_FILL_SHIFT_ALLOC_TIME));

        
        assert(fill->fill_time == (H5O_FILL_MASK_FILL_TIME & fill->fill_time));
        flags = (uint8_t)(flags | ((H5O_FILL_MASK_FILL_TIME & fill->fill_time) << H5O_FILL_SHIFT_FILL_TIME));

        
        if (fill->size < 0) {
            
            flags |= H5O_FILL_FLAG_UNDEFINED_VALUE;

            
            *p++ = (uint8_t)flags;

            
            assert(!fill->buf);
        } 
        else if (fill->size > 0) {
            
            flags |= H5O_FILL_FLAG_HAVE_VALUE;

            
            *p++ = (uint8_t)flags;

            
            INT32ENCODE(p, fill->size);

            
            assert(fill->buf);
            H5_CHECK_OVERFLOW(fill->size, ssize_t, size_t);
            H5MM_memcpy(p, fill->buf, (size_t)fill->size);
        } 
        else {
            
            *p++ = (uint8_t)flags;

            
            assert(!fill->buf);
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__fill_old_encode(H5F_t H5_ATTR_UNUSED *f, uint8_t *p, const void *_fill)
{
    const H5O_fill_t *fill = (const H5O_fill_t *)_fill;

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(p);
    assert(fill && NULL == fill->type);

    UINT32ENCODE(p, fill->size);
    if (fill->buf)
        H5MM_memcpy(p, fill->buf, (size_t)fill->size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__fill_copy(const void *_src, void *_dst)
{
    const H5O_fill_t *src       = (const H5O_fill_t *)_src;
    H5O_fill_t       *dst       = (H5O_fill_t *)_dst;
    H5T_t            *dst_type  = NULL;
    H5T_t            *tmp_type  = NULL;
    void             *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(src);

    if (!dst && NULL == (dst = H5FL_MALLOC(H5O_fill_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill message");

    
    *dst = *src;

    
    if (src->type) {
        if (NULL == (dst->type = H5T_copy(src->type, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "can't copy datatype");
    } 
    else
        dst->type = NULL;

    
    if (src->buf) {
        H5_CHECK_OVERFLOW(src->size, ssize_t, size_t);
        if (NULL == (dst->buf = H5MM_malloc((size_t)src->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for fill value");
        H5MM_memcpy(dst->buf, src->buf, (size_t)src->size);

        
        if (src->type) {
            H5T_path_t *tpath; 

            
            if (NULL == (tpath = H5T_path_find(src->type, dst->type)))
                HGOTO_ERROR(H5E_OHDR, H5E_UNSUPPORTED, NULL,
                            "unable to convert between src and dst data types");

            
            if (!H5T_path_noop(tpath)) {
                uint8_t *bkg_buf = NULL; 
                size_t   bkg_size;       

                dst_type = dst->type;
                if (H5T_detect_class(dst_type, H5T_VLEN, false) > 0 ||
                    H5T_detect_class(dst_type, H5T_REFERENCE, false) > 0) {
                    if (NULL == (tmp_type = H5T_copy(dst_type, H5T_COPY_TRANSIENT)))
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, NULL, "unable to copy destination datatype");
                    dst_type = tmp_type;
                }

                
                bkg_size = MAX(H5T_get_size(dst->type), H5T_get_size(src->type));
                if (H5T_path_bkg(tpath) && NULL == (bkg_buf = H5FL_BLK_CALLOC(type_conv, bkg_size)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

                
                if (H5T_convert(tpath, src->type, dst_type, (size_t)1, (size_t)0, (size_t)0, dst->buf,
                                bkg_buf) < 0) {
                    if (bkg_buf)
                        bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTCONVERT, NULL, "datatype conversion failed");
                } 

                
                if (bkg_buf)
                    bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);
            } 
        }     
    }         
    else
        dst->buf = NULL;

    
    ret_value = dst;

done:
    if (tmp_type && (H5T_close(tmp_type) < 0))
        HDONE_ERROR(H5E_OHDR, H5E_CANTCLOSEOBJ, NULL, "unable to close temporary datatype");

    if (!ret_value && dst) {
        if (dst->buf)
            H5MM_xfree(dst->buf);
        if (dst->type)
            (void)H5T_close_real(dst->type);
        if (!_dst)
            dst = H5FL_FREE(H5O_fill_t, dst);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__fill_new_size(const H5F_t H5_ATTR_UNUSED *f, const void *_fill)
{
    const H5O_fill_t *fill      = (const H5O_fill_t *)_fill;
    size_t            ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(fill);

    
    if (fill->version < H5O_FILL_VERSION_3) {
        ret_value = 1 + 
                    1 + 
                    1 + 
                    1;  
        if (fill->fill_defined)
            ret_value += 4 +                                        
                         (fill->size > 0 ? (size_t)fill->size : 0); 
    }                                                               
    else {
        ret_value = 1 + 
                    1;  
        if (fill->size > 0)
            ret_value += 4 +                 
                         (size_t)fill->size; 
    }                                        

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__fill_old_size(const H5F_t H5_ATTR_UNUSED *f, const void *_fill)
{
    const H5O_fill_t *fill = (const H5O_fill_t *)_fill;

    FUNC_ENTER_PACKAGE_NOERR

    assert(fill);

    FUNC_LEAVE_NOAPI(4 + (size_t)fill->size)
} 

herr_t
H5O_fill_reset_dyn(H5O_fill_t *fill)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(fill);

    if (fill->buf) {
        if (fill->type && H5T_detect_class(fill->type, H5T_VLEN, false) > 0) {
            H5S_t *fill_space; 

            
            if (NULL == (fill_space = H5S_create(H5S_SCALAR)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCREATE, FAIL, "can't create scalar dataspace");

            
            if (H5T_reclaim(fill->type, fill_space, fill->buf) < 0) {
                H5S_close(fill_space);
                HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "unable to reclaim variable-length fill value data");
            } 

            
            H5S_close(fill_space);
        } 

        
        fill->buf = H5MM_xfree(fill->buf);
    } 
    fill->size = 0;
    if (fill->type) {
        (void)H5T_close_real(fill->type);
        fill->type = NULL;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__fill_reset(void *_fill)
{
    H5O_fill_t *fill = (H5O_fill_t *)_fill;

    FUNC_ENTER_PACKAGE_NOERR

    assert(fill);

    
    H5O_fill_reset_dyn(fill);

    
    fill->alloc_time   = H5D_ALLOC_TIME_LATE;
    fill->fill_time    = H5D_FILL_TIME_IFSET;
    fill->fill_defined = false;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__fill_free(void *fill)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(fill);

    fill = H5FL_FREE(H5O_fill_t, fill);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__fill_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void *mesg_src, bool H5_ATTR_UNUSED *deleted,
                        const H5O_copy_t *cpy_info, void H5_ATTR_UNUSED *udata)
{
    const H5O_fill_t *fill_src  = (const H5O_fill_t *)mesg_src; 
    herr_t            ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(cpy_info);
    assert(cpy_info->file_dst);

    
    if (fill_src->version > H5O_fill_ver_bounds[H5F_HIGH_BOUND(cpy_info->file_dst)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "fill value message version out of bounds");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__fill_debug(H5F_t H5_ATTR_UNUSED *f, const void *_fill, FILE *stream, int indent, int fwidth)
{
    const H5O_fill_t *fill = (const H5O_fill_t *)_fill;
    H5D_fill_value_t  fill_status; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(fill);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Space Allocation Time:");
    switch (fill->alloc_time) {
        case H5D_ALLOC_TIME_EARLY:
            Rfprintf(stream, "Early\n");
            break;

        case H5D_ALLOC_TIME_LATE:
            Rfprintf(stream, "Late\n");
            break;

        case H5D_ALLOC_TIME_INCR:
            Rfprintf(stream, "Incremental\n");
            break;

        case H5D_ALLOC_TIME_DEFAULT:
        case H5D_ALLOC_TIME_ERROR:
        default:
            Rfprintf(stream, "Unknown!\n");
            break;
    } 
    Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Fill Time:");
    switch (fill->fill_time) {
        case H5D_FILL_TIME_ALLOC:
            Rfprintf(stream, "On Allocation\n");
            break;

        case H5D_FILL_TIME_NEVER:
            Rfprintf(stream, "Never\n");
            break;

        case H5D_FILL_TIME_IFSET:
            Rfprintf(stream, "If Set\n");
            break;

        case H5D_FILL_TIME_ERROR:
        default:
            Rfprintf(stream, "Unknown!\n");
            break;

    } 
    Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Fill Value Defined:");
    H5P_is_fill_value_defined((const H5O_fill_t *)fill, &fill_status);
    switch (fill_status) {
        case H5D_FILL_VALUE_UNDEFINED:
            Rfprintf(stream, "Undefined\n");
            break;

        case H5D_FILL_VALUE_DEFAULT:
            Rfprintf(stream, "Default\n");
            break;

        case H5D_FILL_VALUE_USER_DEFINED:
            Rfprintf(stream, "User Defined\n");
            break;

        case H5D_FILL_VALUE_ERROR:
        default:
            Rfprintf(stream, "Unknown!\n");
            break;

    } 
    Rfprintf(stream, "%*s%-*s %zd\n", indent, "", fwidth, "Size:", fill->size);
    Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Data type:");
    if (fill->type) {
        H5T_debug(fill->type, stream);
        Rfprintf(stream, "\n");
    } 
    else
        Rfprintf(stream, "<dataset type>\n");

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_fill_convert(H5O_fill_t *fill, H5T_t *dset_type, bool *fill_changed)
{
    H5T_path_t *tpath;                   
    void       *buf = NULL, *bkg = NULL; 
    herr_t      ret_value = SUCCEED;     

    FUNC_ENTER_NOAPI(FAIL)

    assert(fill);
    assert(dset_type);
    assert(fill_changed);

    
    if (!fill->buf || !fill->type || 0 == H5T_cmp(fill->type, dset_type, false)) {
        
        if (fill->type)
            (void)H5T_close_real(fill->type);
        fill->type = NULL;

        
        *fill_changed = true;

        HGOTO_DONE(SUCCEED);
    } 

    
    if (NULL == (tpath = H5T_path_find(fill->type, dset_type)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to convert between src and dst datatypes");

    
    if (!H5T_path_noop(tpath)) {
        size_t fill_type_size;

        
        fill_type_size = H5T_get_size(fill->type);

        if (NULL == (buf = H5MM_malloc(MAX(fill_type_size, H5T_get_size(dset_type)))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for type conversion");
        H5MM_memcpy(buf, fill->buf, fill_type_size);

        
        if (H5T_path_bkg(tpath) && NULL == (bkg = H5MM_calloc(H5T_get_size(dset_type))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for type conversion");

        
        if (H5T_convert(tpath, fill->type, dset_type, (size_t)1, (size_t)0, (size_t)0, buf, bkg) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "datatype conversion failed");

        
        H5T_vlen_reclaim_elmt(fill->buf, fill->type);
        H5MM_xfree(fill->buf);
        fill->buf = buf;

        (void)H5T_close_real(fill->type);
        fill->type = NULL;
        H5_CHECKED_ASSIGN(fill->size, ssize_t, H5T_get_size(dset_type), size_t);

        
        *fill_changed = true;
    } 

done:
    if (bkg)
        H5MM_xfree(bkg);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O_fill_set_version(H5F_t *f, H5O_fill_t *fill)
{
    unsigned version;             
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(fill);

    
    version = MAX(fill->version, H5O_fill_ver_bounds[H5F_LOW_BOUND(f)]);

    
    if (version > H5O_fill_ver_bounds[H5F_HIGH_BOUND(f)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "Filter pipeline version out of bounds");

    
    fill->version = version;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
