/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5HFpkg.h"    

#define H5HF_FSPACE_SHRINK    80  
#define H5HF_FSPACE_EXPAND    120 
#define H5HF_FSPACE_THRHD_DEF 1   
#define H5HF_FSPACE_ALIGN_DEF 1   

herr_t
H5HF__space_start(H5HF_hdr_t *hdr, bool may_create)
{
    const H5FS_section_class_t *classes[] = {
                                             H5HF_FSPACE_SECT_CLS_SINGLE, H5HF_FSPACE_SECT_CLS_FIRST_ROW,
                                             H5HF_FSPACE_SECT_CLS_NORMAL_ROW, H5HF_FSPACE_SECT_CLS_INDIRECT};
    herr_t                      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5_addr_defined(hdr->fs_addr)) {
        
        if (NULL == (hdr->fspace = H5FS_open(hdr->f, hdr->fs_addr, NELMTS(classes), classes, hdr,
                                             (hsize_t)H5HF_FSPACE_THRHD_DEF, (hsize_t)H5HF_FSPACE_ALIGN_DEF)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize free space info");
    } 
    else {
        
        if (may_create) {
            H5FS_create_t fs_create; 

            
            fs_create.client         = H5FS_CLIENT_FHEAP_ID;
            fs_create.shrink_percent = H5HF_FSPACE_SHRINK;
            fs_create.expand_percent = H5HF_FSPACE_EXPAND;
            fs_create.max_sect_size  = hdr->man_dtable.cparam.max_direct_size;
            fs_create.max_sect_addr  = hdr->man_dtable.cparam.max_index;

            
            if (NULL ==
                (hdr->fspace = H5FS_create(hdr->f, &hdr->fs_addr, &fs_create, NELMTS(classes), classes, hdr,
                                           (hsize_t)H5HF_FSPACE_THRHD_DEF, (hsize_t)H5HF_FSPACE_ALIGN_DEF)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize free space info");
            assert(H5_addr_defined(hdr->fs_addr));
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_add(H5HF_hdr_t *hdr, H5HF_free_section_t *node, unsigned flags)
{
    H5HF_sect_add_ud_t udata;               
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(node);

    
    if (!hdr->fspace)
        if (H5HF__space_start(hdr, true) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize heap free space");

    
    udata.hdr = hdr;

    
    if (H5FS_sect_add(hdr->f, hdr->fspace, (H5FS_section_info_t *)node, flags, &udata) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, FAIL, "can't add section to heap free space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5HF__space_find(H5HF_hdr_t *hdr, hsize_t request, H5HF_free_section_t **node)
{
    htri_t node_found = false; 
    htri_t ret_value  = FAIL;  

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(request);
    assert(node);

    
    if (!hdr->fspace)
        if (H5HF__space_start(hdr, false) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize heap free space");

    
    if (hdr->fspace)
        if ((node_found = H5FS_sect_find(hdr->f, hdr->fspace, request, (H5FS_section_info_t **)node)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't locate free space in fractal heap");

    
    ret_value = node_found;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__space_revert_root_cb(H5FS_section_info_t *_sect, void H5_ATTR_UNUSED *_udata)
{
    H5HF_free_section_t *sect      = (H5HF_free_section_t *)_sect; 
    herr_t               ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(sect);

    
    if (sect->sect_info.type == H5HF_FSPACE_SECT_SINGLE && sect->sect_info.state == H5FS_SECT_LIVE) {
        
        assert(sect->u.single.parent);
        if (H5HF__iblock_decr(sect->u.single.parent) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on section's indirect block");

        
        sect->u.single.parent    = NULL;
        sect->u.single.par_entry = 0;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_revert_root(const H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->fspace)
        
        if (H5FS_sect_iterate(hdr->f, hdr->fspace, H5HF__space_revert_root_cb, NULL) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL,
                        "can't iterate over sections to reset parent pointers");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__space_create_root_cb(H5FS_section_info_t *_sect, void *_udata)
{
    H5HF_free_section_t *sect        = (H5HF_free_section_t *)_sect; 
    H5HF_indirect_t     *root_iblock = (H5HF_indirect_t *)_udata;    
    herr_t               ret_value   = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(root_iblock);

    
    
    
    assert(sect->sect_info.type == H5HF_FSPACE_SECT_SINGLE);

    
    if (H5HF__iblock_incr(root_iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL,
                    "can't increment reference count on section's indirect block");

    
    if (sect->sect_info.state == H5FS_SECT_SERIALIZED)
        sect->sect_info.state = H5FS_SECT_LIVE; 
    else
        assert(!sect->u.single.parent);
    sect->u.single.parent    = root_iblock;
    sect->u.single.par_entry = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_create_root(const H5HF_hdr_t *hdr, H5HF_indirect_t *root_iblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(root_iblock);

    
    if (hdr->fspace)
        
        if (H5FS_sect_iterate(hdr->f, hdr->fspace, H5HF__space_create_root_cb, root_iblock) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "can't iterate over sections to set parent pointers");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_size(H5HF_hdr_t *hdr, hsize_t *fs_size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(fs_size);

    
    if (!hdr->fspace)
        if (H5HF__space_start(hdr, false) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize heap free space");

    
    if (hdr->fspace) {
        if (H5FS_size(hdr->fspace, fs_size) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't retrieve FS meta storage info");
    } 
    else
        *fs_size = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_remove(H5HF_hdr_t *hdr, H5HF_free_section_t *node)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->fspace);
    assert(node);

    
    if (H5FS_sect_remove(hdr->f, hdr->fspace, (H5FS_section_info_t *)node) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "can't remove section from heap free space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_close(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->fspace) {
        hsize_t nsects; 

        
        if (H5FS_sect_stats(hdr->fspace, NULL, &nsects) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOUNT, FAIL, "can't query free space section count");

        
        if (H5FS_close(hdr->f, hdr->fspace) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release free space info");
        hdr->fspace = NULL;

        
        if (!nsects) {
            if (H5FS_delete(hdr->f, hdr->fs_addr) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTDELETE, FAIL, "can't delete free space info");
            hdr->fs_addr = HADDR_UNDEF;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_delete(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5FS_delete(hdr->f, hdr->fs_addr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "can't delete to free space manager");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__space_sect_change_class(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, uint16_t new_class)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->fspace);
    assert(sect);

    
    if (H5FS_sect_change_class(hdr->f, hdr->fspace, (H5FS_section_info_t *)sect, new_class) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTMODIFY, FAIL, "can't modify class of free space section");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
