% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/text_transform.R
\name{text_transform}
\alias{text_transform}
\title{Perform text transformations with a custom function}
\usage{
text_transform(data, fn, locations = cells_body())
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{fn}{The function to use for text transformation. It should include \code{x}
as an argument and return a character vector of the same length as the
input \code{x}.}

\item{locations}{The cell or set of cells to be associated with the text
transformation. Only the \code{\link[=cells_body]{cells_body()}}, \code{\link[=cells_stub]{cells_stub()}},
\code{\link[=cells_column_labels]{cells_column_labels()}}, and \code{\link[=cells_row_groups]{cells_row_groups()}} helper functions can be
used here. We can enclose several of these calls within a \code{list()} if we
wish to make the transformation happen at different locations.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Text transforming in \strong{gt} is the act of modifying formatted strings in
targeted cells. The \code{text_transform()} function provides the most flexibility
of all the \verb{text_*()} functions in their family of functions. With it, you
target the cells to undergo modification in the \code{locations} argument while
also supplying a function to the \code{fn} argument. The function given to \code{fn}
should ideally at the very least take \code{x} as an input (it stands for the
character vector that is essentially the targeted cells) and return a
character vector of the same length as the input. Using the construction
\code{function(x) { .. }} for the function is recommended.
}
\section{Examples}{


Use \code{\link{sp500}} to create a \strong{gt} table. Transform the text in the
\code{date} column using a function supplied to \code{text_transform()} (via the \code{fn}
argument). Note that the \code{x} in the \verb{fn = function (x)} part consists
entirely of ISO 8601 date strings (which are acceptable as input to the
\code{\link[=vec_fmt_date]{vec_fmt_date()}} and \code{\link[=vec_fmt_datetime]{vec_fmt_datetime()}} functions).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 |>
  dplyr::slice_head(n = 10) |>
  dplyr::select(date, open, close) |>
  dplyr::arrange(-dplyr::row_number()) |>
  gt() |>
  fmt_currency() |>
  text_transform(
    fn = function(x) \{
      paste0(
        "<strong>",
        vec_fmt_date(x, date_style = "m_day_year"),
        "</strong>",
        "&mdash;W",
        vec_fmt_datetime(x, format = "w")
      )
    \},
    locations = cells_body(columns = date)
  ) |>
  cols_label(
    date = "Date and Week",
    open = "Opening Price",
    close = "Closing Price"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_text_transform_1.png" alt="This image of a table was generated from the first code example in the `text_transform()` help file." style="width:100\%;">
}}

Use \code{\link{gtcars}} to create a \strong{gt} table. First, the numeric values in the \code{n}
column are formatted as spelled-out numbers with \code{\link[=fmt_spelled_num]{fmt_spelled_num()}}. The
output values are indeed spelled out but exclusively with lowercase letters.
We actually want these words to begin with a capital letter and end with a
period. To make this possible, the \code{text_transform()} function will be used
since it can modify already-formatted text. Through the \code{fn} argument, we
provide a custom function that uses R's \code{toTitleCase()} operating on \code{x} (the
numbers-as-text strings) within a \code{paste0()} so that a period can be properly
placed.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::select(mfr, ctry_origin) |>
  dplyr::filter(ctry_origin \%in\% c("Germany", "Italy", "Japan")) |>
  dplyr::group_by(mfr, ctry_origin) |>
  dplyr::count() |>
  dplyr::ungroup() |>
  dplyr::arrange(ctry_origin, desc(n)) |>
  gt(rowname_col = "mfr", groupname_col = "ctry_origin") |>
  cols_label(n = "No. of Entries") |>
  tab_stub_indent(rows = everything(), indent = 2) |>
  cols_align(align = "center", columns = n) |>
  fmt_spelled_num() |>
  text_transform(
    fn = function(x) \{
      paste0(tools::toTitleCase(x), ".")
    \},
    locations = cells_body(columns = n)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_text_transform_2.png" alt="This image of a table was generated from the second code example in the `text_transform()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

4-4
}

\section{Function Introduced}{

\code{v0.2.0.5} (March 31, 2020)
}

\seealso{
Other text transforming functions: 
\code{\link{text_case_match}()},
\code{\link{text_case_when}()},
\code{\link{text_replace}()}
}
\concept{text transforming functions}
