% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fold.R
\name{fold}
\alias{fold}
\title{Create balanced folds for cross-validation.}
\usage{
fold(data, k = 5, cat_col = NULL, id_col = NULL, method = "n_dist")
}
\arguments{
\item{data}{Dataframe or Vector}

\item{k}{Number of folds, fold size, or step size (depending on chosen method)
Given as whole numbers or percentage (0 < n < 1).}

\item{cat_col}{Categorical variable to balance between folds.

 E.g. when predicting a binary variable (a or b), it is necessary to have
 both represented in every fold

 N.B. If also passing an id_col, cat_col should be a constant for that ID.}

\item{id_col}{Factor with IDs.
 This will be used to keep all rows with an ID in the same fold
 (if possible).

 E.g. If we have measured a participant multiple times and want to see the
 effect of time, we want to have all observations of this participant in
 the same fold.}

\item{method}{\code{greedy, n_dist, n_fill, n_last, n_rand, or staircase}

 \code{greedy} divides up the data greedily given a specified group size
 \eqn{(e.g. 10, 10, 10, 10, 10, 7)}

 \code{n_dist} divides the data into a specified number of groups and
 distributes excess data points across groups
 \eqn{(e.g. 11, 11, 12, 11, 12)}

 \code{n_fill} divides the data into a specified number of groups and
 fills up groups with excess data points from the beginning
 \eqn{(e.g. 12, 12, 11, 11, 11)}

 \code{n_last} divides the data into a specified number of groups.
 The algorithm finds the most equal group sizes possible,
 using all data points. Only the last group is able to differ in size
 \eqn{(e.g. 11, 11, 11, 11, 13)}

 \code{n_rand} divides the data into a specified number of groups.
 Excess data points are placed randomly in groups (only 1 per group)
 \eqn{(e.g. 12, 11, 11, 11, 12)}

 \code{staircase} uses step_size to divide up the data.
 Group size increases with 1 step for every group,
 until there is no more data
 \eqn{(e.g. 5, 10, 15, 20, 7)}}
}
\value{
Dataframe with grouping factor for subsetting in cross-validation.
}
\description{
Divides data into groups by a range of methods.
 Balances a given categorical variable between folds and keeps (if possible)
 all data points with the same ID (e.g. participant_id) in the same fold.
}
\details{
cat_col: data is first subset by cat_col.
 Subsets are folded/grouped and merged. ||
 id_col: folds are created from unique IDs. ||
 cat_col AND id_col: data is subset by cat_col
 and folds are created from unique IDs in each subset.
 Subsets are merged.
}
\examples{
# Attach packages
library(groupdata2)
library(dplyr)

# Create dataframe
df <- data.frame(
 "participant" = factor(rep(c('1','2', '3', '4', '5', '6'), 3)),
 "age" = rep(sample(c(1:100), 6), 3),
 "diagnosis" = rep(c('a', 'b', 'a', 'a', 'b', 'b'), 3),
 "score" = sample(c(1:100), 3*6))
df <- df[order(df$participant),]
df$session <- rep(c('1','2', '3'), 6)

# Using fold()
# Without cat_col and id_col
df_folded <- fold(df, 3, method = 'n_dist')

# With cat_col
df_folded <- fold(df, 3, cat_col = 'diagnosis',
 method = 'n_dist')

# With id_col
df_folded <- fold(df, 3, id_col = 'participant',
 method = 'n_dist')

# With cat_col and id_col
df_folded <- fold(df, 3, cat_col = 'diagnosis',
 id_col = 'participant', method = 'n_dist')

# Order by folds
df_folded <- df_folded[order(df_folded$.folds),]

}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}

