\name{gmeta}
\alias{gmeta}
\alias{gmeta.default}
\alias{print.gmeta.e}
\alias{print.gmeta.m}
\alias{print.gmeta.p}
\alias{print.summary.gmeta.e}
\alias{print.summary.gmeta.m}
\alias{print.summary.gmeta.p}
\alias{summary.gmeta.e}
\alias{summary.gmeta.m}
\alias{summary.gmeta.p}
\alias{hiring}
\alias{ulcer}

\title{
	Meta-Analysis via a Unified Framework under Confidence Distribution
}
\description{
	A unified method for meta-analysis includes combining p-values, fitting meta-analysis fixed-effect and random-effects models, and synthesizing 2x2 tables evidence all under a framework of combining confidence distributions (CDs). The function produces an object of class \code{gmeta} with associated functions \code{print}, \code{summary}, and \code{plot}.
}
\usage{
gmeta(gmi, gmi.type = c('pivot', 'cd', 'pvalue', '2x2'),
	method = c('fixed-mle',
		'fixed-robust1', 'fixed-robust2', 'fixed-robust2(sqrt12)',
		'random-mm', 'random-reml', 'random-tau2',
		'random-robust1', 'random-robust2', 'random-robust2(sqrt12)',
		'fisher', 'normal', 'stouffer', 'min', 'tippett', 'max', 'sum',
		'MH', 'Mantel-Haenszel', 'Peto', 'exact1', 'exact2'),
	linkfunc = c('inverse-normal-cdf', 'inverse-laplace-cdf'), 
	weight = NULL, study.names = NULL, gmo.xgrid = NULL, ci.level = 0.95, 
	tau2 = NULL, mc.iteration = 10000, eta = 'Inf', verbose = FALSE, 
	report.error = FALSE)
}
\arguments{
	\item{gmi}{input, see \sQuote{Details}.}
	\item{gmi.type}{type of input, including 'pivot', 'cd', 'pvalue', and '2x2', see \sQuote{Details}.}
	\item{method}{method used for meta-analysis, including 'fisher', 'normal', 'stouffer', 'min', 'tippett', 'max', and 'sum' for combining p-values; 'fixed-mle', 'fixed-robust1', 'fixed-robust2', 'fixed-robust2(sqrt12)' for fixed-effect meta-analysis models; 'random-mm', 'random-reml', 'random-tau2', 'random-robust1', 'random-robust2', and 'random-robust2(sqrt12)' for random-effects meta-analysis models; and 'MH', 'Mantel-Haenszel', 'Peto', 'exact1', and 'exact2' for synthesizing 2x2 tables, see \sQuote{Details}.}
	\item{linkfunc}{the link function selected for a user specified combination method, see Yang et al. (2013) and \sQuote{Details}.}
	\item{weight}{a vector of user-specified weights for each study. If NULL, the default value depends on the \code{method} and \code{linkfunc} used for meta-analysis.}
	\item{study.names}{a vector of strings to give a user-specified study name for each study. If NULL, the default will be 'study-1', ..., 'study-k'.}
	\item{gmo.xgrid}{the position to evaluate a combined CD. The output will be reported as empirical cumulative density function (ECDF) on the points specified by gmo.xgrid. If NULL, the default will be \code{gmo.xgrid = seq(from=-1,to=1,by=0.001)}.}
	\item{ci.level}{the confidence level for confidence intervals.}
	\item{tau2}{a numeric value to provide the heterogeneity estimation or a string to specified the method used to estimate the heterogeneity, see \sQuote{Details}.}
	\item{mc.iteration}{number of iterations to compute the error of coverage probability of the computed confidence interval in 2x2-"exact1" method.}
	\item{eta}{a numeric vector for confidence levels of the one-sided confidence intervals for combining 2x2 tables used in "exact2" method. For example, set \code{eta=Inf} to indicate all confidence levels, or set \code{eta=seq(from=0.05,to=0.95,length=23)}.}
	\item{verbose}{a logical value indicating whether detailed combining information is produced.}
	\item{report.error}{a logical value indicating whether the exact error of coverage probability of the computed confidence interval for 2x2-"exact1" and 2x2-"exact2" method is reported}
}
\details{
	\bold{gmi}

	The format of \code{gmi} depends on the value of \code{gmi.type} (see below in this section).

	For a classical p-value combination, \code{gmi} is a vector of p-values for testing the same hypothesis. For example, \code{gmi=c(0.02,0.03,0.14)} with \code{gmi.type='pvalue'}.

	For model-based meta-analysis, \code{gmi} is a list of CDs if \code{gmi.type='cd'} with \code{gmi.type='cd'} (i.e., \code{x=seq(from=-10,to=10,by=0.001), gmi=rbind(pnorm(x,2,3), pt(x-1,4), pgamma(x-1,1,1))}); \code{gmi} is a matrix or a data.frame two columns of mean and standard deviations with \code{gmi.type='pivot'} (i.e., \code{gmi=data.frame(mean=c(2,1,1),sd=(3,4,1))}).
	
	Note that the input confidence distributions should be as complete as possible, which means the distributions should start from almost 0 and end at almost 1, though it is fine if not so. If the distributions are not all specified under different grids, interpolation will be used for interpolating the corresponding probabilities, the closest point values will used for extrapolating the probabilities outside of the original range (see \code{\link{approx}}). If summary statistics is provided, the corresponding confidence distribution is generated by normal approximation and within 4 standard deviations range symmetric around means.

	For 2x2 table-based (log) odds ratio/risk difference combination, \code{gmi} is a matrix of Kx4, where K is the number of trials. The first and third column are number of events in case and control group respectively. The second and fourth column are marginal total of case and control group respectively.

	\bold{gmi.type}

	\code{gmi.type} is a string specifying the type of input data set. The choices are \code{pvalue} for classical p-value combination, \code{cd} for model-based meta-analysis using a list of CDs, \code{pivot} for model-based meta-analysis using summary statistics (means and standard deviations), and \code{pivot} for 2x2 table-based (log) odds ratio/risk difference combination.

	\bold{method}

	\code{method} is a string specifying the method used for meta-analysis.

	For classical p-value combination, choices are \code{fisher}, \code{normal}, \code{stouffer}, \code{tippett}, \code{min}, \code{max}, and \code{sum}.	

	For model-based meta-analysis, choices are \code{fixed-mle}, \code{fixed-robust1}, \code{fixed-robust2}, \cr 
	\code{fixed-robust2(sqrt12)}, \code{random-mm}, \code{random-reml}, \code{random-tau2}, \code{random-robust1}, \cr
	\code{random-robust2}, and \code{random-robust2(sqrt12)}. 

	For 2x2 table-based (log) odds ratio/risk difference combination, choices are \code{exact1}, \code{exact2}, \code{ Mantel-Haenszel}, \code{MH}, and \code{Peto}.

	\bold{linkfunc}

	\code{linkfunc} is the link function used for combining studies.	

	The choice of \code{inverse-normal-cdf} covers most elementary model-based meta-analysis, and achieves the Fisher efficiency asymptotically.

	The choice of \code{inverse-laplace-cdf} is more robust and achieves Bahadur efficiency.
	The default option of \code{linkfunc} is \code{inverse-normal-cdf} for model-based meta-analysis, and null for p-value or 2x2 table combination.

	\bold{tau2}

	\code{tau2} is either a numeric value for estimating heterogeneity, or a string specifying the method to estimate heterogeneity. 

	\code{tau2} is only for meta-analysis random-effects models (with \code{method}=\code{random-mm}, \code{random-reml}, \code{random-tau2}, \code{random-robust1}, \code{random-robust2}, or \code{random-robust2(sqrt12)})).

	Choices for \code{tau2} are \code{DL}, \code{HS}, \code{SJ}, \code{HE}, \code{ML}, \code{REML} and \code{EB} for DerSimonian-Laird, Hedges, Sidik-Jonkman, Hunter-Schmidt, Maximum-Likelihood, Restricted-Maximum-Likelihood, and Empirical-Bayesian estimator, respectively.
}
\value{

	An object of class \code{gmeta}, which has information of the combined inference (summarized in a CD form).

	For p-value combination, it is a list of \code{individual.pvalues}, \code{method}, and \code{combined.pvalue}.

	For model-based meta-analysis and 2x2 combination, it is a list of \code{x.grids}, \code{individual.cds}, \code{individual.means}, \code{individual.stddevs}, \code{individual.medians}, \code{individual.cis}, \code{combined.cd}, \code{combined.density}, \code{combined.mean}, \code{combined.sd}, \code{combined.median}, \code{individual.ci}, \code{method}, \code{linkfunc}, \code{weight}, \code{tau2}, \code{ci.level}, etc.
}
\author{
	Guang Yang <gyang.rutgers@gmail.com>, Jerry Q. Cheng <jcheng1@rwjms.rutgers.edu>, and Minge Xie <mxie@stat.rutgers.edu>
}
\note{
	Revised on 2014/12/10.
}
\references{ 
	Efron, B. (1996)
	Empirical Bayes Methods for Combining Likelihoods.
	\emph{Journal of the American Statistical Association}, \bold{91} 538--550. 

	Liu, D., Liu, R. and Xie, M. (2014)
	Exact meta-analysis approach for discrete data and its application to 2x2 tables with rare events.
	\emph{Journal of the American Statistical Assocation}, \bold{109} 1450-1465.

	Mantel, N. and Haenszel, W. (1959)
	Statistical aspects of the analysis of data from retrospective studies of disease. 
	\emph{Journal of the National Cancer Institute}, \bold{22} 719-748.

	Robins, J., Breslow, N. and Greenland, S. (1986)
	Estimators of the Mantel-Haenszel Variance Consistent in Both Sparse Data and Large-Strata Limiting Models.
	\emph{Biometrics}, \bold{42} 311-323.
	
	Tian, L., Cai, T., Pfeffer, M. A., Piankov, N., Cremieux, P.-Y., and Wei, L. J. (2009)
	Exact and efficient inference procedure for meta-analysis and its application to the analysis of independent 2x2 tables with all available data but without artifficial continuity correction.
	\emph{Biostatistics}, \bold{10} 275-281.

	Xie, M. and Singh, K. (2013)
	Confidence distribution, the frequentist distribution estimator of a parameter (with discussions). 
	\emph{International Statistical Review}, \bold{81} 3-39.

	Xie, M., Singh, K., and Strawderman, W. E. (2011).
	Confidencedence distributions and a unifying framework for meta-analysis.
	\emph{Journal of the American Statistical Association}, \bold{106} 320-333.

	Yang, G., Lui, D., Wang, J. and Xie, M. (2015). 
	Meta-analysis of rare events in drug safety studies: an all-embracing framework for exact inferences. 
	\emph{Preprint}.

	Yusuf, S., Peto, R., Lewis, J., Colins, R. and P. Sleight(1985)
	Beta blockade during and after myocardial infarction-An overview of randomized trials.
	\emph{Progress in Cardiovascular Disease}, \bold{27} 335-371.
}
\seealso{
	\code{\link{plot.gmeta}}
}
\examples{
#### gmeta: generalized meta-analysis approach ####

data(ulcer)
ulcer.o <- as.matrix(ulcer)

# p-value combination #
# impute 0.5
ulcer <- ifelse(ulcer.o==0, 0.5, ulcer.o)
# summary statistics
ulcer.theta <- log( (ulcer[,1]*ulcer[,4]) / (ulcer[,2]*ulcer[,3]) ) 
ulcer.sigma <- sqrt(1/ulcer[,1] + 1/ulcer[,2] + 1/ulcer[,3] + 1/ulcer[,4]) 
# p-values from individual studies for K0: LOR >=0 vs. Ka: LOR < 0
ulcer.pvalues <- pnorm(ulcer.theta, mean=0, sd=ulcer.sigma)
# p-value combination using gmeta function
gmo.pvalue <- gmeta(ulcer.pvalues, gmi.type='pvalue', method='normal')
gmo.pvalue <- gmeta(ulcer.pvalues, gmi.type='pvalue', method='tippett')
print(gmo.pvalue)
summary(gmo.pvalue)

# model-based meta-analysis #
# data.frame of summary statistics
ulcer.pivots <- data.frame(mns=ulcer.theta, sds=ulcer.sigma)
# fixed-effect model
gmo.mdlfx <- gmeta(ulcer.pivots, method='fixed-mle', gmo.xgrid=seq(from=-10,to=10,by=0.01))
summary(gmo.mdlfx)
# random-effects model, method of moments
gmo.mdlrm <- gmeta(ulcer.pivots, method='random-tau2', weight=rep(1,41), tau2=2, 
	gmo.xgrid=seq(from=-10,to=10,by=0.01)) 
summary(gmo.mdlrm)
# plot of the gmeta output - forest plot of CDs
plot(gmo.mdlrm, studies=c(4,8,15,16,23,41)) # default: confidence-distribution-density
plot(gmo.mdlrm, studies=c(4,8,15,16,23,41), plot.option='cv') # using confidence-curve

# 2x2 table-based (log) odds ratio/risk difference combination #
# MH odd-ratio (OR) and Peto's log-odd-ratio (LOR)
ulcer.2x2 <- cbind(ulcer[,1], ulcer[,1]+ulcer[,2], ulcer[,3], ulcer[,3]+ulcer[,4])
# Mantel-Haenszel odd-ratio
gmo.2x2MH <- gmeta(ulcer.2x2, gmi.type='2x2', method='MH', gmo.xgrid=seq(-5,5,by=0.001))
summary(gmo.2x2MH)
plot(gmo.2x2MH, studies=c(4,8,15,16,23,41))
# Peto's log-odd-ratio
gmo.2x2Pt <- gmeta(ulcer.2x2, gmi.type='2x2', method='Peto', gmo.xgrid=seq(-5,5,by=0.001))
summary(gmo.2x2Pt)
plot(gmo.2x2Pt, studies=c(4,8,15,16,23,41))
# Exact meta-analysis on LOR based on Liu et al (2012) and RD based on Tian et al (2009)
ulcer.exact <- cbind(ulcer.o[,1], ulcer.o[,1]+ulcer.o[,2], ulcer.o[,3], ulcer.o[,3]+ulcer.o[,4])
# Exact meta-analysis on log-odd-ratio (LOR) based on Liu et al (2012)
#gmo.exact1 <- gmeta(ulcer.exact, gmi.type='2x2', method='exact1', 
#	gmo.xgrid=seq(-5,5,by=0.001), report.error=TRUE) # log-odd-ratio 
#summary(gmo.exact1)
#plot(gmo.exact1, studies=c(4,8,15,16,23,41))
# Exact meta-analysis on risk difference (RD) based on Tian et al (2009)
#gmo.exact2 <- gmeta(ulcer.exact, gmi.type='2x2', method='exact2', 
#	gmo.xgrid=seq(-1,1,by=0.001), report.error=TRUE) # risk-difference 
#summary(gmo.exact2)
#plot(gmo.exact2, studies=c(4,8,15,16,23,41), plot.option='cv')
}
\keyword{meta-analysis}
