% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rtgin.R
\name{rtgin}
\alias{rtgin}
\title{Generating random numbers from the generalized inverse normal distribution truncated to the positive or negative reals}
\usage{
rtgin(
  size,
  alpha,
  mu,
  tau,
  sign,
  algo = "hormann",
  method = "Fortran",
  verbose = FALSE
)
}
\arguments{
\item{size}{number of desired draws. Output is numpy vector of length equal to size.}

\item{alpha}{degrees-of-freedom parameter.}

\item{mu}{similar to location parameter, controls asymmetry of the distribution.}

\item{tau}{similar to scale parameter, controls spread of the distribution.}

\item{sign}{logical. \code{TRUE} implies truncation to positive numbers (\code{z} > 0)
and \code{FALSE} to negative numbers (\code{z} < 0).}

\item{algo}{string with desired algorithm to compute minimal bounding rectangle.
If "hormann", use the method from Hörmann and Leydold (2014). When "leydold", use the one from Leydold (2001).
Defaults to "hormann" and returns an error for any other values.}

\item{method}{string with the method used to compute the parabolic cylinder function
in the normalization constant. \code{method = "Fortran"} uses a compiled Fortran version,
which is the default. \code{method = "R"} uses an R translation of this function.}

\item{verbose}{logical; should the acceptance rate from the ratio-of-uniforms
method be provided along with additional information? Defaults to FALSE.}
}
\value{
If \code{verbose = FALSE} (default), a numeric vector of length \code{size}.
Otherwise, a list with components \code{value}, \code{avg_arate}, and \code{ARiters}
}
\description{
Generating random numbers from the generalized inverse normal distribution truncated to the positive or negative reals
}
\details{
Currently, only values of \code{alpha} > 2 are supported. For Bayesian posterior sampling,
\code{alpha} is always larger than 2 even for non-informative priors.
Generate from positive region (\code{z} > 0) hen \code{sign = TRUE}, and from
negative region (\code{z} < 0) when \code{sign = FALSE}. When \code{verbose = TRUE},
a list is returned containing the actual draw in \code{value}, as well as average
acceptance rate \code{avg_arate} and total number of acceptance-rejection steps \code{ARiters}.
}
\examples{
# Generate 1000 values from the truncated distributions with alpha = 5, mu = 0, tau = 1
set.seed(123456)
n_draws <- 1000
z_p <- rtgin(n_draws, 5, 0, 1, TRUE)
z_n <- rtgin(n_draws, 5, 0, 1, FALSE)

# Compare generation from truncation to positive reals with true density
n_values <- 200
z_vals <- seq(-5, 5, length.out = n_values)
fz_p <- sapply(z_vals[z_vals > 0], function(z) dtgin(z, 5, 0, 1, TRUE, FALSE))
fz_p <- c(rep(0, n_values - sum(z_vals > 0)), fz_p)
temp <- hist(z_p, breaks = 100, plot = FALSE)
plot(temp, freq = FALSE, xlim = c(-5, 5), ylim = range(c(fz_p, temp$density)),
     main = '', xlab = 'Values', ylab = 'Density', col = 'blue')
lines(z_vals, fz_p, col = 'red', lwd = 2)

# Compare generation from truncation to negative reals with true density
fz_n <- sapply(z_vals[z_vals < 0], function(z) dtgin(z, 5, 0, 1, FALSE, FALSE))
fz_n <- c(fz_n, rep(0, n_values - sum(z_vals < 0)))
temp <- hist(z_n, breaks = 100, plot = FALSE)
plot(temp, freq = FALSE, xlim = c(-5, 5), ylim = range(c(fz_n, temp$density)),
     main = '', xlab = 'Values', ylab = 'Density', col = 'blue')
lines(z_vals, fz_n, col = 'red', lwd = 2)

# verbose = TRUE provides info on the acceptance rate of the
# ratio-of-uniforms acceptance-rejection method for sampling the variables
draw_list <- rtgin(50, 5, 0, 1, sign = TRUE, verbose = TRUE)
draw_list$ARiters      # Acceptance-Rejection iterations
draw_list$avg_arate    # Average of 1/ARiters
}
