% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normalize.R
\name{normalize}
\alias{normalize}
\title{Indicator Normalization}
\usage{
normalize(inds, method = c("min-max", "goalpost"), ind.pol,
          gp.range = c(70, 130), time = NULL, ref.time = NULL,
          ref.value = NULL)
}
\arguments{
\item{inds}{a numeric vector, matrix, or data frame which provides indicators
to be normalized.}

\item{method}{normalization method to be used. See \sQuote{Details}.}

\item{ind.pol}{a character vector whose elements can be \code{"pos"} (positive) or
\code{"neg"} (negative), indicating the polarity of indicators. An indicator's
polarity is the sign of the relation between the indicator and the
phenomenon to be measured.}

\item{gp.range}{a vector of the form \code{c(a,b)} giving the normalization range
for method \code{"goalpost"}. The default value is \code{c(70,130)}.}

\item{time}{a vector of temporal factors for input indicators. The length of
\code{time} must equal the number of rows in \code{inds}. If \code{NULL}, the input data
will be treated as cross-sectional.}

\item{ref.time}{a value denoting the reference time for normalization. See
\sQuote{Details}.}

\item{ref.value}{a vector containing reference values for indicators to
facilitate the interpretation of results, required by method \code{"goalpost"}.
When normalizing each indicator, their reference values will be mapped to
the midpoint of \code{gp.range}. See \sQuote{Details}.}
}
\value{
An object of class \code{"data.frame"} containing normalized indicators.
}
\description{
Perform normalization based on indicators' polarity.
}
\details{
By default, each indicator \eqn{x} is normalized by method \code{"min-max"} with
the formulas
\deqn{\displaystyle\tilde{x}^{+}_i = \frac{x_i -
\mathrm{inf}_x}{\mathrm{sup}_x - \mathrm{inf}_x},}
or
\deqn{\displaystyle \tilde{x}^{-}_i = 1 - \tilde{x}^{+}_i,}
where \eqn{\mathrm{sup}_x} and \eqn{\mathrm{inf}_x} are respectively the
superior and inferior values of the indicator. The former formula is applied
to indicators with positive polarity while the latter one is used for those
with negative polarity.

If either \code{time} or \code{ref.time} is \code{NULL}, the superior and inferior values
are respectively the maximum and minimum values of \eqn{x}. If both \code{time}
and \code{ref.time} are not \code{NULL}, the superior and inferior values are
respectively the maximum and minimum values of \eqn{x} observed at the
reference time. In other words, if \code{time} is not provided or provided without
specifying a value for \code{ref.time}, the input data will be treated as
cross-sectional.

For method \code{"goalpost"}, a vector of reference values for indicators is
required. If not specified by users (\code{ref.value = NULL}), these values are
automatically set #' to the indicator means for cross-sectional data or to
the indicator means at the reference time for longitudinal data.

Method \code{"goalpost"} computes two goalposts for normalization as
\eqn{\mathrm{gp\_min}_x = \mathrm{ref}_x - \Delta} and
\eqn{\mathrm{gp\_max}_x = \mathrm{ref}_x + \Delta}, where
\eqn{\mathrm{ref}_x} is the reference value of \eqn{x} and
\eqn{\Delta = (\mathrm{sup}_x - \mathrm{inf}_x)/2}. Indicators with
positive polarity are rescaled using the formula
\deqn{\displaystyle \tilde{x}^{+}_i =
\frac{x_i - \mathrm{gp\_min}_x}{\mathrm{gp\_max}_x -
\mathrm{gp\_min}_x} (b - a) + a,}
while indicators with negative polarity are rescaled using the formula
\deqn{\displaystyle \tilde{x}^{-}_i = a + b - \tilde{x}^{+}_i.}
If an indicator follows a symmetric probability distribution and its
reference value is set to the mean, the normalized values will theoretically
remain in the range \eqn{[a,b]}. In other cases, the normalized values may
extend beyond \code{gp.range}.
}
\examples{
# Generate data samples
set.seed(1)
df1 <- data.frame(X1 = rnorm(100, 0, 5),
                  X2 = runif(100, 1, 10),
                  X3 = rpois(100, 10))
set.seed(1)
df2 <- data.frame(X1 = rnorm(300, 0, 5),
                  X2 = runif(300, 1, 10),
                  X3 = rpois(300, 10),
                  time = rep(c(2020:2022), rep(100,3)))

# Min-max normalization
df1.mm <- normalize(inds = df1,
                    ind.pol = c("pos", "neg", "pos"))
summary(df1.mm)
df2.mm <- normalize(inds = df2[, 1:3],
                    ind.pol = c("pos", "neg", "pos"),
                    time = df2[, 4], ref.time = 2020)
summary(df2.mm)

# Goalpost normalization
df1.gp <- normalize(inds = df1, method = "goalpost",
                    ind.pol = c("pos", "neg", "pos"))
summary(df1.gp)
df2.gp <- normalize(inds = df2[, 1:3], method = "goalpost",
                    ind.pol = c("pos", "neg", "pos"),
                    time = df2[, 4], ref.time = 2020)
summary(df2.gp)
}
\references{
Mazziotta, M. and Pareto, A. (2016). On a Generalized Non-compensatory
Composite Index for Measuring Socio-economic Phenomena. \emph{Social
Indicators Research}, \bold{127}, 983--1003.
}
\seealso{
\code{\link{giniCI}}.
}
\author{
Viet Duong Nguyen, Chiara Gigliarano, Mariateresa Ciommi
}
