% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/pairwiseD.test.r
\name{pairwiseD.test}
\alias{pairwiseD.test}
\title{Pairwise Group Comparisons}
\usage{
pairwiseD.test(f1, f2 = NULL, RRPP = FALSE, int.first = FALSE,
  iter = 999)
}
\arguments{
\item{f1}{A formula for the linear model from which groups are to be compared (e.g., y~x1*x2)}

\item{f2}{A right-side formula for one or more covariates (e.g., ~ CS + altitude)}

\item{RRPP}{a logical value indicating whether residual randomization should be used for significance testing}

\item{int.first}{A logical value to indicate if interactions of first main effects should precede subsequent main effects}

\item{iter}{Number of iterations for permutation test}
}
\value{
Function returns a list with the following components:
  \item{Obs.dist}{A matrix of Euclidean distances among group means or Least Squares group means}
  \item{Prob.Dist}{A matrix of pairwise significance levels based on permutation}
}
\description{
Function performs pairwise comparisons among groups using the Euclidean distances among group means.
}
\details{
The function performs pairwise comparisons to identify shape differences among groups. The function is designed as a post-hoc
 test to Procrustes ANOVA, where the latter has identified significant shape variation explained by a grouping factor.

The Function can handle, single-factor ANOVA, factorial ANOVA, and ANCOVA designs.

As input the user provides a formula describing the linear model of how shape (y) varies as a function of a factor (a)
 or factorial interaction (a*b). A single covariate, matrix of covariates, or data frame of covariates can also be added.
 E.g., covariates = x1, covariates = cbind(x1, x2, x3,...), or covariates = data.frame(x1, x2, x3,...).
 The shape data (y) can be in the form of a two-dimensional data matrix of dimension (n x [p x k]) or a 3D array (p x k x n).
 It is assumed that the landmarks have previously been aligned using Generalized Procrustes Analysis (GPA)
 [e.g., with \code{\link{gpagen}}]. From the data, the Euclidean distances among group means are estimated, and used
 as test values.

To evaluate significance of group differences, two possible resampling procedures are provided. First, if
  RRPP=FALSE, the rows of the matrix of shape variables are randomized relative to the design matrix. This is
  analogous to a 'full' randomization. Second, if RRPP=TRUE, a residual randomization permutation procedure
  is utilized (Collyer et al. 2014). Here, residual shape values from a reduced model are
  obtained, and are randomized with respect to the linear model under consideration. These are then added to
  predicted values from the remaining effects to obtain pseudo-values from which SS are calculated. NOTE: for
  single-factor designs, the two approaches are identical.  However, when evaluating factorial models it has been
  shown that RRPP attains higher statistical power and thus has greater ability to identify patterns in data should
  they be present (see Anderson and terBraak 2003). Effect-sizes (Z-scores) are computed as standard deviates of the SS
  sampling distributions generated, which might be more intuitive for P-values than F-values (see Collyer et al. 2014).

This test is essentially the same as procD.lm with post-hoc comparisons among least squares (LS) means for appropriate
  models.  However, differences in means are calculated simultaneously with the same random permutations peformed for ANOVA,
  making it less so a post-hoc test and more so a simultaneous test of pairwise contrasts (see Collyer et al. 2014).
}
\examples{
data(plethodon)
Y.gpa<-gpagen(plethodon$land)    #GPA-alignment
y<-two.d.array(Y.gpa$coords)
### Procrustes ANOVA
procD.lm(y~plethodon$species,iter=99)

### Pairwise comparisons: ANOVA design with full randomization
pairwiseD.test(y~plethodon$species*plethodon$site,iter=9)

### Pairwise comparisons: ANOVA design with residual randomization
pairwiseD.test(y~plethodon$species*plethodon$site,iter=9,RRPP=TRUE)

### Pairwise comparisons: ANCOVA design with full randomization
pairwiseD.test(y~plethodon$species*plethodon$site, ~ Y.gpa$Csize, iter=9)

### Pairwise comparisons: ANCOVA design with residual randomization
pairwiseD.test(y~plethodon$species*plethodon$site, ~ Y.gpa$Csize, iter=9, RRPP = TRUE)
}
\author{
Michael Collyer and Dean Adams
}
\references{
Anderson MJ. and C.J.F. terBraak. 2003. Permutation tests for multi-factorial analysis of variance.
   Journal of Statistical Computation and Simulation 73: 85-113.

Collyer, M.L., D.J. Sekora, and D.C. Adams. 2015. A method for analysis of phenotypic change for phenotypes described
by high-dimensional data. Heredity. 113: doi:10.1038/hdy.2014.75.
}
\keyword{analysis}

