% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emfrail_arguments.R
\name{emfrail_control}
\alias{emfrail_control}
\title{Control parameters for emfrail}
\usage{
emfrail_control(eps = 1e-04, maxit = Inf, opt_fit = TRUE,
  verbose = FALSE, fast_fit = TRUE, zerotol = 1e-04, se_fit = TRUE,
  se_adj = TRUE, ca_test = TRUE, only_ca_test = FALSE,
  opt_control = list(method = "bobyqa", itnmax = NULL, control = list()))
}
\arguments{
\item{eps}{Convergence criterion for the inner loops (the EM algorithm) for fixed frailty parameters}

\item{maxit}{Maximum number of iterations in the inner loops (the EM algorithm)}

\item{opt_fit}{Logical. Whether the outer optimization should be carried out.
If \code{FALSE}, then the frailty parameter is treated as fixed and the \code{emfrail} function returns only log-likelihood. See details.}

\item{verbose}{Logical. Whether to print out information about what is going on during maximization.}

\item{fast_fit}{Logical. Whether to try to calculate the E step directly, when possible. See details.}

\item{zerotol}{Lower limit for 1/frailtypar (variance in case of gamma / pvf). Below this value, the variance is taken to be 0 and the
EM is not actually performed. The log-likelihood returned to the maximizer is that for the Cox model.}

\item{se_fit}{Logical. Whether to calculate the variance / covariance matrix.}

\item{se_adj}{Logical. Whether to calculate the adjusted variance / covariance matrix (needs \code{se_fit == TRUE})}

\item{ca_test}{Logical. Should the Commenges-Andersen test be calculated?}

\item{only_ca_test}{Logical. Should ONLY the Commenges-Andersen test be calculated?}

\item{opt_control}{A list with arguments to be sent to the maximizer.}
}
\value{
An object of the type \code{emfrail_control}.
}
\description{
Control parameters for emfrail
}
\details{
The \code{fast_fit} option make a difference when the distribution is gamma (with or without left truncation) or
inverse Gaussian, i.e. pvf with m = -1/2 (without left truncation). For all the other scenarios, the fast_fit option will
automatically be changed to FALSE. When the number of events in a cluster / individual is not very small, the cases for which
fast fitting is available will show an improvement in performance.

The \code{zerotol} option defaults to \code{1e-04}, which in practical terms means, for example, that for
the gamma / pvf distribution, a frailty variance below \code{1e-04} can not be detected.
}
\examples{
emfrail_control()
emfrail_control(eps = 10e-7)

\dontrun{
# A data set with very small heterogeneity
set.seed(10)
x <- sample(c(0,1/2), 500, TRUE)
tstart <- rep(0, 500)
tstop <- rexp(1 * exp(x))
status <- rep(1, 500)
id <- rep(1:100, each = 5)
dat <- data.frame(id, tstart, tstop, status, x)

# What coxph does:
library(survival)
m_cph <- coxph(Surv(tstart, tstop, status) ~ x + frailty(id), dat, ties = "breslow")
m_cph$history
# For the frailty variance, the program tries: 0, 1, 0.5, 0.005, 0.00005, etc. Stops at 5e-7.

m_ft <- emfrail(dat, Surv(tstart, tstop, status) ~ x + cluster(id))
m_ft

# The algorithm gives as frailty parameter 10587.88,
# which means frailty variance 1/10587.88 = 9.44e-05
# That is because by default, zerotol = 1e-04,
# which is the point where the algorithm decides that the frailty is 0.

# If you want the exact value of the estimate,
# increase the precision so the point stays somewhat interior to the parameter space:
m_ft_08 <- emfrail(dat, Surv(tstart, tstop, status) ~ x + cluster(id),
                   .control = emfrail_control(zerotol = 1e-8))
# This gives a more precise estimate, 5845410 so frailty variance 1/5845410 = 1.71e-07
}
}
\seealso{
\code{\link{emfrail}}, \code{\link{emfrail_distribution}}, \code{\link{emfrail_pll}}
}
