% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{plot.FEM.time}
\alias{plot.FEM.time}
\title{Plot a \code{FEM.time} object at a given time}
\usage{
\method{plot}{FEM.time}(x, time_locations = NULL, locations = NULL,
                lambdaS = NULL, lambdaT = NULL, num_refinements = NULL, Nt = 100,
                add = FALSE, main = NULL, col = "red", ...)
}
\arguments{
\item{x}{A \code{FEM.time} object.}

\item{time_locations}{A vector with the instants in which plot the spatial field}

\item{locations}{A 2-column matrix when \code{x$FEMbasis$mesh} is of class \code{mesh.2D}
or a 3-column matrix otherwise with the spatial locations for which plot the temporal evolution}

\item{lambdaS}{Index of the space penalization parameter to use for the plot, useful when \code{FEM.time} returned by \code{smooth.FEM.time} using GCV}

\item{lambdaT}{Index of the time penalization parameter to use for the plot, useful when \code{FEM.time} returned by \code{smooth.FEM.time} using GCV}

\item{num_refinements}{A natural number specifying how many bisections should be applied to each triangular element for
plotting purposes. This functionality is useful where a discretization with 2nd order Finite Element is applied.
This parameter can be specified only when a FEM object defined over a 2D mesh is plotted.}

\item{Nt}{The number of instants to plot when \code{locations} is not NULL}

\item{add}{Boolean, used only when locations is not NULL, if TURE it performs the graphic over the last plot}

\item{main}{The title of the plot when \code{locations} is not NULL}

\item{col}{The color of the plot when \code{locations} is not NULL. May be a single color or a vector of colors}

\item{...}{Arguments representing graphical options to be passed to \link[rgl]{plot3d}.}
}
\value{
No return value
}
\description{
Plot of a \code{FEM.time} object, generated by \code{FEM.time} or returned by
\code{smooth.FEM.time}. \code{time_locations} and \code{locations} must not be both provided.
If \code{time_locations} is provided, the spatial field is plotted for the provided temporal instnts.
If \code{locations} is provided, the temporal evolution in the provided space locations is plotted.
If both \code{time_locations} and \code{locations} are NULL a default plot is provided.
If the \code{mesh} of the \code{FEMbasis} component is of class \code{mesh.2D} both the 3rd axis and the color represent
the value of the coefficients for the Finite Element basis expansion (\code{coeff} component of the \code{FEM.time} object).
If the \code{mesh} is of class \code{mesh.3D}, the color of each triangle or tetrahedron represent the mean value of
the coefficients for the Finite Element basis expansion (\code{coeff}).
}
\examples{
library(fdaPDE)
## Upload the horseshoe2D data
data(horseshoe2D)
boundary_nodes = horseshoe2D$boundary_nodes
boundary_segments = horseshoe2D$boundary_segments
locations = horseshoe2D$locations

## Create the 2D mesh
mesh = create.mesh.2D(nodes = rbind(boundary_nodes, locations), segments = boundary_segments)
## Create the FEM basis
FEMbasis = create.FEM.basis(mesh)
## Compute the coeff vector evaluating the desired function at the mesh nodes
## In this case we consider the fs.test() function introduced by Wood et al. 2008
time = 1:5
coeff = rep(fs.test(mesh$nodes[,1], mesh$nodes[,2]),5)*time
## Create the FEM.time object
FEM_time_function = FEM.time(coeff=coeff, time_mesh=1:5, FEMbasis=FEMbasis, FLAG_PARABOLIC=TRUE)

## Plot the FEM function
plot(FEM_time_function)

## plot spatial field in some instants
t = c(1.2,1.5,3.6,2.4,4.5)
plot(FEM_time_function, t)

## plot time evolution in some locations
plot(FEM_time_function, locations = locations[1:10,])
}
\seealso{
\code{\link{FEM.time}}, \code{\link{image.FEM.time}}
}
