% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/idf.R
\docType{class}
\name{idf}
\alias{idf}
\title{Read, modify, and run an EnergyPlus model}
\description{
eplusr provides parsing EnergyPlus Input Data File (IDF) files and strings
in a hierarchical structure, which was extremely inspired by \href{https://openstudio-sdk-documentation.s3.amazonaws.com/cpp/OpenStudio-2.5.1-doc/utilities/html/idf_page.html}{OpenStudio utilities library},
but with total different data structure under the hook.
}
\section{Overview}{


eplusr uses \code{Idf} class to present the whole IDF file and use \code{IdfObject}
to present a single object in IDF. Both \code{Idf} and \code{IdfObject} contain member
functions for helping modify the data in IDF so it complies with the
underlying IDD (EnergyPlus Input Data Dictionary).

Under the hook, eplusr uses a SQL-like structure to store both IDF and IDD
data in \code{data.frame} format. To speed up the whole process, the
\code{\link[data.table:data.table]{data.table::data.table()}} is used instead of the base \code{data.frame}. Every
IDF is parsed and stored in four tables:
\itemize{
\item \code{object}: contains object IDs and names.
\item \code{value}: contains field values.
\item \code{comment}: contains object comments.
\item \code{value_reference}: contains cross-reference of field values.
}

IDD file is parsed and stored in a similar structure. For details, please see
\link[=idd]{Idd class}.

So to modify an EnergyPlus model in eplusr is equal to change the data in
those four tables accordingly, in the context of specific IDD data.

All IDF reading process starts with \code{\link[=read_idf]{read_idf()}} which returns an \code{Idf}
object. The model will be printed in a similar style you see in IDFEditor,
with additional heading lines show the \code{Path}, \code{Version} of the model. The
classes of objects in the model are ordered by group and the number of
objects in classes are shown in square bracket.
}

\section{Usage}{


\preformatted{
model$version()
model$path()
model$group_name(all = FALSE)
model$class_name(all = FALSE)
model$is_valid_group(group, all = FALSE)
model$is_valid_class(class, all = FALSE)
model$definition(class)
model$object_id(class = NULL, simplify = FALSE)
model$object_name(class = NULL, simplify = FALSE)
model$object_num(class = NULL)
model$is_valid_id(id)
model$is_valid_name(name)
model$object(id)
model$object_in_class(class)
model$search_object(pattern, class = NULL)
model$ClassName
model[[ClassName]]
model$dup_object(object, new_name = NULL)
model$add_object(class, value = NULL, comment = NULL, default = TRUE, all = FALSE)
model$ins_object(object)
model$set_object(object)
model$del_object(object, referenced = FALSE)
model$search_value(pattern)
model$replace_value(pattern, replacement)
model$validate()
model$is_valid
model$string(comment = TRUE, header = TRUE, ...)
model$is_unsaved()
model$save(path = NULL, format = c("sorted", "new_top", "new_bot"), overwrite = FALSE, copy_external = TRUE)
model$clone()
model$run(weather = NULL, dir = NULL, wait = TRUE, force = FALSE)
model$print(plain = FALSE)
print(model)
}
}

\section{Basic Info}{
\preformatted{model$version()
model$path()
model$group_name(all = FALSE)
model$class_name(all = FALSE)
model$is_valid_group(group, all = FALSE)
model$is_valid_class(class, all = FALSE)
}

\code{$version()} will return the version of current model.

\code{$path()} will return the path of current model or \code{NULL} if the model is
created using a character vector.

\code{$group_name()} will return all groups the model contains when \code{all} is \code{FALSE}
or all groups the Idd contains when \code{all} is \code{TRUE}.

\code{$class_name()} will return all classes the model contains when \code{all} is \code{FALSE}
or all classes the Idd contains when \code{all} is \code{TRUE}.

\code{$is_valid_group()} will return \code{TRUE}s if given group names are valid for
current model (when \code{all} is \code{FALSE}) or current Idd (when \code{all} is
\code{TRUE}).

\code{$is_valid_class()} will return \code{TRUE}s if given class names are valid for
current model (when \code{all} is \code{FALSE}) or current Idd (when \code{all} is
\code{TRUE}).

\strong{Arguments}
\itemize{
\item \code{all}: If \code{FALSE}, only values in current model will be returned. If
\code{TRUE}, all values in Idd will be returned. Default: \code{FALSE}.
\item \code{group}: A character vector contains group names.
\item \code{class}: A character vector contains class names.
}
}

\section{Definition}{
\preformatted{model$definition(class)
}

\code{$definition()} will return the definitions, i.e. the \code{IddObject}s, of given
classes which contain all data used for parsing \code{IdfObject}s. For details
of \code{IdfObject}, please see \link[=idd_object]{IddObject class}.

\strong{Arguments}
\itemize{
\item \code{class}: A character vector contains class names.
}
}

\section{Object Info}{
\preformatted{model$object_id(class = NULL, simplify = FALSE)
model$object_name(class = NULL, simplify = FALSE)
model$object_num(class = NULL)
model$is_valid_id(id)
model$is_valid_name(name)
}

\code{$object_id()} and \code{$object_name()} will return all object IDs and names
in specified class respectively. For \code{$object_name()}, if the specified
class does not have name attributes, such as \code{SimulationContrl}, \code{NA}
will be returned.

\code{$is_valid_id()} and \code{$is_valid_name()} will return \code{TRUE}s if given integers
or strings are valid object IDs or object names respectively.

\code{$object_num()} will return the number of objects in specified classes.

\strong{Arguments}
\itemize{
\item \code{id}: An integer vector to check.
\item \code{name}: A character vector to check.
\item \code{class}: A character vector contains valid class names.
\item \code{simplify}: If FALSE, a list with each member being the data per class will
be returned. Otherwise, an integer vector (for \code{$object_id}) or a
character vector (for \code{$object_name}) will be returned.
}
}

\section{Object Query}{


\preformatted{
model$object(which)
model$object_in_class(class)
model$search_object(pattern, class = NULL)
model$ClassName
model[[ClassName]]
}

\code{$object()} will return a list of \code{IdfObject}s specified by object IDs or
names.

\code{$object_in_class()} will return a list of all \code{IdfObject}s in specified
classes.

\code{$search_object()} will return a list of \code{IdfObject}s whose names meet the
given pattern in specified classes.

eplusr also provides custom S3 method of \code{$} and \code{[[} to make it more
convenient to get \code{IdfObject}s in class. Basically, \code{model$ClassName} and
\code{model[[ClassName]]}, where \code{ClassName} is a single valid class name, is
equivalent to \code{model$object_in_class(ClassName)}.

All above methods will return a named list of \code{IdfObject}s. If the class does
not have name attribute, then \code{NA} will be used.

\code{IdfObject} is a class that provides more detailed information methods to
modify a single object in an \code{Idf} object. For detailed explanations,
please see \link[=idf_object]{IdfObject class}.

\strong{Arguments}
\itemize{
\item \code{object}: Either an integer vector of valid object IDs or a character vector
of valid object names.
\item \code{class}: A character vector of valid class names.
\item \code{pattern}: A regular expression. It will be directly passed to
\code{stringr::str_detect}.
\item \code{ClassName}: A single length character vector of one valid class name,
where all characters other than letters and numbers are replaced by a
underscore \code{_}.
}
}

\section{Object Modification}{
\preformatted{model$dup_object(object, new_name = NULL)
model$add_object(class, value = NULL, comment = NULL, default = TRUE, all = FALSE)
model$ins_object(object)
model$set_object(object)
model$del_object(object, referenced = FALSE)
model$search_value(pattern)
model$replace_value(pattern, replacement)
}

\code{$dup_object()} will duplicate objects specified by object IDs or names. The
newly created objects will be renamed automatically if new names are not
given, with a suffix \code{"_1"}, \code{"_2"} and etc.

\code{$add_object()} will add objects in the specified class.

\code{$ins_object()} will insert objects from other IDF into current IDF.

\code{$set_object()} will set the value of fields in the objects specified by object
IDs or names.

\code{$del_object()} will delete objects specified by object IDs or names.

\code{$search_value()} will return values that match the given pattern.

\code{$replace_value()} will return replace values that match the given pattern.

\strong{NOTE}: There is no field validation when using \code{$replace_value()} to
change field values. \code{$replace_value()} should be treated as a low-level
method which should be used with caution.

\strong{Arguments}
\itemize{
\item \code{object}: Either an integer vector of valid object IDs or a character vector
of valid object names.
\item \code{new_name}: A character vector with the same length as the number of
objects to be duplicated.
\item \code{value}: A list which contains field values to set to the newly created
objects. The class of each field value should comply with the definition
in corresponding IDD. Field names of value in each class can be given. If
not named, the input values will be set to fields according to their
order of appearance.
\item \code{comment}: A list which contains comments to set to the newly created
objects.
\item \code{default}: If \code{TRUE}, all empty fields will be filled with their default
values if possible.
\item \code{all}: If \code{TRUE}, all fields in the class will be returned, even if there
are no input values for them. If \code{FALSE}, only minimum fields will be
returned.
\item \code{referenced}: If \code{TRUE}, all objects that reference the targets to delete
will also be deleted.
\item \code{pattern}: A regular expression used to search for values.
\item \code{replacement}: A regular expression used to replace values.
}
}

\section{Validation}{
\preformatted{model$validate()
model$is_valid()
}

\code{$validate()} will check if there are errors in current model under different
strictness level.

\code{$is_valid()} will check if there are no errors in current model under different
strictness level.

The strictness level can be changed using \code{\link[=eplusr_option]{eplusr_option()}}. Default is
\code{"final".}There are three different validate levels, i.e. \code{"none"},
\code{"draft"} and \code{"final"}:
\itemize{
\item For \code{"none"}, none validation will be done;
\item For \code{"draft"}, checking of invalid autosize, autocalculate, numeric,
integer, and choice field values will be done;
\item For \code{"final"}, besides above, checking of missing required objects,
duplicated unique objects, object name conflicts, missing required fields
and invalid field value reference will also be done.
}
}

\section{Format Output}{
\preformatted{model$string(comment = TRUE, header = TRUE)
}

\code{$string()} will return the text format of an IDF file.

\strong{Arguments}
\itemize{
\item \code{comment}: If \code{FALSE}, all comments will not be included.
\item \code{header}: If \code{FALSE}, the header will not be included.
}
}

\section{Save}{
\preformatted{model$is_unsaved()
model$save(path = NULL, format = c("asis", "sorted", "new_top", "new_bot"), overwrite = FALSE, copy_external = TRUE)
}

\code{$is_unsaved()} will check if there are modifications on the model since it was
read or since last time it was saved.

\code{$save()} will save the model into local disk.

\strong{Arguments}
\itemize{
\item \code{path}: A path where to save the model. If \code{NULL}, the path of the model
itself will be used.
\item \code{format}: A string to specify the saving format. Should be one of \code{"asis"},
\code{"sorted"}, \code{"new_top"}, and \code{"new_bot"}. If \code{"asis"}, which is the default, the
model will be saved in the same format as it is. If the model does not
contain any format saving option, which is typically the case when the
model was not saved using eplusr or IDFEditor, \code{"sorted"} will be used.
\code{"sorted"}, \code{"new_top"} and \code{"new_bot"} are the same as the save options
\code{"Sorted"}, \code{"Original with New at Top"}, and \code{"Original with New at Bottom"}
in IDFEditor.
\item \code{overwrite}: Whether to overwrite the file if it already exists. Default is
\code{FALSE}.
\item \code{copy_external}: If \code{TRUE}, the external files will also be copied into the
same directory. Currently, only \code{Schedule:File} class is supported.
Default is \code{FALSE}.
}
}

\section{Clone}{
\preformatted{model$clone(deep = FALSE)
}

\code{$clone()} will copy and returned the cloned model. Because \code{Idf} use
\code{R6Class} under the hook, \code{idf_2 <- idf_1} does not copy \code{idf_1} at all
but only create a new binding to \code{idf_1}. Modify \code{idf_1} will also affect
\code{idf_2} as well, as these two are exactly the same thing underneath.

\strong{Arguments}
\itemize{
\item \code{deep}: Not used. Keep it here just for compatible with the default clone
method provided by \code{R6Class}.
}
}

\section{Run Model}{
\preformatted{model$run(weather, dir = NULL, wait = TRUE, force = FALSE)
}

\code{$run()} will run the current model within specified weather using
corresponding version of EnergyPlus. The model and the weather used will
be copied to the output directory. An \code{EplusJob} will be returned which
provides detailed info of the simulation and methods to collect
simulation results. Please see \code{\link[=eplus_job]{eplus_job()}} for more detailed.

eplusr uses the EnergyPlus command line interface which was introduced since
EnergyPlus 8.3.0. So \code{$run} only supports models with version higher
than 8.3.0.

eplusr uses the EnergyPlus SQL output for extracting simulation results. In
order to do so, a object in \code{Output:SQLite} with \code{Option Type} value of
\code{SimpleAndTabular} will be automatically created if it does not exists.

\strong{Arguments}
\itemize{
\item \code{weather}: A path to an \code{.epw} file or an \code{Epw} object.
\item \code{dir}: The directory to save the simulation results. If \code{NULL}, which is
the default, the model folder will be used.
\item \code{echo}: Whether to print the standard output and error of EnergyPlus to
the screen. Default is \code{FALSE}.
\item \code{force}: Whether to stop the background EnergyPlus process and start the
simulation again.
}
}

\section{Print}{
\preformatted{model$print(plain = FALSE)
print(model)
}

\code{$print()} will print the model in the similar format as what you will see in
IDFEditor.

\strong{Arguments}
\itemize{
\item \code{plain}: If \code{TRUE}, the model will be printed in plain text format.
}
}

\seealso{
\link[=idf_object]{IdfObject class}
}
\author{
Hongyuan Jia
}
