#' Fit a parametric function that describes dependency of embryo growth to temperature
#'
#' \tabular{ll}{
#'  Package: \tab embryogrowth\cr
#'  Type: \tab Package\cr
#'  Version: \tab 7.4.1 build 739\cr
#'  Date: \tab 2018-09-26\cr
#'  License: \tab GPL (>= 2)\cr
#'  LazyLoad: \tab yes\cr
#'  }
#' @title The package embryogrowth
#' @author Marc Girondot \email{marc.girondot@@u-psud.fr}
#' @docType package
#' @name embryogrowth-package
#' @description Tools to analyze the embryo growth and the sexualisation thermal reaction norms.\cr
#' The lastest version of this package can always been installed using:\cr
#' install.packages("http://www.ese.u-psud.fr/epc/conservation/CRAN/HelpersMG.tar.gz", repos=NULL, type="source")\cr
#' install.packages("http://www.ese.u-psud.fr/epc/conservation/CRAN/embryogrowth.tar.gz", repos=NULL, type="source")
#' @references Girondot, M. & Kaska, Y. 2014. A model to predict the thermal 
#'          reaction norm for the embryo growth rate from field data. Journal of
#'          Thermal Biology. 45, 96-102.
#' @seealso Delmas, V., Prevot-Julliard, A.-C., Pieau, C. & Girondot, M. 2008. 
#'          A mechanistic model of temperature-dependent sex determination 
#'          in a Chelonian, the European pond turtle. Functional 
#'          Ecology, 22, 84-93.
#' @seealso Girondot, M., Ben Hassine, S., Sellos, C., Godfrey, M. & Guillon, 
#'          J.-M. 2010. Modeling thermal influence on animal growth and sex 
#'          determination in Reptiles: being closer of the target gives new 
#'          views. Sexual Development, 4, 29-38.
#' @seealso Girondot, M. 1999. Statistical description of temperature-dependent 
#'          sex determination using maximum likelihood. Evolutionary Ecology 
#'          Research, 1, 479-486.
#' @seealso Girondot, M., & Kaska, Y. (2014). Nest temperatures in a loggerhead-
#'          nesting beach in Turkey is more determined by sea surface temperature 
#'          than air temperature. Journal of Thermal Biology, 47, 13-18.
#' @references Fuentes, M.M.P.B., Monsinjon, J., Lopez, M., Lara, P., Santos, A., 
#'             dei Marcovaldi, M.A.G., Girondot, M., 2017. Sex ratio estimates for 
#'             species with temperature-dependent sex determination differ according 
#'             to the proxy used. Ecological Modelling 365, 55-67.
#' @references Girondot, M., Monsinjon, J., Guillon, J.-M., 2018. Delimitation of 
#'             the embryonic thermosensitive period for sex determination using an 
#'             embryo growth model reveals a potential bias for sex ratio prediction 
#'             in turtles. Journal of Thermal Biology 73, 32-40.
#' @references Monsinjon, J., Jribi, I., Hamza, A., Ouerghi, A., Kaska, Y., 
#'             Girondot, M., 2017. Embryonic growth rate thermal reaction norm of 
#'             Mediterranean Caretta caretta embryos from two different thermal 
#'             habitats, Turkey and Libya. Chelonian Conservation and 
#'             Biology 16, 172-179.
#' @keywords Temperature Embryo Ecology Growth Gompertz Sex-determination
#' @examples
#' \dontrun{
#' library("embryogrowth")
#' packageVersion("embryogrowth")
#' data(nest)
#' formated <- FormatNests(nest)
#' # The initial parameters value can be:
#' # "T12H", "DHA",  "DHH", "Rho25"
#' # Or
#' # "T12L", "DT", "DHA",  "DHH", "DHL", "Rho25"
#' x <- structure(c(115.758929130522, 428.649022170996, 503.687251738993, 
#' 12.2621455821612, 306.308841227278, 116.35048615105), .Names = c("DHA", 
#' "DHH", "DHL", "DT", "T12L", "Rho25"))
#' # or
#' x <- structure(c(118.431040984352, 498.205702157603, 306.056280989839, 
#' 118.189669472381), .Names = c("DHA", "DHH", "T12H", "Rho25"))
#' # pfixed <- c(K=82.33) or rK=82.33/39.33
#' pfixed <- c(rK=2.093313)
#' 
#' ################################################################################
#' #
#' # The values of rK=2.093313 and M0=1.7 were used in 
#' # Girondot, M. & Kaska, Y. 2014. A model to predict the thermal 
#' # reaction norm for the embryo growth rate from field data. Journal of
#' # Thermal Biology. 45, 96-102.
#' #
#' # Based on recent analysis on table of development for both Emys orbicularis and 
#' # Caretta caretta, best value for pfixed should be 1.209 and M0 should be 0.34.
#' # Girondot, M., Monsinjon, J., Guillon, J.-M., Submitted. Delimitation of the 
#' # embryonic thermosensitive period for sex determination using an embryo growth 
#' # model reveals a potential bias for sex ratio prediction in turtles.
#' #
#' # See the example in the stages datasets
#' # 
#' ################################################################################
#' 
#' resultNest_4p_SSM4p <- searchR(parameters=x, fixed.parameters=pfixed, 
#' 	temperatures=formated, derivate=dydt.Gompertz, M0=1.7, 
#' 	test=c(Mean=39.33, SD=1.92))
#' data(resultNest_4p_SSM4p)
#' par(mar=c(4, 4, 1, 1))
#' plot(resultNest_4p_SSM4p$data[[1]][, 1]/60/24,resultNest_4p_SSM4p$data[[1]][, 2], bty="n", las=1, 
#'      xlab="Days of incubation", ylab="Temperatures in °C", 
#'      type="l", xlim=c(0,70),ylim=c(20, 35))
#' for (i in 2:resultNest_4p_SSM4p$data$IndiceT[3]) {
#'   par(new=TRUE)
#'   plot(resultNest_4p_SSM4p$data[[i]][, 1]/60/24,resultNest_4p_SSM4p$data[[i]][, 2], 
#'   bty="n", las=1, xlab="", ylab="", type="l", xlim=c(0,70),ylim=c(20, 35), axes = FALSE)
#' }
#' par(mar=c(4, 4, 1, 1))
#' pMCMC <- TRN_MHmcmc_p(resultNest_4p_SSM4p, accept=TRUE)
#' # Take care, it can be very long, sometimes several days
#' resultNest_mcmc_4p_SSM4p <- GRTRN_MHmcmc(result=resultNest_4p_SSM4p,  
#' 	parametersMCMC=pMCMC, n.iter=10000, n.chains = 1, n.adapt = 0,  
#' 	thin=1, trace=TRUE)
#' data(resultNest_mcmc_4p_SSM4p)
#' out <- as.mcmc(resultNest_mcmc_4p_SSM4p)
#' # This out obtained after as.mcmc can be used with coda package
#' # plot() can use the direct output of GRTRN_MHmcmc() function.
#' plot(resultNest_mcmc_4p_SSM4p, parameters=1, xlim=c(0,550))
#' plot(resultNest_mcmc_4p_SSM4p, parameters=3, xlim=c(290,320))
#' # But rather than to use the SD for each parameter independantly, it is
#' # more logical to estimate the distribution of the curves
#' new_result <- ChangeSSM(resultmcmc = resultNest_mcmc_4p_SSM4p, result = resultNest_4p_SSM4p,
#'                         temperatures = seq(from = 20, to = 35, by = 0.1), 
#'                         initial.parameters = NULL)
#' par(mar=c(4, 4, 1, 5)+0.4)
#' 
#' plotR(result = resultNest_4p_SSM4p, parameters = new_result$par, 
#'            ylabH = "Temperatures\ndensity", ylimH=c(0, 0.3), atH=c(0, 0.1, 0.2), 
#'            ylim=c(0, 3), show.hist=TRUE)
#'       
#' # Beautiful density plots
#' 
#' plotR(result = resultNest_4p_SSM4p, 
#'              resultmcmc=resultNest_mcmc_4p_SSM4p, 
#'              curves = "MCMC quantiles", show.density=TRUE)
#' 
#' plotR(resultNest_6p_SSM6p, resultmcmc=resultNest_mcmc_6p_SSM6p, 
#'             ylim=c(0, 4), show.density=TRUE, show.hist=TRUE, 
#'             curves = "MCMC quantiles", 
#'             ylimH=c(0,0.5), atH=c(0, 0.1, 0.2))
#' }

NULL
