% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsd.R
\name{tsd}
\alias{tsd}
\title{Estimate the parameters that best describe temperature-dependent sex determination}
\usage{
tsd(df = NULL, males = NULL, females = NULL, N = NULL,
  temperatures = NULL, durations = NULL, l = 0.05,
  parameters.initial = c(P = NA, S = -0.5, K = 0, K1 = 1, K2 = 0),
  males.freq = TRUE, fixed.parameters = NULL, SE = NULL,
  equation = "logistic", replicates = 1000, range.CI = 0.95,
  limit.low.TRT.minimum = 5, limit.high.TRT.maximum = 90, print = TRUE,
  temperatures.plot = seq(from = 20, to = 40, by = 0.1),
  durations.plot = seq(from = 15, to = 100, by = 0.1))
}
\arguments{
\item{df}{A dataframe with at least two columns named males, females or N and temperatures, Incubation.temperature or durations column}

\item{males}{A vector with male numbers}

\item{females}{A vector with female numbers}

\item{N}{A vector with total numbers}

\item{temperatures}{The constant incubation temperatures used to fit sex ratio}

\item{durations}{The duration of incubation or TSP used to fit sex ratio}

\item{l}{The limit to define TRT (see Girondot, 1999)}

\item{parameters.initial}{Initial values for P, S or K search as a vector, ex. c(P=29, S=-0.3)}

\item{males.freq}{If TRUE data are shown as males frequency}

\item{fixed.parameters}{Parameters that will not be changed}

\item{SE}{Standard errors for parameters}

\item{equation}{Could be "logistic", "Hill", "Richards", "Hulin", "Double-Richards" or "GSD"}

\item{replicates}{Number of replicates to estimate confidence intervals}

\item{range.CI}{The range of confidence interval for estimation, default=0.95}

\item{limit.low.TRT.minimum}{Minimum lower limit for TRT}

\item{limit.high.TRT.maximum}{Maximum higher limit for TRT}

\item{print}{Do the results must be printed at screen? TRUE (default) or FALSE}

\item{temperatures.plot}{Sequences of temperatures that will be used for plotting. If NULL, does not estimate them}

\item{durations.plot}{Sequences of durations that will be used for plotting. If NULL, does not estimate them}
}
\value{
A list the pivotal temperature, transitional range of temperatures and their SE
}
\description{
Estimate the parameters that best describe temperature-dependent sex determination
}
\details{
tsd estimates the parameters that best describe temperature-dependent sex determination
}
\examples{
\dontrun{
CC_AtlanticSW <- subset(STSRE_TSD, RMU=="Atlantic, SW" & 
                          Species=="Caretta caretta" & Sexed!=0)
tsdL <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature-Correction.factor, 
                                 equation="logistic"))
tsdH <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature-Correction.factor, 
                                 equation="Hill"))
tsdR <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature-Correction.factor, 
                                 equation="Richards"))
tsdDR <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature-Correction.factor, 
                                 equation="Double-Richards"))
gsd <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature-Correction.factor, 
                                 equation="GSD"))
compare_AIC(Logistic_Model=tsdL, Hill_model=tsdH, Richards_model=tsdR, 
               DoubleRichards_model=tsdDR, GSD_model=gsd)
##############
eo <- subset(STSRE_TSD, Species=="Emys orbicularis", c("Males", "Females", 
                                       "Incubation.temperature"))
                                       
eo_Hill <- with(eo, tsd(males=Males, females=Females, 
                                       temperatures=Incubation.temperature,
                                       equation="Hill"))
eo_Hill <- tsd(df=eo, equation="Hill")
eo_logistic <- tsd(eo)
eo_Richards <- with(eo, tsd(males=Males, females=Females, 
                                 temperatures=Incubation.temperature, 
                                 equation="Richards"))
### The Hulin model is a modification of Richards (See Hulin et al. 2009)
### limit.low.TRT and limit.high.TRT must be setup for Hulin equation
par <- eo_Richards$par
names(par)[which(names(par)=="K")] <- "K2"
par <- c(par, K1=0)
eo_Hulin <- with(eo, tsd(males=Males, females=Females, 
                                 parameters.initial=par, 
                                 temperatures=Incubation.temperature, 
                                 equation="Hulin", 
                                 limit.low.TRT.minimum=25, 
                                 limit.high.TRT.maximum=35))
### The Double-Richards model is a Richards model with K1 and K2 using the two values
### below and above P
par <- eo_Richards$par
names(par)[which(names(par)=="K")] <- "K2"
par <- c(par, K1=as.numeric(par["K2"])-0.1)
par["K1"] <- par["K1"]-0.1
eo_Double_Richards <- with(eo, tsd(males=Males, females=Females,
                                 parameters.initial=par,
                                 temperatures=Incubation.temperature,
                                 equation="Double-Richards"))
compare_AIC(Logistic=eo_logistic, Hill=eo_Hill, Richards=eo_Richards, 
             Hulin=eo_Hulin, Double_Richards=eo_Double_Richards)
### Note the asymmetry of the Double-Richards model
predict(eo_Double_Richards, 
       temperatures=c(eo_Double_Richards$par["P"]-0.2, eo_Double_Richards$par["P"]+0.2))
predict(eo_Double_Richards)
### It can be used also for incubation duration
CC_AtlanticSW <- subset(STSRE_TSD, RMU=="Atlantic, SW" & 
                          Species=="Caretta caretta" & Sexed!=0)
tsdL_IP <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
                                 durations=IP.mean, 
                                 equation="logistic"))
plot(tsdL_IP, xlab="Incubation durations in days")
}
}
\author{
Marc Girondot
}
\references{
Girondot, M. 1999. Statistical description of temperature-dependent sex determination using maximum likelihood. Evolutionary Ecology Research, 1, 479-486.

Godfrey, M.H., Delmas, V., Girondot, M., 2003. Assessment of patterns of temperature-dependent sex determination using maximum likelihood model selection. Ecoscience 10, 265-272.

Hulin, V., Delmas, V., Girondot, M., Godfrey, M.H., Guillon, J.-M., 2009. Temperature-dependent sex determination and global change: are some species at greater risk? Oecologia 160, 493-506.
}
\seealso{
Other Functions.for.temperature.dependent.sex.determination: \code{\link{STSRE_TSD}},
  \code{\link{TSP.list}}, \code{\link{predict.tsd}},
  \code{\link{stages}}, \code{\link{tsd_MHmcmc_p}},
  \code{\link{tsd_MHmcmc}}
}

