% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ecm.R
\name{ecm}
\alias{ecm}
\title{Build an error correction model}
\usage{
ecm(y, xeq, xtr, lags = 1, includeIntercept = TRUE, weights = NULL, ...)
}
\arguments{
\item{y}{The target variable}

\item{xeq}{The variables to be used in the equilibrium term of the error correction model}

\item{xtr}{The variables to be used in the transient term of the error correction model}

\item{lags}{The number of lags to use}

\item{includeIntercept}{Boolean whether the y-intercept should be included}

\item{weights}{Optional vector of weights to be passed to the fitting process}

\item{...}{Additional arguments to be passed to the 'lm' function (careful in that these may need to be modified for ecm or may not be appropriate!)}
}
\value{
an lm object representing an error correction model
}
\description{
Builds an lm object that represents an error correction model (ECM) by automatically differencing and
lagging predictor variables according to ECM methodology.
}
\details{
The general format of an ECM is \deqn{\Delta y = \beta_{0} + \beta_{1}\Delta x_{1,t} +...+ \beta_{i}\Delta x_{i,t} + \gamma(y_{t-1} - (\alpha_{1}x_{1,t-1} +...+ \alpha_{i}x_{i,t-1})).}
The ecm function here modifies the equation to the following: \deqn{\Delta y = \beta_{0} + \beta_{1}\Delta x_{1,t} +...+ \beta_{i}\Delta x_{i,t} + \gamma y_{t-1} + \gamma_{1}x_{1,t-1} +...+ \gamma_{i}x_{i,t-1},}
\deqn{where \gamma_{i} = -\gamma \alpha_{i},} 
so it can be modeled as a simpler ordinary least squares (OLS) function using R's lm function.

Ordinarily, the ECM uses lag=1 when differencing the transient term and lagging the equilibrium term, as specified in the equation above. However, the ecm function here gives the user the ability to specify a lag greater than 1. 

Notice that an ECM models the change in the target variable (y). This means that the predictors will be lagged and differenced,
and the model will be built on one observation less than what the user inputs for y, xeq, and xtr. If these arguments contain vectors with too few observations (eg. one single observation),
the function will not work. Additionally, for the same reason, if using weights in the ecm function, the length of weights should be one less than the number of rows in xeq or xtr.

When inputting a single variable for xeq or xtr in base R, it is important to input it in the format "xeq=df['col1']" so they inherit the class 'data.frame'. Inputting such as "xeq=df[,'col1']" or "xeq=df$col1" will result in errors in the ecm function. You can load data via other R packages that store data in other formats, as long as those formats also inherit the 'data.frame' class.

ECM models are used for time series data. This means the user may need to consider stationarity and/or cointegration before using the model.
}
\examples{
##Not run

#Use ecm to predict Wilshire 5000 index based on corporate profits, 
#Federal Reserve funds rate, and unemployment rate
data(Wilshire)

#Use 2015-12-01 and earlier data to build models
trn <- Wilshire[Wilshire$date<='2015-12-01',]

#Assume all predictors are needed in the equilibrium and transient terms of ecm
xeq <- xtr <- trn[c('CorpProfits', 'FedFundsRate', 'UnempRate')]
model1 <- ecm(trn$Wilshire5000, xeq, xtr, includeIntercept=TRUE)

#Assume CorpProfits and FedFundsRate are in the equilibrium term, 
#UnempRate has only transient impacts
xeq <- trn[c('CorpProfits', 'FedFundsRate')]
xtr <- trn['UnempRate']
model2 <- ecm(trn$Wilshire5000, xeq, xtr, includeIntercept=TRUE)

}
\seealso{
\code{lm}
}
