% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihood_MTLNR.R
\name{LogLikMTLNR}
\alias{LogLikMTLNR}
\alias{LLMTLNR}
\title{Log-Likelihood functions for the mutliple-treshold log-normal race model}
\usage{
LogLikMTLNR(data, paramDf, precision = 6, data_names = list(), ...)
}
\arguments{
\item{data}{a dataframe where each row is one trial. Containing following variables:
\itemize{
\item condition    (not necessary; convertible to integer (e.g. factor); for different levels of stimulus quality),
\item rating            (convertible to integer (e.g. factor); discrete confidence judgments),
\item rt                    (numeric; giving reaction times for decision task),
\item stimulus     (values at least convertible to c(1,2), i.e. integer or factor; stimulus category (index of accumulator with higher drift))
\item response     (values at least convertible to c(1,2); direction of decision; (index of accumulator reaching the boundary first))

}}

\item{paramDf}{a list or data frame with one row. Column names should match the names of
\link{dMTLNR} parameter names (only \code{mu_v1} and \code{mu_v2} are not used in this context but
replaced by the parameter \code{v}). For different stimulus quality/mean
drift rates, names should be \code{v1}, \code{v2}, \code{v3},....
Additionally, the confidence
thresholds should be given by names with \code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},.... (see Details for the correspondence to the data)}

\item{precision}{numerical scalar. Precision of calculation for integration over t0.}

\item{data_names}{list. Possibility of giving alternative column names for the variables in the data. By default column names are identical to the
ones given in the data argument description.}

\item{...}{Another possibility of giving alternative variable names in data frame (in the form \code{condition = "SOA"}).}
}
\value{
Numeric scalar. The summed Log-likelihood of the data given the parameters in the respective model. If one or more row-wise probabilities is <=0,
the function returns -1e+12.
}
\description{
Computes the Log-likelihood for given data and parameters in the MTLNR.
It is a wrapped version of the respective density \code{\link{dMTLNR}},
where one can find more information about the parameters. .
The function is mainly used inside \code{\link{fitRTConf}} for the MTLNR but exported
for individual usage in other contexts.
}
\details{
Note, that the requirements on the format of the columns for the likelihood functions are much stricter, than in \code{\link{fitRTConf}}.
This is because the function is very frequently called in the optimization routines of the fitting process and the preprocessing steps are
therefore included in the other function.

\strong{rating, condition}. If integer, values should range from 1
to number of possible ratings/conditions. If factor,
the number of levels should be equal to number of possible
ratings/conditions. This should be consistent with the
parameter vector. The confidence thresholds should be named as
\code{thetaUpper1}, \code{thetaLower1},.... (or \code{theta1},... for symmetric
thresholds), with the number of ratings -1 and the mean drift rates
(and possibly the standard deviation in drift rates)
should be denoted as \code{v1}, \code{v2},...
If only one condition is used \code{v} will be accepted as well as \code{v1}.

\strong{stimulus, response}. stimulus and response should always
be given in numerical format with values 1 and 2.
Stimulus determines which of two accumulators has positive drift.
The other has negative drift with the same absolute
value. Response gives the index of the accumulator that reaches the
boundary first.
}
\examples{
# 1. Generate data from an artificial participants
# Get random index for accumulator with positive
# drift (i.e. stimulus category) and
# stimulus discriminability (two steps: hard, easy)
stimulus <- sample(c(1, 2), 200, replace=TRUE)
discriminability <- sample(c(1, 2), 200, replace=TRUE)
# generate data for participant 1
data <- rMTLNR(200,
               mu_v1 = as.numeric(stimulus==1)*discriminability*0.5,
               mu_v2 = as.numeric(stimulus==2)*discriminability*0.5,
               mu_d1=1, mu_d2=1, t0=0.1)
# discretize confidence ratings (only 2 steps: unsure vs. sure)
data$rating <- as.numeric(cut(data$conf, breaks = c(0, 3, Inf), include.lowest = TRUE))
data$stimulus <- stimulus
data$discriminability <- discriminability
data <- data[,-c(3,4)] # drop Tdec and conf measure (unobservable variable)
head(data)

# 2. Define some parameter set in a data.frame
paramDf <- data.frame(v1=0.5, v2=1.0, t0=0.1, st0=0,
                      mu_d1=1, mu_d2=1,
                      s_v1=0.5, s_v2=0.5,
                      s_d1=0.3, s_d2=0.3,
                      rho_v=0.2, rho_d=0.1,
                      theta1=2.5)

# 3. Compute log likelihood for parameter and data
LogLikMTLNR(data, paramDf, condition="discriminability")

}
\references{
Reynolds, A., Kvam, P. D., Osth, A. F., & Heathcote, A. (2020). Correlated racing evidence accumulator models. \emph{Journal of Mathematical Psychology, 96}, 102331. doi: doi: 10.1016/j.jmp.2020.102331
}
\author{
Sebastian Hellmann.
}
