% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictratingdist_WEV.R
\name{predictWEV}
\alias{predictWEV}
\alias{predictWEV_Conf}
\alias{predictdynWEV}
\alias{predict2DSD}
\alias{predictWEV_RT}
\title{Prediction of Confidence Rating and Response Time Distribution in dynWEV
and 2DSD confidence models}
\usage{
predictWEV_Conf(paramDf, model = "dynWEV", maxrt = 15,
  subdivisions = 100L, simult_conf = FALSE, stop.on.error = FALSE,
  precision = 1e-05, .progress = TRUE)

predictWEV_RT(paramDf, model = NULL, maxrt = 9, subdivisions = 100L,
  minrt = NULL, simult_conf = FALSE, scaled = FALSE, DistConf = NULL,
  precision = 1e-05, .progress = TRUE)
}
\arguments{
\item{paramDf}{a list or dataframe with one row. Column names should match the names
of \link{dynWEV} and \link{2DSD} model specific parameter names.
For different stimulus quality/mean drift rates, names should be \code{v1}, \code{v2}, \code{v3},....
Different \code{sv} and/or \code{s} parameters are possible with \code{sv1}, \code{sv2}, \code{sv3}... (\code{s1}, \code{s2}, \code{s3},...
respectively) with equally many steps as for drift rates. Additionally, the confidence
thresholds should be given by names with \code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},....}

\item{model}{character scalar. One of "dynWEV" or "2DSD".}

\item{maxrt}{numeric. The maximum RT for the integration/density computation.
Default: 15 (for \code{predictWEV_Conf} (integration)), 9 (for \code{predictWEV_RT}).}

\item{subdivisions}{integer (default: 100).
For \code{predictWEV_Conf} it is used as argument for the inner integral routine.
For \code{predictWEV_RT} it is the number of points for which the density is computed.}

\item{simult_conf}{logical. Whether in the experiment confidence was reported simultaneously
with the decision, as then decision and confidence judgment are assumed to have happened
subsequent before response and computations are different, when there is an observable
interjudgment time (then \code{simult_conf} should be FALSE).}

\item{stop.on.error}{logical. Argument directly passed on to integrate. Default is FALSE,
since the densities invoked may lead to slow convergence of the integrals (which are still
quite accurate) which causes R to throw an error.}

\item{precision}{numerical scalar value. Precision of calculation. Corresponds to the
step size of integration w.r.t. \code{z} and \code{t0}. Default is 1e-5.}

\item{.progress}{logical. if TRUE (default) a progress bar is drawn to the console.}

\item{minrt}{numeric or NULL(default). The minimum rt for the density computation.}

\item{scaled}{logical. For \code{predictWEV_RT}. Whether the computed density
should be scaled to integrate to one (additional column \code{densscaled}). Otherwise the output
contains only the defective density (i.e. its integral is equal to the probability of a
response and not 1). If \code{TRUE}, the argument \code{DistConf} should be given, if available.
Default: \code{FALSE}.}

\item{DistConf}{\code{NULL} or \code{data.frame}. A \code{data.frame} or \code{matrix} with column
names, giving the distribution of response and rating choices for
different conditions and stimulus categories in the form of the output of
\code{predictWEV_Conf}. It is only necessary, if \code{scaled=TRUE}, because these
probabilities are used for scaling. If \code{scaled=TRUE} and \code{DistConf=NULL}, it will be computed
with the function \code{predictWEV_Conf}, which takes some time and the function will
throw a message. Default: \code{NULL}}
}
\value{
\code{predictWEV_Conf} returns a \code{data.frame}/\code{tibble} with columns: \code{condition}, \code{stimulus},
\code{response}, \code{rating}, \code{correct}, \code{p}, \code{info}, \code{err}. \code{p} is the predicted probability of a response
and \code{rating}, given the stimulus category and condition. \code{info} and \code{err} refer to the
respective outputs of the integration routine used for the computation.
\code{predictWEV_RT} returns a \code{data.frame}/\code{tibble} with columns: \code{condition}, \code{stimulus},
\code{response}, \code{rating}, \code{correct}, \code{rt} and \code{dens} (and \code{densscaled}, if \code{scaled=TRUE}).
}
\description{
\code{predictWEV_Conf} predicts the categorical response distribution of
decision and confidence ratings, \code{predictWEV_RT} computes the predicted
RT distribution (density) in the 2DSD Model (Pleskac & Busemeyer, 2010) and the
dynWEV model (Hellmann et al., in press), given specific parameter constellations.
See \code{\link{dWEV}} and \code{\link{d2DSD}} for more information about parameters.
}
\details{
The function \code{predictWEV_Conf} consists merely of an integration of
the response time density, \code{\link{dWEV}} and \code{\link{d2DSD}}, over the response time in a reasonable
interval (\code{t0} to \code{maxrt}). The function \code{predictWEV_RT} wraps these density
functions to a parameter set input and a data.frame output.
For the argument \code{paramDf}, the output of the fitting function \code{\link{fitRTConf}}
with the respective model may be used.
}
\note{
Different parameters for different conditions are only allowed for drift rate
\code{v}, drift rate variability \code{sv}, and process variability \code{s}. Otherwise, \code{s} is
not required in \code{paramDf} but set to 1 by default. All other parameters are used for all
conditions.
}
\examples{
# Examples for "dynWEV" model (equivalent applicable for "2DSD" model (with less parameters))
# 1. Define some parameter set in a data.frame
paramDf <- data.frame(a=2.5,v1=0.5, v2=1, t0=0.1,z=0.55,
                      sz=0,sv=0.2, st0=0,  tau=3, w=0.3,
                      theta1=0.8, svis=0.5, sigvis=0.8)

# 2. Predict discrete Choice x Confidence distribution:
preds_Conf <- predictWEV_Conf(paramDf, "dynWEV", maxrt = 15)
head(preds_Conf)
\donttest{
  # To set simult_conf=TRUE makes a minor difference in the discrete distribution,
  # because we integrate over response times (we just adapt maxrt for comparison)
  preds_Conf2 <- predictWEV_Conf(paramDf, "dynWEV", simult_conf = TRUE, maxrt = 15+paramDf$tau)
  summary(preds_Conf$p-preds_Conf2$p) # difference in predicted probabilities
}

# 3. Compute RT density
preds_RT <- predictWEV_RT(paramDf, "dynWEV", maxrt=4, subdivisions=200) #(scaled=FALSE)
# same output with scaled density column:
preds_RT <- predictWEV_RT(paramDf, "dynWEV", maxrt=4, subdivisions=200,
                         scaled=TRUE, DistConf = preds_Conf)
head(preds_RT)
\donttest{
  # produces a warning, if scaled=TRUE and DistConf missing
  preds_RT <- predictWEV_RT(paramDf, "dynWEV", maxrt=4, subdivisions=200,
                           scaled=TRUE)
}

\donttest{
  # Example of visualization
  library(ggplot2)
  preds_Conf$rating <- factor(preds_Conf$rating, labels=c("unsure", "sure"))
  preds_RT$rating <- factor(preds_RT$rating, labels=c("unsure", "sure"))
  ggplot(preds_Conf, aes(x=interaction(rating, response), y=p))+
    geom_bar(stat="identity")+
    facet_grid(cols=vars(stimulus), rows=vars(condition), labeller = "label_both")
  ggplot(preds_RT, aes(x=rt, color=interaction(rating, response), y=dens))+
    geom_line(stat="identity")+
    facet_grid(cols=vars(stimulus), rows=vars(condition), labeller = "label_both")+
    theme(legend.position = "bottom")
  ggplot(aggregate(densscaled~rt+correct+rating+condition, preds_RT, mean),
         aes(x=rt, color=rating, y=densscaled))+
    geom_line(stat="identity")+
    facet_grid(cols=vars(condition), rows=vars(correct), labeller = "label_both")+
    theme(legend.position = "bottom")
}
# Use PDFtoQuantiles to get predicted RT quantiles
head(PDFtoQuantiles(preds_RT, scaled = FALSE))

}
\references{
Hellmann, S., Zehetleitner, M., & Rausch, M. (in press). Simultaneous modeling of choice, confidence and response time in visual perception. \emph{Psychological Review}. \url{https://osf.io/9jfqr/}

Pleskac, T. J., & Busemeyer, J. R. (2010). Two-Stage Dynamic Signal Detection:
A Theory of Choice, Decision Time, and Confidence, \emph{Psychological Review}, 117(3),
864-901. doi:10.1037/a0019737
}
\author{
Sebastian Hellmann.
}
