#' Create formal \code{dtwclust} objects
#'
#' Helper function to manually create formal \code{\link{dtwclust-class}} objects
#'
#' @export
#'
#' @param ... Any valid slots of \code{\link{dtwclust-class}}.
#' @param override.family Attempt to substitute the default family with one that conforms to the
#'   provided elements? See details.
#'
#' @details
#'
#' This function can calculate some of the slots if certain elements are provided by the user. In
#' order to get a fully functional object at least the following slots should be provided:
#'
#' \itemize{
#'   \item \code{type}: "partitional", "hierarchical", "fuzzy" or "tadpole".
#'   \item \code{datalist}: The data in one of the supported formats.
#'   \item \code{centroids}: The time series centroids in one of the supported formats.
#'   \item \code{cluster}: The cluster indices for each series in the \code{datalist}.
#'   \item \code{control*}: A \code{dtwclustControl} object (or a named list) with the desired
#'     parameters.
#'   \item \code{distance*}: A string indicating the distance that should be used.
#'   \item \code{centroid*}: A string indicating the centroid to use (only necessary for partitional
#'     clustering).
#' }
#'
#' *Necessary when overriding the default family for the calculation of other slots, CVIs or
#' prediction. Maybe not always needed, e.g. for plotting.
#'
#' @return A \code{\link{dtwclust-class}} object.
#'
#' @examples
#'
#' data(uciCT)
#'
#' # Assuming this was generated by some clustering procedure
#' centroids <- CharTraj[seq(1L, 100L, 5L)]
#' control <- new("dtwclustControl", window.size = 8L, norm = "L2")
#' cluster <- unclass(CharTrajLabels)
#'
#' pc_obj <- create_dtwclust(type = "partitional", datalist = CharTraj, centroids = centroids,
#'                           control = control, cluster = cluster,
#'                           distance = "sbd", centroid = "dba",
#'                           dots = list(step.pattern = symmetric1))
#'
#' fc_obj <- create_dtwclust(type = "fuzzy", datalist = CharTraj, centroids = centroids,
#'                           control = control, cluster = cluster,
#'                           distance = "sbd", centroid = "fcm")
#'
#' fc_obj
#'
create_dtwclust <- function(..., override.family = TRUE) {
    tic <- proc.time()

    dots <- list(...)

    ## even if it's NULL, it'll be converted correctly
    dots$control <- as(dots$control, "dtwclustControl")

    ## some minor checks
    if (!is.null(dots$datalist)) dots$datalist <- any2list(dots$datalist)
    if (!is.null(dots$centroids)) dots$centroids <- any2list(dots$centroids)

    ## avoid infinite recursion
    if (is.null(dots$call)) {
        call <- match.call()

    } else {
        call <- dots$call
        dots$call <- NULL
    }

    .Object <- do.call(methods::new, enlist(Class = "dtwclust", dots = dots))
    .Object@call <- call

    ## some "defaults"
    if (is.null(dots$preproc)) .Object@preproc <- "none"
    if (is.null(dots$iter)) .Object@iter <- 1L
    if (is.null(dots$converged)) .Object@converged <- TRUE
    if (is.null(dots$k)) .Object@k <- length(.Object@centroids)

    ## more helpful for hierarchical/tadpole
    if (override.family) {
        if (length(.Object@type) == 0L)
            warning("Could not override family, 'type' slot is missing.")
        else if (length(.Object@distance) == 0L)
            warning("Could not override family, 'distance' slot is missing.")
        else {
            centroids <- .Object@centroids
            datalist <- .Object@datalist

            if (.Object@type == "partitional" && length(.Object@centroid))
                allcent <- all_cent(.Object@centroid,
                                    distmat = .Object@distmat,
                                    control = .Object@control,
                                    fuzzy = isTRUE(.Object@type == "fuzzy"))
            else if (.Object@type == "hierarchical" && length(formals(.Object@family@allcent)))
                allcent <- .Object@family@allcent
            else if (.Object@type == "hierarchical" && length(centroids))
                allcent <- function(dummy) {
                    datalist[which.min(apply(.Object@distmat, 1L, sum))] # for CVI's global_cent
                }
            else if (.Object@type == "tadpole" && length(centroids))
                allcent <- function(dummy) { centroids[1L] } # for CVI's global_cent
            else if (.Object@type == "fuzzy")
                allcent <- .Object@centroid
            else
                allcent <- .Object@family@allcent

            .Object@family <- new("dtwclustFamily",
                                  dist = .Object@distance,
                                  allcent = allcent,
                                  preproc = .Object@family@preproc,
                                  distmat = NULL,
                                  control = .Object@control,
                                  fuzzy = isTRUE(.Object@type == "fuzzy"))

            assign("distfun", .Object@family@dist, environment(.Object@family@allcent))

            if (.Object@type == "partitional" && .Object@centroid == "shape") {
                .Object@family@preproc <- zscore
                .Object@preproc <- "zscore"
            }
        }
    }

    if (!nrow(.Object@cldist) && length(formals(.Object@family@dist)) && length(.Object@cluster)) {
        ## no cldist available, but dist and cluster can be used to calculate it
        dm <- do.call(.Object@family@dist,
                      enlist(.Object@datalist,
                             .Object@centroids,
                             dots = .Object@dots))

        .Object@cldist <- as.matrix(dm[cbind(1L:length(.Object@datalist), .Object@cluster)])

        dimnames(.Object@cldist) <- NULL
    }

    if (!nrow(.Object@clusinfo) && length(.Object@cluster) && nrow(.Object@cldist)) {
        ## no clusinfo available, but cluster and cldist can be used to calculate it
        size <- as.vector(table(.Object@cluster))
        clusinfo <- data.frame(size = size, av_dist = 0)
        clusinfo[clusinfo$size > 0L, "av_dist"] <-
            as.vector(tapply(.Object@cldist[ , 1L], .Object@cluster, mean))

        .Object@clusinfo <- clusinfo
    }

    if (.Object@type == "fuzzy" && !nrow(.Object@fcluster) && length(formals(.Object@family@dist))) {
        ## no fcluster available, but dist and cluster function can be used to calculate it
        dm <- do.call(.Object@family@dist,
                      enlist(.Object@datalist,
                             .Object@centroids,
                             dots = .Object@dots))

        .Object@fcluster <- .Object@family@cluster(dm, m = .Object@control@fuzziness)
        colnames(.Object@fcluster) <- paste0("cluster_", 1:.Object@k)
    }

    ## default for when it doesn't apply
    if (.Object@type != "fuzzy") .Object@fcluster <- matrix(NA_real_)

    ## just a filler
    if (!length(.Object@proctime)) .Object@proctime <- proc.time() - tic

    ## return
    .Object
}
