#' Time series clustering under DTW
#'
#' This function uses the DTW distance and related lower bounds to cluster time series. For now, all series
#' must be univariate.
#'
#' Partitional algorithms are implemented via \code{\link[flexclust]{kcca}}. Hierarchical algorithms use the
#' \code{\link[stats]{hclust}} function. The \code{tadpole} algorithm uses the \code{\link{TADPole}} function.
#'
#' The \code{data} may be a matrix or a list, but the matrix will be coerced to a list. A matrix input requires
#' that all time series have equal lengths. If the lengths vary slightly between time series, reinterpolating them to
#' a common length is most likely an acceptable approach (Ratanamahatana and Keogh, 2004). If this is not the
#' case, then clustering them directly is probably ill-advised. See the examples.
#'
#' @section Distance:
#'
#' If a custom distance function is provided, it will receive the data as the first argument.
#'
#' For partitional
#' algorithms, the second argument will be the cluster centers (i.e. other time series). If \code{data} is a
#' matrix, it will be coerced to a list of series, and the centers will also be provided in the form of a list.
#'
#' If hierarchical algorithms are used, the function will also receive the elements of \code{...}.
#'
#' The function should return a distance matrix, ideally of class \code{crossdist}. The time series in the
#' data should be along the rows, and the cluster centers along the columns of the distance matrix.
#'
#' The other option is to provide a string. The string can represent a compatible registered distance of
#' \code{proxy}'s \code{\link[proxy]{dist}}. Extra parameters can be provided in \code{...}. See the examples
#' but please finish reading this section before that.
#'
#' Additionally, with either type of algorithm, it can be a string of one of the following custom
#' implementations (all registered with \code{proxy}):
#'
#' \itemize{
#'   \item \code{"dtw"}: DTW with L1 norm and optionally a Sakoe-Chiba/Slanted-band constraint.
#'   \item \code{"dtw2"}: DTW with L2 norm and optionally a Sakoe-Chiba/Slanted-band constraint.
#'   \item \code{"dtw_lb"}: DTW with L1 or L2 norm and optionally a Sakoe-Chiba constraint. Some computations
#'   are avoided by first estimating the distance matrix with Lemire's lower bound and then iteratively
#'   refining with DTW. See \code{\link{dtw_lb}}.
#'   \item \code{"lbk"}: Keogh's lower bound with either L1 or L2 norm for the Sakoe-Chiba constraint.
#'   \item \code{"lbi"}: Lemire's lower bound with either L1 or L2 norm for the Sakoe-Chiba constraint.
#'   \item \code{"sbd"}: Shape-based distance. Each series is z-normalized in this case. As a result,
#'   the cluster centers (for partitional methods) are also z-normalized. See \code{\link{SBD}} for more
#'   details.
#' }
#'
#' Note that only \code{dtw}, \code{dtw2} and \code{sbd} support series of different lengths.
#'
#' If the user wants to create its own distance and register it with \code{proxy}, it should include a \code{...}
#' in its definition so that it is correctly called. Such functions will always receive the following parameters:
#'
#' \itemize{
#'   \item \code{window.type}: Either \code{"none"} for a \code{NULL} \code{window.size}, or \code{"slantedband"}
#'   otherwise
#'   \item \code{window.size}: The provided window size
#'   \item \code{norm}: The provided desired norm
#'   \item \code{...}: Any additional parameters provided in the original call's ellipsis
#' }
#'
#' Whether the function makes use of them or not, is up to the user.
#'
#' @section Centroid:
#'
#' In the case of partitional algorithms, a suitable function should calculate the cluster centers. In this
#' case, the centers are themselves time series.
#'
#' If a custom function is provided, it will receive different inputs depending on the format of \code{data}:
#'
#' \itemize{
#'   \item For matrix input, it will receive a matrix as single input. Each row will be a series that belongs
#'   to a given cluster. The function should return a numeric vector with the centroid time series.
#'   \item For a list input, the function will receive three inputs in the following order: the \emph{whole}
#'   data list; a numeric vector with length equal to the number of series in \code{data}, indicating which
#'   cluster a series belongs to; the current number of total clusters.
#' }
#'
#' The other option is to provide a character string for the custom implementations. The following options
#' are available:
#'
#' \itemize{
#'   \item "mean": The average along each dimension. In other words, the average of all \eqn{x^j_i}
#'   among the \eqn{j} series that belong to the same cluster for all time points \eqn{t_i}.
#'   \item "median": The median along each dimension. Similar to mean.
#'   \item "shape": Shape averaging. See \code{\link{shape_extraction}} for more details.
#'   \item "dba": DTW Barycenter Averaging. See \code{\link{DBA}} for more details.
#'   \item: "pam": Partition around medoids. This basically means that the cluster centers are always
#'   one of the time series in the data. In this case, the distance matrix can be pre-computed once using all
#'   time series in the data and then re-used at each iteration. It usually saves overhead overall.
#' }
#'
#' Note that only \code{dba} and \code{pam} support series of different lengths.
#'
#' @section Sakoe-Chiba Constraint:
#'
#' A global constraint to speed up the DTW calculation is the Sakoe-Chiba band (Sakoe and Chiba, 1978). To
#' use it, a window width must be defined via \code{window.size}.
#'
#' The windowing constraint uses a centered window. The calculations expect a value in \code{window.size}
#' that represents the distance between the point considered and one of the edges of the window. Therefore,
#' if, for example, \code{window.size = 10}, the warping for an observation \eqn{x_i} considers the points
#' between \eqn{x_{i-10}} and \eqn{x_{i+10}}, resulting in \code{10*2 + 1 = 21} observations falling within
#' the window.
#'
#' The computations actually use a \code{slantedband} window, which is equivalent to the Sakoe-Chiba one
#' if series have equal lengths, and stays along the diagonal of the local cost matrix if series have
#' different lengths.
#'
#' @section Preprocessing:
#'
#' It is strongly advised to use z-normalization in case of \code{centroid = "shape"}, because the resulting
#' series have this normalization (see \code{\link{shape_extraction}}). The user can, however, specify a
#' custom function that performs any transformation on the data, but the user must make sure that the format
#' stays consistent, i.e. a matrix where each row is a series or a list of time series. For example,
#' the z-normalization could be implemented as \code{t(apply(data, 1, zscore))} or \code{lapply(data, zscore)}
#' respectively.
#'
#' The function will receive the data as first argument and, in case hierarchical methods are used, the
#' contents of \code{...} as the second argument.
#'
#' @section Repetitions:
#'
#' Due to their stochastic nature, partitional clustering is usually repeated several times with different random
#' seeds to allow for different starting points. This function can now run several repetitions by using the
#' \code{doRNG} package. This package ensures that each repetition uses a statistically independent random sequence.
#' The user only needs to provide an initial seed in the corresponding parameter of this function.
#'
#' Repetitions are greatly optimized when PAM centroids are used and the whole distance matrix is precomputed,
#' since said matrix is reused for every repetition, and can be comptued in parallel (see next section).
#'
#' @section Parallel Computing:
#'
#' Please note that running tasks in parallel does \strong{not} guarantee faster computations.
#' The overhead introduced is sometimes too large, and it's better to run tasks sequentially.
#'
#' The user can register a parallel backend with the \code{doParallel} package (and possibly any other package compatible with
#' \code{foreach}'s \code{\%dopar\%} operator) in order to do the
#' repetitions in parallel, as well as distance calculations (see the examples). \code{\link{TADPole}} also
#' takes advantage of parallel support.
#'
#' Unless each repetitions requires a few seconds, parallel computing probably isn't worth it. As such, I would only
#' use this feature with \code{shape} and \code{DBA} centroids, or for \code{pam.precompute} \code{=} \code{FALSE}
#' with a relatively small \code{k} and an expensive distance function like \code{DTW}.
#'
#' If you do more than 1 repetition sequentially, you can safely ignore the warning given by \code{dopar} about
#' no parallel backend registration.
#'
#' If the user registers a parallel backend, the function will also try to do the calculation of the distance
#' matrices in parallel. This should work with any function registered with \code{\link[proxy]{dist}} via
#' \code{\link[proxy]{pr_DB}} whose \code{loop} flag is set to \code{TRUE}. If the function requires special packages
#' to be loaded, provide their names in the \code{packages} argument. In addition, "dtwclust" is always loaded
#' in each parallel worker, so the user doesn't need to include that.
#'
#' Note that, by default, if a parallel backend is registered, multiple repetitions are to be performed (partitional clustering,
#' \code{reps} \code{>=} 1)
#' AND \code{centroid} \code{!=} \code{"pam"}, each parallel worker will get a repetition task, but any distance calculations
#' within each worker will be done sequentially. Load balance for such a scenario should be fine as long as the \code{reps}
#' \code{>=} the number of parallel workers. If you believe your task would benefit more from parallelization within each repetition,
#' consider registering the parallel backend and calling \code{dtwclust} several times sequentially, with \code{reps = 1} and
#' different \code{seeds}.
#'
#' @section Notes:
#'
#' Notice that the lower bounds are defined only for time series of equal lengths. \code{DTW} and \code{DTW2}
#' don't require this, but they are much slower to compute.
#'
#' The lower bounds are \strong{not} symmetrical, and \code{DTW} is only symmetrical if series are of equal
#' lengths.
#'
#' Specifying \code{distance = "sbd"} and \code{centroid = "shape"} is equivalent to the k-Shape algorithm
#' (Papparizos and Gravano, 2015). See \code{\link{SBD}} and \code{\link{shape_extraction}} for more info.
#'
#' @references
#'
#' Sakoe H and Chiba S (1978). ``Dynamic programming algorithm optimization for spoken word recognition.'' \emph{Acoustics, Speech
#' and
#' Signal Processing, IEEE Transactions on}, \strong{26}(1), pp. 43-49. ISSN 0096-3518, \url{http://doi.org/10.1109/TASSP.1978.1163055}.
#'
#' Ratanamahatana A and Keogh E (2004). ``Everything you know about dynamic time warping is wrong.'' In \emph{3rd Workshop on Mining
#' Temporal and Sequential Data, in conjunction with 10th ACM SIGKDD Int. Conf. Knowledge Discovery and Data Mining (KDD-2004),
#' Seattle, WA}.
#'
#' Paparrizos J and Gravano L (2015). ``k-Shape: Efficient and Accurate Clustering of Time Series.'' In \emph{Proceedings of the 2015
#' ACM SIGMOD International Conference on Management of Data}, series SIGMOD '15, pp. 1855-1870. ISBN 978-1-4503-2758-9, \url{
#' http://doi.org/10.1145/2723372.2737793}.
#'
#' @examples
#'
#' #### Load data
#' data(uciCT)
#'
#' # Reinterpolate to same length and coerce as matrix
#' data <- t(sapply(CharTraj, reinterpolate, newLength = 180))
#'
#' # Subset for speed
#' data <- data[1:20, ]
#' labels <- CharTrajLabels[1:20]
#'
#' #### Simple partitional clustering with L2 distance and PAM
#' kc.l2 <- dtwclust(data, k = 4, distance = "L2", centroid = "pam",
#'                   seed = 3247, trace = TRUE)
#' cat("Rand index for L2+PAM:", randIndex(kc.l2, labels), "\n\n")
#'
#' #### TADPole clustering
#' kc.tadp <- dtwclust(data, type = "tadpole", k = 4,
#'                     window.size = 20, dc = 1.5,
#'                     trace = TRUE)
#' cat("Rand index for TADPole:", randIndex(kc.tadp, labels), "\n\n")
#' plot(kc.tadp)
#'
#' # Modify plot
#' plot(kc.tadp, clus = 3:4, labs.arg = list(title = "TADPole, clusters 3 and 4",
#'                                           x = "time", y = "series"))
#'
#' #### Registering a custom distance with the 'proxy' package and using it
#' # Normalized DTW distance
#' ndtw <- function(x, y, ...) {
#'   dtw::dtw(x, y, step.pattern = symmetric2,
#'            distance.only = TRUE, ...)$normalizedDistance
#' }
#'
#' # Registering the function with 'proxy'
#' proxy::pr_DB$set_entry(FUN = ndtw, names=c("nDTW"),
#'                        loop = TRUE, type = "metric", distance = TRUE,
#'                        description = "Normalized DTW with L1 norm")
#'
#' # Subset of (original) data for speed
#' # Change pam.precompute to FALSE to see time difference
#' kc.ndtw <- dtwclust(CharTraj[31:40], distance = "nDTW",
#'                     trace = TRUE, pam.precompute = TRUE,
#'                     seed = 8319)
#' cat("Rand index for nDTW (subset):",
#'     randIndex(kc.ndtw, CharTrajLabels[31:40]), "\n\n")
#' plot(kc.ndtw)
#'
#' #### Hierarchical clustering based on shabe-based distance (different lengths)
#' hc.sbd <- dtwclust(CharTraj, type = "hierarchical",
#'                    distance = "sbd", trace = TRUE)
#' cl.sbd <- cutree(hc.sbd, 20)
#' cat("Rand index for HC+SBD:", randIndex(cl.sbd, CharTrajLabels), "\n\n")
#'
#' \dontrun{
#' #### Saving and modifying the ggplot object with custom time
#' t <- seq(Sys.Date(), len = 180, by = "day")
#' gkc <- plot(kc.l2, time = t, plot = FALSE)
#'
#' require(scales)
#' gkc + scale_x_date(labels = date_format("%b-%Y"),
#'                    breaks = date_breaks("2 months"))
#'
#' #### Using parallel computation to optimize several random repetitions
#' #### and distance matrix calculation
#' require(doParallel)
#'
#' # Create parallel workers
#' cl <- makeCluster(detectCores())
#' registerDoParallel(cl)
#'
#' ## Use full DTW and PAM
#' kc.dtw <- dtwclust(CharTraj, k = 20, seed = 3251, trace = TRUE)
#'
#' ## Use full DTW with DBA centroids
#' kc.dba <- dtwclust(CharTraj, k = 20, centroid = "dba", seed = 3251, trace = TRUE)
#'
#' ## Use constrained DTW with original series of different lengths
#' kc.cdtw <- dtwclust(CharTraj, k = 20, window.size = 20,
#'                     seed = 3251, trace = TRUE)
#'
#' ## This uses the "nDTW" function registered in another example above
#' # For reference, this took around 2.25 minutes with 8 cores (all 8 repetitions).
#' kc.ndtw.list <- dtwclust(CharTraj, k = 20, distance = "nDTW",
#'                          centroid = "dba", window.size = 10,
#'                          preproc = zscore, seed = 8319,
#'                          save.data = FALSE, reps = 8L)
#'
#' # Stop parallel workers
#' stopCluster(cl)
#'
#' # Return to sequential computations
#' registerDoSEQ()
#'
#' # See Rand Index for each repetition
#' sapply(kc.ndtw.list, randIndex, y = CharTrajLabels)
#' }
#'
#' @seealso
#'
#' Please check the brief description in \code{\link{dtwclust-package}}.
#'
#' Additionally: \code{\link{plot-dtwclust}}, \code{\link{dtwclust-class}}.
#'
#' @author Alexis Sarda-Espinosa
#'
#' @param data A list where each element is a time series, or a numerical matrix where each row is a time
#' series. All series must have equal lengths in case of \code{type = "tadpole"}.
#' @param type What type of clustering method to use, \code{partitional}, \code{hierarchical} or \code{tadpole}.
#' @param k Numer of desired clusters in partitional methods.
#' @param method Which linkage method to use in hierarchical methods. See \code{\link[stats]{hclust}}.
#' @param distance One of the supported distance definitions (see Distance section). Ignored for
#' \code{type = "tadpole"}.
#' @param centroid Either a supported string or an appropriate function to calculate centroids
#' when using partitional methods (see Centroid section).
#' @param preproc Function to preprocess data. Defaults to \code{zscore} \emph{only} if \code{centroid}
#' \code{=} \code{"shape"}, but will be replaced by a custom function if provided. See Preprocessing section.
#' @param window.size Window constraint for DTW and LB calculations. See Sakoe-Chiba section.
#' @param norm Pointwise distance for DTW, DBA and the LB. Either \code{L1} for Manhattan distance or \code{L2}
#' for Euclidean. Ignored for \code{distance = "DTW"} (which always uses \code{L1}) and
#' \code{distance = "DTW2"} (which always uses \code{L2}).
#' @param dc Cutoff distance for TADPole algorithm.
#' @param dba.iter Maximum number of iterations for \code{\link{DBA}} centroids.
#' @param pam.precompute Precompute the whole distance matrix once and reuse it at each iteration if using PAM
#' centroids. Otherwise calculate distances at every iteration.
#' @param control Parameters for partitional clustering algorithms. See
#' \code{\link[flexclust]{flexclustControl}}.
#' @param save.data Return a copy of the data in the returned object? Ignored for hierarchical clustering.
#' @param seed Random seed for reproducibility of partitional algorithms.
#' @param trace Boolean flag. If true, more output regarding the progress is printed to screen.
#' @param reps How many times to repeat partitional clustering with different starting points. See section Repetitions.
#' @param packages Character vector with the names of any packages required for custom \code{proxy} functions. See
#' Parallel Computing section.
#' @param ... Additional arguments to pass to \code{\link[proxy]{dist}} or a custom function.
#'
#' @return An object with formal class \code{\link{dtwclust-class}} if \code{type = "partitional" | "tadpole"}.
#' Otherwise an object with class \code{hclust} as returned by \code{\link[stats]{hclust}}. If \code{reps > 1}
#' and a partitional procedure is used, a list of objects is returned.
#'
#' @export
#' @import flexclust
#' @import doRNG
#' @import proxy
#' @import foreach
#' @importFrom parallel splitIndices
#' @importFrom stats median
#' @importFrom stats hclust
#' @importFrom modeltools ModelEnvMatrix

dtwclust <- function(data = NULL, type = "partitional", k = 2L, method = "average",
                     distance = "dtw", centroid = "pam", preproc = NULL,
                     window.size = NULL, norm = "L1", dc = NULL,
                     dba.iter = 15L, pam.precompute = TRUE, control = NULL,
                     save.data = TRUE, seed = NULL, trace = FALSE,
                     reps = 1L, packages = character(0),
                     ...)
{
     ## =================================================================================================================
     ## Start
     ## =================================================================================================================

     if (is.null(data))
          stop("No data provided")

     tic <- proc.time()

     type <- match.arg(type, c("partitional", "hierarchical", "tadpole"))
     norm <- match.arg(norm, c("L1", "L2"))

     if (type == "partitional") {

          ## =================================================================================================================
          ## Partitional
          ## =================================================================================================================

          if (k < 2)
               stop("At least two clusters must be defined")
          if (reps < 1)
               stop("At least one repetition must be performed")

          ## For parallel computation
          packages <- c("dtwclust", packages)

          if (is.character(centroid))
               centroid <- match.arg(centroid, c("mean", "median", "shape", "dba", "pam"))

          ## Distance function (cent is unusued, allcent is set appropriately below)
          if (is.function(distance)) {
               family <- kccaFamily(name = as.character(substitute(distance))[1],
                                    dist = distance,
                                    cent = NULL)

          } else if (is.character(distance)) {
               # dtwdistfun.R
               distfun <- dtwdistfun(distance = distance,
                                     window.size = window.size,
                                     norm = norm,
                                     distmat = NULL,
                                     packages = packages,
                                     ...)

               family <- flexclust::kccaFamily(name = distance,
                                               dist = distfun,
                                               cent = NULL)

          } else {
               stop("Unsupported distance definition")
          }

          ## ----------------------------------------------------------------------------------------------------------
          ## Replace specific functions if necessary
          ## ----------------------------------------------------------------------------------------------------------

          # precompute distance matrix?
          if (is.character(centroid) && centroid == "pam" && pam.precompute) {
                    distmat <- distmat_pam(data, family) # utils.R

                    ## Redefine dist with new distmat (to update closure)
                    family@dist <- dtwdistfun(distance = distance,
                                              window.size = window.size,
                                              norm = norm,
                                              distmat = distmat,
                                              packages = packages,
                                              ...)


          } else {
               distmat <- NULL
          }

          if (!is.null(preproc)) {
               if (is.function(preproc))
                    family@preproc <- preproc # highest priority, possibly replaces zscore
               else
                    stop("Invalid preprocessing")

          } else if (is.character(centroid) && centroid == "shape") {
               preproc <- "zscore" # character to indicate what was done
               family@preproc <- zscore

          } else {
               preproc <- "none" # character to indicate what was done
          }

          ## Closures, all_cent.R
          family@allcent <- all_cent(case = centroid,
                                     distmat = distmat,
                                     distfun = family@dist,
                                     dba.iter = dba.iter,
                                     window.size = window.size,
                                     norm = norm,
                                     trace = trace)

          ## ----------------------------------------------------------------------------------------------------------
          ## Further options
          ## ----------------------------------------------------------------------------------------------------------

          if (is.null(control))
               ctrl <- new("flexclustControl")
          else
               ctrl <- as(control, "flexclustControl")

          if (trace) {
               if (reps > 1L)
                    message("Tracing will not be available if parallel computing is used.")

               ctrl@verbose <- 1L
          }

          if (!is.null(seed))
               set.seed(seed)

          ## ----------------------------------------------------------------------------------------------------------
          ## Cluster
          ## ----------------------------------------------------------------------------------------------------------

          if (is.list(data)) {
               ## Cluster list
               lengths <- sapply(data, length)

               if (length(unique(lengths)) > 1) {
                    consistency_check(distance, "dist")
                    consistency_check(centroid, "cent")
               }

               if (reps > 1L) {
                    ## I need to re-register any custom distances in each parallel worker
                    if (is.character(distance))
                         dist_entry <- proxy::pr_DB$get_entry(distance)
                    else
                         dist_entry <- NULL

                    kc.list <- foreach(i = 1:reps,
                                       .combine = list,
                                       .multicombine = TRUE,
                                       .packages = packages) %dorng% {
                                            if (!is.null(dist_entry) && !proxy::pr_DB$entry_exists(dist_entry$names[1]))
                                                 do.call(proxy::pr_DB$set_entry, dist_entry)

                                            kcca.list(x = data,
                                                      k = k,
                                                      family = family,
                                                      control = ctrl)
                                       }
               } else {
                    ## Just one repetition
                    kc.list <- kcca.list(x = data,
                                         k = k,
                                         family = family,
                                         control = ctrl)
               }

          } else {

               ## Cluster matrix
               if (reps > 1L) {
                    ## I need to re-register any custom distances in each parallel worker
                    if (is.character(distance))
                         dist_entry <- proxy::pr_DB$get_entry(distance)
                    else
                         dist_entry <- NULL

                    kc.list <- foreach(i = 1:reps,
                                       .combine = list,
                                       .multicombine = TRUE,
                                       .packages = packages) %dorng% {
                                            if (!is.null(dist_entry) && !proxy::pr_DB$entry_exists(dist_entry$names[1]))
                                                 do.call(proxy::pr_DB$set_entry, dist_entry)

                                            flexclust::kcca(x = data,
                                                            k = k,
                                                            family = family,
                                                            simple = TRUE,
                                                            control = ctrl,
                                                            save.data = save.data)
                                       }
               } else {
                    ## Just one repetition
                    kc.list <- flexclust::kcca(x = data,
                                               k = k,
                                               family = family,
                                               simple = TRUE,
                                               control = ctrl,
                                               save.data = save.data)
               }
          }

          ## ----------------------------------------------------------------------------------------------------------
          ## Prepare results
          ## ----------------------------------------------------------------------------------------------------------

          toc <- proc.time() - tic

          if (save.data) {
               datalist <- consistency_check(data, "tsmat")

               if (reps > 1L)
                    message("\nConsider setting save.data to FALSE if performing several repetitions.")
          } else
               datalist <- list()

          if (is.list(kc.list)) {

               dtwc <- lapply(kc.list, function(kc) {
                    new("dtwclust", kc,
                        type = type,
                        distance = ifelse(is.function(distance), as.character(substitute(distance))[1], distance),
                        centroid = ifelse(is.function(centroid), as.character(substitute(centroid))[1], centroid),
                        preproc = ifelse(is.function(preproc), as.character(substitute(preproc))[1], preproc),
                        datalist = datalist,
                        proctime = toc)
               })

          } else {

               dtwc <- new("dtwclust", kc.list,
                           type = type,
                           distance = ifelse(is.function(distance), as.character(substitute(distance))[1], distance),
                           centroid = ifelse(is.function(centroid), as.character(substitute(centroid))[1], centroid),
                           preproc = ifelse(is.function(preproc), as.character(substitute(preproc))[1], preproc),
                           datalist = datalist,
                           proctime = toc)
          }

          if (trace)
               cat("\n\tElapsed time is", toc["elapsed"], "seconds.\n\n")

          dtwc

     } else if (type == "hierarchical") {

          ## =================================================================================================================
          ## Hierarchical
          ## =================================================================================================================

          if (trace)
               cat("\n\tCalculating distance matrix...\n")

          x <- consistency_check(data, "tsmat")

          lengths <- sapply(x, length)

          if (length(unique(lengths)) > 1)
               consistency_check(distance, "dist")

          if (!is.null(preproc) && is.function(preproc)) {
               x <- preproc(x, ...)
          }

          if (is.function(distance)) {
               D <- distance(x, ...)

          } else if (is.character(distance)) {
               ## Take advantage of the function I defined for the partitional methods
               ## Which can do calculations in parallel if appropriate
               distfun <- dtwdistfun(distance = distance,
                                     window.size = window.size,
                                     norm = norm,
                                     distmat = NULL,
                                     packages = packages,
                                     ...)

               ## TRUE is to calculate whole distance matrix
               D <- distfun(x, x, whole = TRUE)

          } else {
               stop("Unspported distance definition")
          }

          if (trace)
               cat("\n\tPerforming hierarchical clustering...\n")

          ## Required form for 'hclust'
          DD <- D[lower.tri(D)]

          ## Needed attribute for 'hclust' (case sensitive)
          attr(DD, "Size") <- length(x)
          attr(DD, "method") <- attr(D, "method")

          hc <- stats::hclust(DD, method = method)

          toc <- proc.time() - tic
          class(toc) <- "numeric"

          if (trace)
               cat("\n\tElapsed time is", toc["elapsed"], "seconds.\n\n")

          hc

     } else if (type == "tadpole") {

          ## =================================================================================================================
          ## TADPole
          ## =================================================================================================================

          MYCALL <- match.call()
          window.size <- consistency_check(window.size, "window")

          if (is.null(dc))
               stop("The user must specify 'dc' for this method")
          if (dc < 0)
               stop("The cutoff distance 'dc' must be positive")

          ## ----------------------------------------------------------------------------------------------------------
          ## Adjust inputs
          ## ----------------------------------------------------------------------------------------------------------

          if (!is.null(preproc)) {
               if (is.function(preproc))
                    data <- preproc(data, ...)
               else
                    stop("Invalid preprocessing")

          } else {
               preproc <- "none"
          }

          if (is.list(data)) {
               consistency_check(data, "tslist")
               x <- data
               data <- t(sapply(data, rbind))

          } else if (is.matrix(data)) {
               x <- consistency_check(data, "tsmat")

          } else {
               stop("Unsupported format for data")
          }

          ## ----------------------------------------------------------------------------------------------------------
          ## Cluster
          ## ----------------------------------------------------------------------------------------------------------

          if (trace)
               cat("\nEntering TADPole...\n")

          R <- TADPole(x, window.size = window.size, k = k, dc = dc, error.check = FALSE)

          if (trace) {
               cat("\nTADPole completed, pruning percentage = ",
                   formatC(100-R$distCalcPercentage, digits = 3, width = -1, format = "fg"),
                   "%\n",
                   sep = "")
          }

          ## ----------------------------------------------------------------------------------------------------------
          ## Prepare results
          ## ----------------------------------------------------------------------------------------------------------

          toc <- proc.time() - tic

          if (save.data) {
               tadpc <- new("dtwclust",
                            type = type,
                            distance = "DTW2",
                            centroid = "TADPole (PAM)",
                            preproc = ifelse(is.function(preproc), as.character(substitute(preproc))[1], preproc),
                            proctime = toc,

                            call = MYCALL,
                            centers = data[R$centers, , drop = FALSE],
                            k = as.integer(k),
                            cluster = as.integer(R$cl),
                            data = modeltools::ModelEnvMatrix(designMatrix = data),
                            datalist = x)
          } else {
               tadpc <- new("dtwclust",
                            type = type,
                            distance = "DTW2",
                            centroid = "TADPole (PAM)",
                            preproc = ifelse(is.function(preproc), as.character(substitute(preproc))[1], preproc),
                            proctime = toc,

                            call = MYCALL,
                            centers = data[R$centers, , drop = FALSE],
                            k = as.integer(k),
                            cluster = as.integer(R$cl))
          }

          if (trace)
               cat("\n\tElapsed time is", toc["elapsed"], "seconds.\n\n")

          tadpc
     }
}
