#' Chainrule
#'
#' Calculates the partial derivatives of the function \eqn{h(x_1,x_2,...,x_{K})=f(g(x_1,x_2,...,x_{K}))} up to order four. Here \eqn{K} is the number of inputs for function \eqn{g(\cdot)}.
#' The function \eqn{f(\cdot)} can only have a single input. Alternatively \code{chainrule} can calculate the partial derivatives of \eqn{h(x_1,x_2,...,x_{K})=f(g_1(x_1),g_2(x_2),...,g_K(x_{K}))} up to order four.
#' Here each \eqn{g_k(\cdot)} can only take a single input \eqn{x_k}. If \eqn{x_i \neq x_j} for \eqn{i,j \in \{1,...,K \}} then the input argument \code{g} can be a list with elements \eqn{g_k(x_k)}.
#' The function checks the number of inputs of \eqn{f(\cdot)} by counting the number of partial derivatives and then decides automatically how to proceed.
#'
#' @param f vector of \eqn{f(\cdot)} evaluated at \eqn{g(\cdot)} with derivatives as attributes.
#' @param g matrix of \eqn{g(\cdot)} with derivatives as attributes. Alternatively written as \eqn{g(\cdot)=g_1(x_1),g_2(x_2),...,g_K(x_{K}))}. In this case, \code{g} can be a vector or a list of length \code{k}
#' @param deriv derivative of order \code{deriv}. Available are \code{0},\code{2} and \code{4}.
#' @param tri optional, index arrays for upper triangular for g, generated by \code{\link[mgcv:trind.generator]{trind.generator()}}.
#'
#' @return A list with partial derivatives. The index of the list corresponds to a matrix with all partial derivatives of that order.
#'
#' @details Mostly internal function, which is helpful in calculating the partial derivatives of the loglikelihood.
#'
#'
#' @examples
#' x<-1 #For K=1, x_1 value is set to 1.
#'
#' g<-1/x #g(x_1) = 1/x
#' attr(g,"gradient")<-matrix(-1/x^2,ncol=1)
#' attr(g,"hessian")<-matrix(2/x^3,ncol=1)
#' attr(g,"l3")<-matrix(-6/x^4,ncol=1)
#' attr(g,"l4")<-matrix(24/x^5,ncol=1)
#'
#'
#' zeta_g<-zeta(g, deriv=4) #f(g(x)) = zeta(g(x))
#'
#' chainrule(f=zeta_g, g=g, deriv=4)
#'
#' @export
#'
#chainrule
chainrule<-function(f=NULL, g=NULL, deriv=2, tri=NULL){
  if(is.list(g)){
    nparg<-length(g)

    #Check if index arrays for upper triangular is available, else create it
    if(is.null(tri)){
      tri<-mgcv::trind.generator(nparg)
    }
    tri$i1<-1:nparg
    tri$i1r<-1:nparg

    g_list<-g
    n<-length(g_list[[1]])
    g<-matrix(unlist(g_list), ncol=nparg)

    for(i in 1:nparg){
      if(deriv<=2){
        for(j in c("gradient","hessian")){
          attr(g,j)<-cbind(attr(g,j),attr(g_list[[i]],j))
        }
      } else {
        for(j in c("gradient","hessian","l3","l4")){
          attr(g,j)<-cbind(attr(g,j),attr(g_list[[i]],j))
        }
      }
    }

    #Initialize derivative matrix h
    g1<-matrix(0, nrow=n, ncol=length(tri$i1r))
    g2<-matrix(0, nrow=n, ncol=length(tri$i2r))
    if(deriv>2){
      g3<-matrix(0, nrow=n, ncol=length(tri$i3r))
      g4<-matrix(0, nrow=n, ncol=length(tri$i4r))
    }

    for(i in 1:nparg){
      g1[,tri$i1[i]]<-attr(g_list[[i]],"gradient")
      g2[,tri$i2[i,i]]<-attr(g_list[[i]],"hessian")
      if(deriv>2){
        g3[,tri$i3[i,i,i]]<-attr(g_list[[i]],"l3")
        g4[,tri$i4[i,i,i,i]]<-attr(g_list[[i]],"l4")
      }
    }
  } else {
    g1<-attr(g,"gradient")
    g2<-attr(g,"hessian")
    if(deriv>2){
      g3<-attr(g,"l3")
      g4<-attr(g,"l4")
    }
    #Get number of parameters
    nparg<-ncol(g1)

    #Number of observations
    n<-nrow(g1)

    #Check if index arrays for upper triangular is available, else create it
    if(is.null(tri)){
      tri<-mgcv::trind.generator(nparg)
    }

    tri$i1<-1:nparg
    tri$i1r<-1:nparg
  }

  f1<-attr(f,"gradient")
  f2<-attr(f,"hessian")


  if(deriv>2){
    f3<-attr(f,"l3")
    f4<-attr(f,"l4")
  }

  #Initialize derivative matrix h
  h1<-matrix(NA, nrow=n, ncol=length(tri$i1r))
  h2<-matrix(NA, nrow=n, ncol=length(tri$i2r))
  h3<-matrix(NA, nrow=n, ncol=length(tri$i3r))
  h4<-matrix(NA, nrow=n, ncol=length(tri$i4r))

  if(length(f1)!=n){
    f_multioutput<-F
  } else {
    f_multioutput<-T
  }

  if(length(f1)==n){
    for(i in 1:nparg){
      #First derivative
      h1[,i]<-f1*g1[,i]

      for(j in i:nparg){
        #Second derivative
        h2[,tri$i2[i,j]]<-f2*g1[,tri$i1[j]]*g1[,tri$i1[i]]+f1*g2[,tri$i2[i,j]]

        if(deriv>2){
          for(k in j:nparg){
            #Third derivative
            h3[,tri$i3[i,j,k]]<-f3*g1[,tri$i1[k]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+
              f2*g2[,tri$i2[j,k]]*g1[,tri$i1[i]]+
              f2*g1[,tri$i1[j]]*g2[,tri$i2[i,k]]+
              f2*g1[,tri$i1[k]]*g2[,tri$i2[i,j]]+
              f1*g3[,tri$i3[i,j,k]]

            for(l in k:nparg){
              #Fourth derivative
              h4[,tri$i4[i,j,k,l]]<-f4*g1[,tri$i1[l]]*g1[,tri$i1[k]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+
                f3*g2[,tri$i2[k,l]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+
                f3*g1[,tri$i1[k]]*g2[,tri$i2[j,l]]*g1[,tri$i1[i]]+
                f3*g1[,tri$i1[k]]*g1[,tri$i1[j]]*g2[,tri$i2[i,l]]+
                #next line
                f3*g1[,tri$i1[l]]*g2[,tri$i2[j,k]]*g1[,tri$i1[i]]+
                f2*g3[,tri$i3[j,k,l]]*g1[,tri$i1[i]]+
                f2*g2[,tri$i2[j,k]]*g2[,tri$i2[i,l]]+
                #next line
                f3*g1[,tri$i1[l]]*g1[,tri$i1[j]]*g2[,tri$i2[i,k]]+
                f2*g2[,tri$i2[j,l]]*g2[,tri$i2[i,k]]+
                f2*g1[,tri$i1[j]]*g3[,tri$i3[i,k,l]]+
                #next line
                f3*g1[,tri$i1[l]]*g1[,tri$i1[k]]*g2[,tri$i2[i,j]]+
                f2*g2[,tri$i2[k,l]]*g2[,tri$i2[i,j]]+
                f2*g1[,tri$i1[k]]*g3[,tri$i3[i,j,l]]+
                #next line
                f2*g1[,tri$i1[l]]*g3[,tri$i3[i,j,k]]+
                f1*g4[,tri$i4[i,j,k,l]]

            }
          }
        }
      }
    }
  } else {
    for(i in 1:nparg){
      #First derivative
      h1[,i]<-f1[,i]*g1[,i]

      for(j in i:nparg){
        #Second derivative
        h2[,tri$i2[i,j]]<-f2[,tri$i2[i,j]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+f1[,tri$i1[i]]*g2[,tri$i2[i,j]]
        if(deriv>2){
          for(k in j:nparg){
            #Third derivative
            h3[,tri$i3[i,j,k]]<-f3[,tri$i3[i,j,k]]*g1[,tri$i1[k]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+
              f2[,tri$i2[i,j]]*g2[,tri$i2[j,k]]*g1[,tri$i1[i]]+
              f2[,tri$i2[i,j]]*g1[,tri$i1[j]]*g2[,tri$i2[i,k]]+
              f2[,tri$i2[i,k]]*g1[,tri$i1[k]]*g2[,tri$i2[i,j]]+
              f1[,tri$i1[i]]*g3[,tri$i3[i,j,k]]

            for(l in k:nparg){
              #Fourth derivative
              h4[,tri$i4[i,j,k,l]]<-f4[,tri$i4[i,j,k,l]]*g1[,tri$i1[l]]*g1[,tri$i1[k]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+
                f3[,tri$i3[i,j,k]]*g2[,tri$i2[k,l]]*g1[,tri$i1[j]]*g1[,tri$i1[i]]+
                f3[,tri$i3[i,j,k]]*g1[,tri$i1[k]]*g2[,tri$i2[j,l]]*g1[,tri$i1[i]]+
                f3[,tri$i3[i,j,k]]*g1[,tri$i1[k]]*g1[,tri$i1[j]]*g2[,tri$i2[i,l]]+
                #next line
                f3[,tri$i3[i,j,l]]*g1[,tri$i1[l]]*g2[,tri$i2[j,k]]*g1[,tri$i1[i]]+
                f2[,tri$i2[i,j]]*g3[,tri$i3[j,k,l]]*g1[,tri$i1[i]]+
                f2[,tri$i2[i,j]]*g2[,tri$i2[j,k]]*g2[,tri$i2[i,l]]+
                #next line
                f3[,tri$i3[i,j,l]]*g1[,tri$i1[l]]*g1[,tri$i1[j]]*g2[,tri$i2[i,k]]+
                f2[,tri$i2[i,j]]*g2[,tri$i2[j,l]]*g2[,tri$i2[i,k]]+
                f2[,tri$i2[i,j]]*g1[,tri$i1[j]]*g3[,tri$i3[i,k,l]]+
                #next line
                f3[,tri$i3[i,k,l]]*g1[,tri$i1[l]]*g1[,tri$i1[k]]*g2[,tri$i2[i,j]]+
                f2[,tri$i2[i,k]]*g2[,tri$i2[k,l]]*g2[,tri$i2[i,j]]+
                f2[,tri$i2[i,k]]*g1[,tri$i1[k]]*g3[,tri$i3[i,j,l]]+
                #next line
                f2[,tri$i2[i,l]]*g1[,tri$i1[l]]*g3[,tri$i3[i,j,k]]+
                f1[,tri$i1[i]]*g4[,tri$i4[i,j,k,l]]
            }
          }
        }
      }
    }
  }


  #Write output as a list
  out<-remove_attr(f)
  attr(out,"gradient")<-h1
  attr(out,"hessian")<-h2

  if(deriv>2){
    attr(out,"l3")<-h3
    attr(out,"l4")<-h4
  }

  #Return output
  return(out)
}

