% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sum.R
\name{dot_sst}
\alias{dot_sst}
\alias{dot_chisq}
\alias{dot_fisher}
\alias{dot_art}
\alias{dot_arta}
\alias{dot_rtp}
\alias{dot_tpm}
\title{Methods for combining decorrelated summary statistics}
\usage{
dot_chisq(Z, C, ...)

dot_fisher(Z, C, ...)

dot_art(Z, C, k = NULL, ...)

dot_arta(Z, C, k = NULL, w = NULL, ...)

dot_rtp(Z, C, k = NULL, ...)

dot_tpm(Z, C, tau = 0.05, ...)
}
\arguments{
\item{Z}{vector of association test statistics (i.e., Z-scores).}

\item{C}{matrix of  correlation among the  test statistics, as  obtained by
\code{\link[=cst]{cst()}}.}

\item{...}{additional parameters}

\item{k}{combine \code{k} smallest (decorrelated) P-values.}

\item{w}{weight assigned to partial sums in ARTA implementation; default is 1.}

\item{tau}{combine (decorrelated) P-values no large than tau; default is 0.05.}
}
\value{
a list of
\itemize{
\item{\code{X}:} {decorrelated  association statistics.}
\item{\code{H}:} {orthogonal transformation, such that \code{X = H \%*\% Z}.}
\item{\code{Y}:} {the overall combined statistic.}
\item{\code{P}:} {the P-value corresponding to \code{Y}.}
}

for Augmented Rank Truncated Adaptive (ARTA) test,
\itemize{
\item{k:} {the number of decorrelated P-values that were adaptively picked.}}

for Truncated Product Method (TPM),
\itemize{
\item{k:} {the number of decorrelated P-values \eqn{\le} \code{tau}.}}
}
\description{
Decorrelates and combines per-variant genetic association test statistics, \code{Z},
given the correlation matrix among them, \code{C}.
}
\details{
These functions  first call \code{\link[=dot]{dot()}}  to decorrelate the  genetic association
test  statistics and  then provide  various options  to combine  independent
statistics or corresponding P-values into the overall statistic and P-value.

The two  rank truncated  tests (i.e.,  \code{\link[=dot_art]{dot_art()}}, \code{\link[=dot_rtp]{dot_rtp()}})  require an
additional parameter \code{k} that specifes the number of smallest (decorrelated)
P-values to combine. By default, \code{k}  equals half of the number of variants.
The adaptive  rank truncation  method, \code{\link[=dot_arta]{dot_arta()}}, determines  the optimal
truncation value between 1 and \code{k}.

The truncated  product method,  \code{\link[=dot_tpm]{dot_tpm()}}, combines  P-values at  least as
small as \code{tau} (0.05 by default).  If  \code{tau} is equal to 1, then \code{\link[=dot_tpm]{dot_tpm()}}
provides  the  same result  as  \code{\link[=dot_fisher]{dot_fisher()}}  (i.e., Fisher's  method  for
combining  P-values). Similarly,  if \code{k}  is equal  to the  total number  of
tests, the results  of \code{\link[=dot_art]{dot_art()}} and \code{\link[=dot_rtp]{dot_rtp()}} will be  the same as that
of \code{\link[=dot_fisher]{dot_fisher()}}.

Reference  (\strong{a})  below  details  how to  combine  decorrelated  test
statistics  or  P-values  via  \code{\link[=dot_art]{dot_art()}},  \code{\link[=dot_rtp]{dot_rtp()}}  and  \code{\link[=dot_arta]{dot_arta()}};
reference (\strong{b}) details \code{\link[=dot_tpm]{dot_tpm()}} method.
}
\section{Functions}{
\itemize{
\item \code{dot_chisq}: decorrelation followed by a Chi-square test.

\item \code{dot_fisher}: decorrelated Fisher's combined P-value test.

\item \code{dot_art}: decorrelated Augmented Rank Truncated (ART) test.

\item \code{dot_arta}: decorrelated Augmented Rank Truncated Adaptive (ARTA) test.

\item \code{dot_rtp}: decorrelated Rank Truncated Product (RTP) test.

\item \code{dot_tpm}: decorrelated Truncated Product Method (TPM) test.
}}

\examples{
## get the test statistics and pre-calculated LD matrix
stt <- readRDS(system.file("extdata", 'art_zsc.rds', package="dotgen"))
sgm <- readRDS(system.file("extdata", 'art_ldm.rds', package="dotgen"))


## decorrelated chi-square test
result <- dot_chisq(stt, sgm)
print(result$Y)  # 37.2854
print(result$P)  # 0.0003736988

## decorrelated Fisher's combined P-value chi-square test
result <- dot_fisher(stt, sgm)
print(result$Y)  # 58.44147
print(result$P)  # 0.0002706851

## decorrelated augmented rank truncated (ART) test.
result <- dot_art(stt, sgm, k=6)
print(result$Y)  # 22.50976
print(result$P)  # 0.0006704994

## decorrelated Augmented Rank Truncated Adaptive (ARTA) test
result <- dot_arta(stt, sgm, k=6)
print(result$Y)  # -1.738662
print(result$k)  #  5 smallest P-values are retained
print(result$P)  #  0.003165 (varies)

## decorrelated Rank Truncated Product (RTP)
result <- dot_rtp(stt, sgm, k=6)
print(result$Y)  # 22.6757
print(result$P)  # 0.0007275518

## decorrelated Truncated Product Method (TPM)
result <- dot_tpm(stt, sgm, tau=0.05)
print(result$Y)  #  1.510581e-08
print(result$k)  #  6 P-values <= tau
print(result$P)  #  0.0007954961

}
\references{
(a) \href{https://www.frontiersin.org/articles/10.3389/fgene.2019.01051/full}{
Vsevolozhskaya, O.   A., Hu, F., &  Zaykin, D.  V. (2019).   \emph{Detecting weak
signals  by  combining  small  P-values  in  genetic  association  studies.}
Frontiers in genetics, 10, 1051.}

(b) \href{https://onlinelibrary.wiley.com/doi/abs/10.1002/gepi.0042}{Zaykin,
D.    V.,   Zhivotovsky,    L.     A.,   Westfall,    P.    H.,   &    Weir,
B. S.  (2002). \emph{Truncated  product method  for combining  P-values.} Genetic
Epidemiology, 22(2), 170-185.}
}
\seealso{
\code{\link[=dot]{dot()}}
}
