#' The \emph{doRNG} package provides functions to perform 
#' reproducible parallel foreach loops, using independent random streams
#' as generated by L'Ecuyer's combined multiple-recursive generator [L'Ecuyer (1999)]. 
#' It enables to easily convert standard %dopar% loops into fully reproducible loops, 
#' independently of the number of workers, the task scheduling strategy, 
#' or the chosen parallel environment and associated foreach backend.
#' It has been tested with the following foreach backend: doMC, doSNOW, doMPI. 
#' 
#' \tabular{ll}{
#' Package: \tab doRNG\cr
#' Type: \tab Package\cr
#' Version: \tab 1.2.3\cr
#' Date: \tab 2012-03-30\cr
#' License: \tab GPL (>= 2)\cr
#' LazyLoad: \tab yes\cr
#' }
#'
#' @author
#' Renaud Gaujoux \email{renaud@@cbio.uct.ac.za},
#'
#' Maintainer: Renaud Gaujoux \email{renaud@@cbio.uct.ac.za},
#' @name doRNG-package
#' @docType package
#' @title Generic Reproducible Parallel Backend for foreach Loops
#' @keywords package
#' @seealso \code{\link{doRNG}}, \code{\link{RNGseq}}
#' @cite Lecuyer1999
#' @examples
#' 
#' # register parallel backend
#' library(doParallel)
#' cl <- makeCluster(2)
#' registerDoParallel(cl)
#'
#' ## standard %dopar% loop are not reproducible
#' set.seed(123)
#' r1 <- foreach(i=1:4) %dopar%{ runif(1) }
#' set.seed(123)
#' r2 <- foreach(i=1:4) %dopar%{ runif(1) }
#' identical(r1, r2)
#' \dontshow{ stopifnot(!identical(r1, r2)) }
#' 
#' ## %dorng% loops _are_ reproducible
#' set.seed(123)
#' r1 <- foreach(i=1:4) %dorng%{ runif(1) }
#' set.seed(123)
#' r2 <- foreach(i=1:4) %dorng%{ runif(1) }
#' identical(r1, r2)
#' \dontshow{ stopifnot(identical(r1, r2)) }
#' 
#' # alternative way of seeding 
#' a1 <- foreach(i=1:4, .options.RNG=123) %dorng%{ runif(1) }
#' a2 <- foreach(i=1:4, .options.RNG=123) %dorng%{ runif(1) }
#' identical(a1, a2) && identical(a1, r1)
#' \dontshow{ stopifnot(identical(a1, a2) && identical(a1, r1)) }
#' 
#' ## sequences of %dorng% loops _are_ reproducible
#' set.seed(123)
#' s1 <- foreach(i=1:4) %dorng%{ runif(1) }
#' s2 <- foreach(i=1:4) %dorng%{ runif(1) }
#' identical(s1, r1) && !identical(s1, s2)
#' \dontshow{ stopifnot(identical(s1, r1) && !identical(s1, s2)) }
#'
#' set.seed(123)
#' s1.2 <- foreach(i=1:4) %dorng%{ runif(1) }
#' s2.2 <- foreach(i=1:4) %dorng%{ runif(1) }
#' identical(s1, s1.2) && identical(s2, s2.2)
#' \dontshow{ stopifnot(identical(s1, s1.2) && identical(s2, s2.2)) }
#'  
#' ## Non-invasive way of converting %dopar% loops into reproducible loops
#' registerDoRNG(123)
#' s3 <- foreach(i=1:4) %dopar%{ runif(1) }
#' s4 <- foreach(i=1:4) %dopar%{ runif(1) }
#' identical(s3, s1) && identical(s4, s2)
#' \dontshow{ stopifnot(identical(s3, s1) && identical(s4, s2)) }
#' 
#' stopCluster(cl) 
#'
NULL

#' Generate a Bibtex File from Package Citations
#' 
#' Generates a Bibtex file from a list of packages or all the installed packages.
#' It is useful for adding relevant citations in Sweave documents.
#' 
#' Multiple citations are handled by adding a numeric suffix to the Bibtex key 
#' (other than the first/main citation) as \code{"<pkgname>\%i"} (e.g. pkg, pkg2, pkg3).
#' 
#' This function has now been integrated by Romain François in the bibtex package.
#'
#' @encoding utf8
#'
#' @param entry a \code{\link{bibentry}} object or a character vector of package 
#' names. If \code{NULL}, then the list of all installed packages is used.
#' @param file output Bibtex file. It can be specified as a filename (as a single 
#' character string), NULL for \code{stdout}, or a \code{link{connection}} object. 
#' If \code{file} is a character string, an extension '.bib' is appended if not 
#' already present.
#' @param append a logical that indicates that the Bibtex entries should be added
#' to the file. If \code{FALSE} (default), the file is overwritten.  
#' @param verbose a logical to toggle verbosity. If \code{file=NULL}, verbosity 
#' is forced off. 
#'
#' @return the list of Bibtex objects -- invisibly.
#' @author
#' Renaud Gaujoux, based on the function \code{Rpackages.bib} 
#' from Achim Zeileis (see \emph{References}).
#' 
#' @references 
#' \emph{[R] Creating bibtex file of all installed packages?}
#' Achim Zeileis. R-help mailing list. 
#' \url{https://stat.ethz.ch/pipermail/r-help/2009-December/222201.html}
#' 
#' @seealso \code{link{connection}}, \code{link{bibentry}}
#'  
#' @export
#' @examples
#' 
#' library(bibtex)
#' write.bib(c('bibtex', 'utils', 'tools'), file='references')
#' bibs <- read.bib('references.bib')
#' write.bib(bibs, 'references2.bib')
#' md5 <- tools::md5sum(c('references.bib', 'references2.bib'))
#' md5[1] == md5[2]
#' \dontshow{ stopifnot(md5[1] == md5[2]) }
#' 
#' # write to stdout()
#' write.bib(c('bibtex', 'utils', 'tools'), file=NULL)
#' 
write.bib <- function(entry=NULL, file="Rpackages.bib", append = FALSE, verbose = TRUE)
{
	# special handling of file=NULL: use stdout()
	if( is.null(file) ){
		file <- stdout()
		verbose <- FALSE
	}	
	## use all installed packages if nothing is specified
	if( is.null(entry) ){ 
		if( verbose ) message("Generating Bibtex entries for all installed packages ", appendLF=FALSE)
		entry <- unique(installed.packages()[,1])
		if( verbose ) message("[", length(entry), "]")
	}
	
	bibs <- 
			if( is(entry, 'bibentry') )	entry
			else if( is.character(entry) ){
				if( length(entry) == 0 ){
					if( verbose ) message("Empty package list: nothing to be done.")
					return(invisible())
				}
				
				pkgs <- entry
				bibs <- sapply(pkgs, function(x) try(citation(x)), simplify=FALSE)
				#bibs <- lapply(pkgs, function(x) try(toBibtex(citation(x))))
				n.installed <- length(bibs)
				
				## omit failed citation calls
				ok <- sapply(bibs, is, 'bibentry')
				pkgs <- pkgs[ok]
				bibs <- bibs[ok]
				n.converted <- sum(ok)
				
				## add bibtex keys to each entry
				pkgs <- lapply(seq_along(pkgs), function(i) if(length(bibs[[i]]) > 1)
								paste(pkgs[i], c('', 2:length(bibs[[i]])), sep = "") else pkgs[i])
				pkgs <- do.call("c", pkgs)
				bibs <- do.call("c", bibs)		
				# formatting function for bibtex keys:
				# names with special characters must be enclosed in {}, others not.
				as.bibkey <- function(x){
					i <- grep("[.]", x)
					if( length(i) > 0 )
						x[i] <- paste("{", x[i], "}", sep='')
					x
				}		
				#bibs <- mapply(function(b,k){ if( is.null(b$key) ) b$key <- as.bibkey(k); b}, bibs, pkgs, SIMPLIFY=FALSE)
				bibs <- mapply(function(b,k){ if( is.null(b$key) ) b$key <- k; b}, bibs, pkgs, SIMPLIFY=FALSE)
				bibs <- do.call("c", bibs)
				
				if(verbose) message("Converted ", n.converted, " of ", n.installed, " package citations to BibTeX")					
				bibs
			} else
				stop("Invalid argument `entry`: expected a bibentry object or a character vector of package names.")
	
	if( length(bibs) == 0 ){
		if( verbose ) message("Empty bibentry list: nothing to be done.")
		return(invisible())
	}
	
	## write everything to the .bib file
	fh <- if( is.character(file) ){
				if( !grepl("\\.bib$", file) ) # add .bib extension if necessary 
					file <- paste(file, '.bib', sep='')
				fh <- file(file, open = if(append) "a+" else "w+" )
				on.exit( if( isOpen(fh) ) close(fh) )
				fh
			} else if( is(file, 'connection') )
				file
			else
				stop("Invalid argument `file`: expected a filename, NULL, or a connection [", class(file), "]")
	
	if( !is(fh, 'connection') )
		stop("Invalid connection: ", fh)		
	file.desc <- summary(fh)['description']
	
	if( verbose ) message(if( append ) "Adding " else "Writing ", length(bibs) , " Bibtex entries ... ", appendLF=FALSE)
	writeLines(toBibtex(bibs), fh)
	if(verbose) message("OK\nResults written to file '", file.desc, "'")
	
	## return Bibtex items invisibly
	invisible(bibs)
}


makeFakeVignette <- function(template, out){
	l <- readLines(template)
	cat(c("\\documentclass[10pt]{article}"
		, l[grep("^%\\\\Vignette", l)]
		, "\\usepackage{url}\n\\usepackage[colorlinks]{hyperref}\n\n\\begin{document}\n\\end{document}")
	, file=out, sep="\n");
}
