#' @title Number line objects
#'
#' @description A set of functions to create and manipulate \code{number_line} objects.
#'
#' @details
#' A \code{number_line} object represents a series of real numbers on a number line.
#'
#' Visually, it's presented as the left (\code{l}) and right (\code{r}) points of the series. This may differ from start and end points.
#' The start point is the lowest number in the series, regardless of whether it's at the left or right point..
#'
#' The location of the start point - left or right, indicate if it's an \code{"increasing"} or \code{"decreasing"} series.
#' This is referred to as the \code{direction} of the \code{number_line} object.
#'
#' @param l Left point of the \code{number_line} object. Should be, or can be coerced to a \code{numeric} object
#' @param r Right point of the \code{number_line} object. Should be, or can be coerced to a \code{numeric} object
#' @param id Unique \code{numeric} ID. Providing this is optional
#' @param gid Unique \code{numeric} Group ID. Providing this is optional
#' @return \code{number_line} object
#'
#' @aliases number_line
#' @examples
#' library(lubridate)
#'
#' number_line(-100, 100); number_line(10, 11.2)
#'
#' # Other numeric based object classes are also compatible for numeric_line objects
#' number_line(dmy_hms("15/05/2019 13:15:07"), dmy_hms("15/05/2019 15:17:10"))
#'
#' # A warning is given if 'l' and 'r' have different classes. Consider if these need to be corrected
#' number_line(2, dmy("05/01/2019"))
#'
#' @export
number_line <- function(l, r, id = NULL, gid = NULL){
  er1 <- try(as.numeric(l), silent = TRUE)
  er2 <- try(as.numeric(r), silent = TRUE)
  er3 <- try(as.numeric(r) - as.numeric(l), silent = TRUE)

  if(missing(l) & missing(r) & missing(id) & missing(gid)) return(new("number_line"))
  if(!is.numeric(er1) | !is.numeric(er2) | !is.numeric(er3)) stop(paste("'l' or 'r' aren't compatible for a number_line object",sep=""))
  if(!(is.numeric(id) | is.null(id))) stop(paste("'id' must be numeric",sep=""))
  if(!(is.numeric(gid) | is.null(gid))) stop(paste("'gid' must be numeric",sep=""))

  if(all(class(l)!=class(r))) warning("'l' and 'r' have different classes. It may need to be reconciled")

  if(is.null(id) | any(!is.finite(id)) ) id <- 1:length(l)
  if(is.null(gid) | any(!is.finite(gid)) ) gid <- 1:length(l)
  nl <- methods::new("number_line", .Data = as.numeric(r) - as.numeric(l), start=l, id = id, gid = gid)
  return(nl)
}


#' @rdname number_line
#' @examples
#' # Convert numeric based objects to number_line objects
#' as.number_line(5.1); as.number_line(dmy("21/10/2019"))
#'
#' @export
as.number_line <- function(x){
  er1 <- suppressWarnings(try(as.numeric(x), silent = TRUE))
  er2 <- suppressWarnings(try(as.numeric(x) + 0, silent = TRUE))

  if(!is.numeric(er1) | !is.numeric(er2)) stop(paste("'x' can't be coerced to a number_line object",sep=""))

  if(all(!diyar::is.number_line(x))){
    x[!is.finite(as.numeric(x))] <- NA
    x <- methods::new("number_line", .Data = as.numeric(x-x), start= x, id = 1:length(x), gid = 1:length(x))
  }

  return(x)
}

#' @rdname number_line
#' @examples
#' # Test for number_line objects
#' a <- number_line(0, -100)
#' b <- number_line(dmy("25/04/2019"), dmy("01/01/2019"))
#' is.number_line(a); is.number_line(b)
#'
#' @export
is.number_line <- function(x) class(x)=="number_line"

#' @rdname number_line
#' @examples
#' # Structure of a number_line object
#' left_point(a); right_point(a); start_point(a); end_point(a)
#'
#' @export
left_point <- function(x){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  x@start
}

#' @rdname number_line
#' @export
right_point <- function(x){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  x@start + x@.Data
}

#' @rdname number_line
#' @export
start_point <- function(x){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  x <- diyar::reverse_number_line(x,"decreasing")
  x@start
}

#' @rdname number_line
#' @export
end_point <- function(x){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  x <- diyar::reverse_number_line(x,"decreasing")
  x@start + x@.Data
}

#' @rdname number_line
#' @export
number_line_width <- function(x){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  diyar::right_point(x) - diyar::left_point(x)
}

#' @rdname number_line
#' @param x \code{number_line} object
#' @param direction Type of \code{"number_line"} objects whose direction are to be reversed. Options are; \code{"increasing"}, \code{"decreasing"} or \code{"both"}.
#' @details
#' \code{reverse_number_line()} - reverses the direction of a \code{number_line} object. A reversed \code{number_line} object has its \code{l} and \code{r} points swapped but maintains the same width or length.
#' The \code{direction} argument determines which type of \code{number_line} objects will be reversed.
#' \code{number_line} objects with non-finite numeric starts or end points i.e. (\code{NA}, \code{NaN} and \code{Inf}) can't be reversed.
#' @examples
#' # Reverse number_line objects
#' reverse_number_line(number_line(dmy("25/04/2019"), dmy("01/01/2019")))
#' reverse_number_line(number_line(200,-100), "increasing")
#' reverse_number_line(number_line(200,-100), "decreasing")
#'
#' @export
reverse_number_line <- function(x, direction = "both"){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  if(!(length(direction)==1 & is.character(direction))) stop(paste("'direction' must be a character of length 1"))
  if(!tolower(direction) %in% c("increasing","decreasing","both") ) stop(paste("`direction` must be either 'increasing', 'decreasing', or 'both'"))
  f <- x

  if(tolower(direction) == "decreasing"){
    f@.Data <- ifelse(x@.Data <0 & is.finite(x@.Data), -x@.Data, x@.Data)
    c <- ifelse(x@.Data <0 & is.finite(x@.Data), x@.Data, 0)
  }else if(tolower(direction) == "increasing"){
    f@.Data <- ifelse(x@.Data >0 & is.finite(x@.Data), -x@.Data, x@.Data)
    c <- ifelse(x@.Data >0 & is.finite(x@.Data), x@.Data, 0)
  } else if(tolower(direction) == "both"){
    f@.Data <- ifelse(is.finite(x@.Data), -x@.Data, x@.Data)
    c <- ifelse(is.finite(x@.Data), x@.Data, 0)
  }

  f@start <- f@start + c

  return(f)
}

#' @rdname number_line
#' @details
#' \code{shift_number_line()} - a convenience function to shift a \code{number_line} object towards the positive or negative end of the number line.
#' @examples
#' # Shift number_line objects
#' number_line(5,6)
#' # Towards the positive end of the number line
#' shift_number_line(number_line(5,6), 2)
#' # Towards the negative end of the number line
#' shift_number_line(number_line(6,1), -2)
#'
#' @export
shift_number_line <- function(x, by=1){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  #if(!(is.finite(by) & length(by) ==1)) stop(paste("'by' must be a numeric based object of length 1",sep=""))

  by[!is.finite(by)] <- NA_real_
  n <- ifelse(is.finite(x@start) & is.finite(x@.Data),1,0)
  by <- by * n

  x@start <- x@start + by
  return(x)
}

#' @rdname number_line
#' @param point \code{"start"} or \code{"start"} point
#' @details
#' \code{expand_number_line()} - a convenience function to increase or decrease the width or length of a \code{number_line} object.
#' @examples
#' # Increase or reduce the width or length of a \code{number_line} object
#' c(number_line(3,6), number_line(6,3))
#' expand_number_line(c(number_line(3,6), number_line(6,3)), 2)
#' expand_number_line(c(number_line(3,6), number_line(6,3)), -1)
#' expand_number_line(c(number_line(3,6), number_line(6,3)), 2, "start")
#' expand_number_line(c(number_line(3,6), number_line(6,3)), -2, "end")
#'
#' @export
expand_number_line <- function(x, by=1, point ="both"){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  #if(!all(is.finite(by))) stop(paste("'by' must be a numeric based object",sep=""))
  if(!all(is.character(point)) | length(point)!=1) stop(paste("'point' must be a character object of length 1"))
  if(all(!tolower(point) %in% c("both","start","end"))) stop(paste("`point` must be either 'start','end' or 'both'"))

  by[!is.finite(by)] <- NA_real_
  n <- ifelse(x@.Data<0 & is.finite(x@.Data),-1,1)
  by <- by * n
  if(point == "both") x <- diyar::number_line(x@start - by, (x@start + x@.Data) + by, id = x@id, gid = x@gid)
  if(point == "start") x <- diyar::number_line(x@start - by, (x@start + x@.Data), id = x@id, gid = x@gid)
  if(point == "end") x <- diyar::number_line(x@start, (x@start + x@.Data) + by, id = x@id, gid = x@gid)

  return(x)
}

#' @rdname number_line
#' @details
#' \code{compress_number_line()} - Collapses overlapping \code{number_line} objects into a new \code{number_line} objects that covers the start and end points of the originals.
#' This results in duplicate \code{number_line} objects with start and end points of the new expanded \code{number_line} object.
#' See \code{\link{overlap}} for further details on overlapping \code{number_line} objects.
#' If a familiar (but unique) \code{id} is used when creating the \code{number_line} objects,
#' \code{compress_number_line()} can be a simple alternative to \code{\link{record_group}} or \code{\link{episode_group}}.
#'
#' @param method Method of overlap
#' @param collapse If \code{TRUE}, collapse the compressed results based on \code{method} of overlaps
#' @param deduplicate if \code{TRUE}, retains only one \code{number_line} object among duplicates
#' @examples
#' # Collapse number line objects
#' x <- c(number_line(10,10), number_line(10,20), number_line(5,30),  number_line(30,40))
#' compress_number_line(x, deduplicate = FALSE)
#' compress_number_line(x)
#' compress_number_line(x, collapse=TRUE)
#' compress_number_line(x, collapse=TRUE, method = "inbetween")
#'
#' @export

compress_number_line <- function(x, method = c("across","chain","aligns_start","aligns_end","inbetween"), collapse =FALSE, deduplicate = TRUE){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object"))
  if(!is.character(method)) stop(paste("'method' must be a character object"))
  if(!(is.logical(collapse) & is.logical(deduplicate) )) stop(paste("'collapse' and 'deduplicate' must be TRUE or FALSE"))
  if(all(!tolower(method) %in% c("across","chain","aligns_start","aligns_end","inbetween"))) stop(paste("`method` must be either 'across','chain','aligns_start','aligns_end' or 'inbetween'"))
  if(!(length(collapse) %in% c(1, length(x)))) stop(paste("length of 'collapse' must be 1 or the same as 'x'",sep=""))

  if(any(duplicated(x@id) | is.na(x@id))) x@id <- 1:length(x@id)
  x <- diyar::reverse_number_line(x, "decreasing")

  j <- 1
  t <- rep(0, length(x))
  if(length(collapse)==1) collapse <- rep(collapse, length(x))
  while (min(t) ==0 & j<=length(x)){
    l <- x[t==0][1]
    h <- (x@id == l@id | diyar::overlap(l, x, method=method)) & ifelse(collapse, TRUE, (t!=1))
    x[which(h)]@.Data <- as.numeric(max(x[which(h),]@start + x[which(h),]@.Data)) - as.numeric(min(x[which(h),]@start))
    x[which(h)]@start <- min(x[which(h),]@start)
    x[which(h)]@gid <- sort(x[which(h),])[1]@id
    t[which(h)] <- 1
    if(min(t)==1) break
    j <- j + 1
  }

  if(deduplicate) x <- unique.number_line(x)
  return(x)
}

#' @rdname number_line
#' @param by increment or decrement
#' @details
#' \code{number_line_sequence()} - a convenience function to convert a \code{number_line} object into a sequence of finite numbers. The sequence will also include the start and end points.
#' The direction of the sequence will correspond to that of the \code{number_line} object.
#' @examples
#' # Convert a number line object to its series of real numbers
#' number_line_sequence(number_line(1, 5))
#' number_line_sequence(number_line(5, 1), .5)
#' number_line_sequence(number_line(dmy("01/04/2019"), dmy("10/04/2019")), 1)
#'
#' # The length of the resulting vector will depend on the object class
#' number_line_sequence(number_line(dmy("01/04/2019"), dmy("04/04/2019")), 1.5)
#'
#' nl <- number_line(dmy_hms("01/04/2019 00:00:00"), dmy_hms("04/04/2019 00:00:00"))
#' head(number_line_sequence(nl, 1.5), 15)
#' d <- duration(1.5,"days")
#' number_line_sequence(nl, d)
#'
#' @export
number_line_sequence <- function(x, by=1){
  if(!diyar::is.number_line(x)) stop(paste("'x' is not a number_line object",sep=""))
  if(!(is.finite(by) & length(by) ==1)) stop(paste("'by' must be a numeric object of length 1",sep=""))

  if(!is.finite(x@start) | !is.finite(x@.Data)){
    s <- c(x@start, x@.Data)
  }else{
    by <- ifelse(x@.Data>0, abs(by), -abs(by))
    s <- unique(c(x@start, seq(x@start, x@start + x@.Data, by), x@start + x@.Data))
  }

  return(s)
}

