% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prodist.R
\name{prodist}
\alias{prodist}
\alias{prodist.lm}
\alias{prodist.glm}
\alias{prodist.negbin}
\alias{prodist.Arima}
\title{Extracting fitted or predicted probability distributions from models}
\usage{
prodist(object, ...)
}
\arguments{
\item{object}{A model object.}

\item{...}{Arguments passed on to methods, typically for calling the
underlying \code{\link[stats]{predict}} methods, e.g., \code{newdata} for
\code{\link[stats]{lm}} or \code{\link[stats]{glm}} objects or \code{n.ahead}
for \code{\link[stats]{arima}} objects.}
}
\value{
An object inheriting from \code{distribution}.
}
\description{
Generic function with methods for various model classes for extracting
fitted (in-sample) or predicted (out-of-sample) probability \code{distributions3}
objects.
}
\details{
To facilitate making probabilistic forecasts based on regression and time
series model objects, the function \code{prodist} extracts fitted or
predicted probability \code{distribution} objects. Currently, methods are
provided for objects fitted by \code{\link[stats]{lm}},
\code{\link[stats]{glm}}, and \code{\link[stats]{arima}} in base R as
well as \code{glm.nb} from the \pkg{MASS} package. All methods essentially
proceed in two steps: First, the standard \code{\link[stats]{predict}}
method for these model objects is used to compute fitted (in-sample, default)
or predicted (out-of-sample) distribution parameters. Typically, this includes
the mean plus further parameters describing scale, dispersion, shape, etc.).
Second, the \code{distributions} objects are set up using the generator
functions from \pkg{distributions3}.
}
\examples{

## Model: Linear regression
## Fit: lm
## Data: 1920s cars data
data("cars", package = "datasets")

## Stopping distance (ft) explained by speed (mph)
reg <- lm(dist ~ speed, data = cars)

## Extract fitted normal distributions (in-sample, with constant variance)
pd <- prodist(reg)
head(pd)

## Compute corresponding medians and 90\% interval
qd <- quantile(pd, c(0.05, 0.5, 0.95))
head(qd)

## Visualize observations with predicted quantiles
plot(dist ~ speed, data = cars)
matplot(cars$speed, qd, add = TRUE, type = "l", col = 2, lty = 1)


## Model: Poisson generalized linear model
## Fit: glm
## Data: FIFA 2018 World Cup data
data("FIFA2018", package = "distributions3")

## Number of goals per team explained by ability differences
poisreg <- glm(goals ~ difference, data = FIFA2018, family = poisson)
summary(poisreg)
## Interpretation: When the ratio of abilities increases by 1 percent,
## the expected number of goals increases by around 0.4 percent

## Predict fitted Poisson distributions for teams with equal ability (out-of-sample)
nd <- data.frame(difference = 0)
prodist(poisreg, newdata = nd)

## Extract fitted Poisson distributions (in-sample)
pd <- prodist(poisreg)
head(pd)

## Extract log-likelihood from model object
logLik(poisreg)

## Replicate log-likelihood via distributions object
sum(log_pdf(pd, FIFA2018$goals))
log_likelihood(pd, FIFA2018$goals)


## Model: Autoregressive integrated moving average model
## Fit: arima
## Data: Quarterly approval ratings of U.S. presidents (1945-1974)
data("presidents", package = "datasets")

## ARMA(2,1) model
arma21 <- arima(presidents, order = c(2, 0, 1))

## Extract predicted normal distributions for next two years
p <- prodist(arma21, n.ahead = 8)
p

## Compute median (= mean) forecast along with 80\% and 95\% interval
quantile(p, c(0.5, 0.1, 0.9, 0.025, 0.975))

}
\seealso{
\code{\link[stats]{predict}}, \code{\link[stats]{lm}},
\code{\link[stats]{glm}}, \code{\link[stats]{arima}}
}
\keyword{distribution}
