\name{depth}
\alias{depth}
\alias{depth.default}
\alias{depth.data.frame}
\alias{depth.matrix}
\alias{depth.list}
\concept{Tukey depth}
\concept{halfspace depth}
\concept{Simplicial depth}
\title{Depth calculation}
\description{
Computes the depth of a point with respect to a multivariate data set.
}
\usage{
depth(u, x, method = "Tukey", approx = FALSE, 
   eps = 1e-8, ndir = 1000)
}
\arguments{
  \item{u}{Numerical vector whose depth is to be calculated. Dimension has to be the same as that of the observations.}
  \item{x}{The data as a matrix, data frame or list. If it is a matrix or data frame, then each row is viewed as one multivariate observation. If it is a list, all components must be numerical vectors of equal length (coordinates of observations).}
  \item{method}{Character string which determines the depth function used. \code{method} can be "Tukey" (the default), "Liu" or "Oja".}
  \item{approx}{Logical. If dimension is 3, should an approximate Tukey depth be computed? Useful when sample size is large.}
  \item{eps}{Error tolerance to control the calculation.}
  \item{ndir}{Number of random directions used when Tukey depth is approximated.}
}
\details{ \code{method} "Tukey" refers to the Tukey or halfspace depth.
In dimension 2, exact calculation is based on Fortran code from Rousseeuw and Ruts (1996). In dimensions higher than 2, calculation utilises Fortran code from Struyf and Rousseeuw (1998). This yields exact calculation when dimension is 3 and \code{approx = FALSE}, and approximate calculation when dimension is higher than 3.

The Liu (or simplicial) depth is computed in dimension 2 only. Calculation is exact and based on Fortran code from Rousseeuw and Ruts (1996).

The Oja depth is derived from a location measure considered by Oja. If \eqn{p}
is the dimension and \eqn{n} the size of the data set, it is defined to be
\eqn{0.5(1+{n\choose
p}^{-1}\sum(\mathrm{Volume}(S(u,x[i_1,],\ldots,x[i_p,])))^{-1}}{0.5(1+Average(Volume(S(u,x[i_1,],\ldots,x[i_p,])))^(-1)},
where \eqn{S(args)} denotes the simplex generated by \eqn{args}, and sum and
average are taken over all \eqn{p}-plets \eqn{x[i_1,],\ldots,x[i_p,]} such that
\eqn{1 <= i_1 <\ldots. < i_p <= n}. Calculation is exact.
}
\value{
Returns the depth of multivariate point \code{u} with respect to data set \code{x}.
}
\references{Liu, R.Y., Parelius, J.M. and Singh, K. (1999), Multivariate analysis by data depth: Descriptive statistics, graphics and inference (with discussion), \emph{Ann. Statist.}, \bold{27}, 783--858.

Rousseuw, P.J. and Ruts, I. (1996), AS 307 : Bivariate location depth, \emph{Appl. Stat.-J. Roy. S. C},  \bold{45}, 516--526.

Rousseeuw, P.J. and Struyf, A. (1998), Computing location depth and regression depth in higher dimensions, \emph{Stat. Comput.}, \bold{8}, 193--203.

Zuo, Y. amd Serfling, R. (2000), General Notions of Statistical Depth Functions, \emph{Ann. Statist.},  \bold{28}, no. 2, 461--482.}

\author{Jean-Claude Masse and Jean-Francois Plante, based on Fortran code by Rousseeuw, Ruts and Struyf from University of Antwerp.}

\seealso{\code{\link{perspdepth}} and \code{\link{isodepth}} for depth graphics.}
\examples{## calculation of Tukey depth
data(starsCYG, package = "robustbase")
depth(mean(starsCYG), starsCYG)

## Tukey depth applied to a large bivariate data set.
set.seed(356)
x <- matrix(rnorm(9999), nc = 3)
depth(rep(0,3), x)

## approximate calculation much easier
depth(rep(0,3), x, approx = TRUE)
}

\keyword{multivariate}
\keyword{nonparametric}
\keyword{robust}
