## OBJECT TYPES ----------------------------------------------------------------

null <- NULL
na <- NA
boolean <- TRUE
number_random <- sample(1:1000, 1)
string_random <- paste0(sample(letters, 5), collapse = "")
vector_strings <- c("foo", "bar")
list_strings <- list("foo", "bar")
df <- mtcars
matrix <- as.matrix(mtcars)

# correct but irrelevant JSON file
fill <- list("a" = 1, "b" = 2)
json <- jsonlite::toJSON(fill)
madeup_json_file <- tempfile(fileext = ".json")
write(json, madeup_json_file)

## DRAW_BLOCKS -----------------------------------------------------------------

test_that("draw_blocks() warns of input errors", {
  expect_error(draw_blocks(type = null), "Invalid type parameter.")
  expect_error(draw_blocks(type = na), "Invalid type parameter.")
  expect_error(draw_blocks(type = boolean), "Invalid type parameter.")

  expect_error(draw_blocks(type = number_random), "Invalid type parameter.")
  expect_error(draw_blocks(type = string_random), "Invalid type parameter.")
  expect_error(draw_blocks(type = df), "Invalid type parameter.")
  expect_error(draw_blocks(type = matrix), "Invalid type parameter.")
  expect_error(draw_blocks(type = vector_strings), "Invalid type parameter.")
  expect_error(draw_blocks(type = list_strings), "Invalid type parameter.")

  expect_error(draw_blocks(type = "sync", object = null), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = na), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = boolean), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = number_random), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = df), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = matrix), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = vector_strings), "Object parameter not pointing to valid response object.")
  expect_error(draw_blocks(type = "sync", object = list_strings), "Object parameter not pointing to valid response object.")

  expect_error(draw_blocks(type = "async", object = realjson, prefix = number_random), "Invalid prefix parameter.")
  expect_error(draw_blocks(type = "async", object = realjson, prefix = df), "Invalid prefix parameter.")
  expect_error(draw_blocks(type = "async", object = realjson, prefix = matrix), "Invalid prefix parameter.")
  expect_error(draw_blocks(type = "async", object = realjson, prefix = vector_strings), "Invalid prefix parameter.")
  expect_error(draw_blocks(type = "async", object = realjson, prefix = list_strings), "Invalid prefix parameter.")

  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "test", dir = null),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "test", dir = na),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "test", dir = boolean),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "test", dir = number_random),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = df),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = matrix),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = vector_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = list_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )

  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = null),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = na),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = boolean),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = string_random),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = vector_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = list_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = df),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = matrix),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = null),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = na),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = boolean),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = string_random),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = vector_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = list_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = df),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = matrix),
    "Invalid linewd parameter. Must be a single number."
  )

  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = null),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = na),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = boolean),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = string_random),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = vector_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = list_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = df),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = matrix),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = null),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = na),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = boolean),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = string_random),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = vector_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = list_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = df),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_blocks(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = matrix),
    "Invalid fontsize parameter. Must be a single number."
  )
})

test_that("draw_blocks() produces a correctly named png file", {
  skip_on_cran()
  skip_on_ci()
  skip_if_not_installed("grDevices")
  skip_if_not_installed("magick")
  realjson <- testthat::test_path("examples", "sample_v1.json")
  draw_blocks(type = "async", object = realjson, dir = tempdir())
  parsed <- jsonlite::fromJSON(realjson)
  pages <- parsed[["pages"]][["pageNumber"]]
  expected_filepaths <- character()
  stem <- substr(basename(realjson), 1, nchar(basename(realjson)) - 5)
  for (i in pages) {
    fname <- glue::glue("{stem}_page{i}_blocks.png")
    fpath <- file.path(tempdir(), fname)
    expected_filepaths <- c(expected_filepaths, fpath)
  }
  for (j in 1:length(expected_filepaths)) {
    expect_true(file.exists(expected_filepaths[j]))
  }
  unlink(expected_filepaths, force = TRUE)
})

## DRAW_PARAGRAPHS -------------------------------------------------------------

test_that("draw_paragraphs() warns of input errors", {
  expect_error(draw_paragraphs(type = null), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = na), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = boolean), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = number_random), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = string_random), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = df), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = matrix), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = vector_strings), "Invalid type parameter.")
  expect_error(draw_paragraphs(type = list_strings), "Invalid type parameter.")

  expect_error(draw_paragraphs(type = "sync", object = null), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = na), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = boolean), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = number_random), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = df), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = matrix), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = vector_strings), "Object parameter not pointing to valid response object.")
  expect_error(draw_paragraphs(type = "sync", object = list_strings), "Object parameter not pointing to valid response object.")

  realjson <- testthat::test_path("examples", "sample_v1.json")

  expect_error(draw_paragraphs(type = "async", object = realjson, prefix = number_random), "Invalid prefix parameter.")
  expect_error(draw_paragraphs(type = "async", object = realjson, prefix = df), "Invalid prefix parameter.")
  expect_error(draw_paragraphs(type = "async", object = realjson, prefix = matrix), "Invalid prefix parameter.")
  expect_error(draw_paragraphs(type = "async", object = realjson, prefix = vector_strings), "Invalid prefix parameter.")
  expect_error(draw_paragraphs(type = "async", object = realjson, prefix = list_strings), "Invalid prefix parameter.")

  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "test", dir = null),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "test", dir = na),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "test", dir = boolean),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "test", dir = number_random),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = df),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = matrix),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = vector_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = list_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )

  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = null),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = na),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = boolean),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = string_random),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = vector_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = list_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = df),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = matrix),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = null),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = na),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = boolean),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = string_random),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = vector_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = list_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = df),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = matrix),
    "Invalid linewd parameter. Must be a single number."
  )

  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = null),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = na),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = boolean),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = string_random),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = vector_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = list_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = df),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = matrix),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = null),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = na),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = boolean),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = string_random),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = vector_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = list_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = df),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_paragraphs(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = matrix),
    "Invalid fontsize parameter. Must be a single number."
  )
})

test_that("draw_paragraphs() produces a correctly named new png file", {
  skip_on_cran()
  skip_on_ci()
  skip_if_not_installed("grDevices")
  skip_if_not_installed("magick")
  realjson <- testthat::test_path("examples", "sample_v1.json")
  draw_paragraphs(type = "async", object = realjson, dir = tempdir())
  parsed <- jsonlite::fromJSON(realjson)
  pages <- parsed[["pages"]][["pageNumber"]]
  expected_filepaths <- character()
  stem <- substr(basename(realjson), 1, nchar(basename(realjson)) - 5)
  for (i in pages) {
    fname <- glue::glue("{stem}_page{i}_paragraphs.png")
    fpath <- file.path(tempdir(), fname)
    expected_filepaths <- c(expected_filepaths, fpath)
  }
  for (j in 1:length(expected_filepaths)) {
    expect_true(file.exists(expected_filepaths[j]))
  }
  unlink(expected_filepaths, force = TRUE)
})

## DRAW_LINES ------------------------------------------------------------------

test_that("draw_lines() warns of input errors", {
  expect_error(draw_lines(type = null), "Invalid type parameter.")
  expect_error(draw_lines(type = na), "Invalid type parameter.")
  expect_error(draw_lines(type = boolean), "Invalid type parameter.")
  expect_error(draw_lines(type = number_random), "Invalid type parameter.")
  expect_error(draw_lines(type = string_random), "Invalid type parameter.")
  expect_error(draw_lines(type = df), "Invalid type parameter.")
  expect_error(draw_lines(type = matrix), "Invalid type parameter.")
  expect_error(draw_lines(type = vector_strings), "Invalid type parameter.")
  expect_error(draw_lines(type = list_strings), "Invalid type parameter.")

  expect_error(draw_lines(type = "sync", object = null), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = na), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = boolean), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = number_random), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = df), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = matrix), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = vector_strings), "Object parameter not pointing to valid response object.")
  expect_error(draw_lines(type = "sync", object = list_strings), "Object parameter not pointing to valid response object.")

  realjson <- testthat::test_path("examples", "sample_v1.json")

  expect_error(draw_lines(type = "async", object = realjson, prefix = number_random), "Invalid prefix parameter.")
  expect_error(draw_lines(type = "async", object = realjson, prefix = df), "Invalid prefix parameter.")
  expect_error(draw_lines(type = "async", object = realjson, prefix = matrix), "Invalid prefix parameter.")
  expect_error(draw_lines(type = "async", object = realjson, prefix = vector_strings), "Invalid prefix parameter.")
  expect_error(draw_lines(type = "async", object = realjson, prefix = list_strings), "Invalid prefix parameter.")

  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "test", dir = null),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "test", dir = na),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "test", dir = boolean),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "test", dir = number_random),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = df),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = matrix),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = vector_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = list_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )

  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = null),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = na),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = boolean),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = string_random),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = vector_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = list_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = df),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = matrix),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = null),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = na),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = boolean),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = string_random),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = vector_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = list_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = df),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = matrix),
    "Invalid linewd parameter. Must be a single number."
  )

  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = null),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = na),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = boolean),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = string_random),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = vector_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = list_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = df),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = matrix),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = null),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = na),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = boolean),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = string_random),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = vector_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = list_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = df),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_lines(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = matrix),
    "Invalid fontsize parameter. Must be a single number."
  )
})

test_that("draw_lines() produces a correctly named new png file", {
  skip_on_cran()
  skip_on_ci()
  skip_if_not_installed("grDevices")
  skip_if_not_installed("magick")
  realjson <- testthat::test_path("examples", "sample_v1.json")
  draw_lines(type = "async", object = realjson, dir = tempdir())
  parsed <- jsonlite::fromJSON(realjson)
  pages <- parsed[["pages"]][["pageNumber"]]
  expected_filepaths <- character()
  stem <- substr(basename(realjson), 1, nchar(basename(realjson)) - 5)
  for (i in pages) {
    fname <- glue::glue("{stem}_page{i}_lines.png")
    fpath <- file.path(tempdir(), fname)
    expected_filepaths <- c(expected_filepaths, fpath)
  }
  for (j in 1:length(expected_filepaths)) {
    expect_true(file.exists(expected_filepaths[j]))
  }
  unlink(expected_filepaths, force = TRUE)
})

## DRAW_TOKENS -----------------------------------------------------------------

test_that("draw_tokens() warns of input errors", {
  expect_error(draw_tokens(type = null), "Invalid type parameter.")
  expect_error(draw_tokens(type = na), "Invalid type parameter.")
  expect_error(draw_tokens(type = boolean), "Invalid type parameter.")
  expect_error(draw_tokens(type = number_random), "Invalid type parameter.")
  expect_error(draw_tokens(type = string_random), "Invalid type parameter.")
  expect_error(draw_tokens(type = df), "Invalid type parameter.")
  expect_error(draw_tokens(type = matrix), "Invalid type parameter.")
  expect_error(draw_tokens(type = vector_strings), "Invalid type parameter.")
  expect_error(draw_tokens(type = list_strings), "Invalid type parameter.")

  expect_error(draw_tokens(type = "sync", object = null), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = na), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = boolean), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = number_random), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = df), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = matrix), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = vector_strings), "Object parameter not pointing to valid response object.")
  expect_error(draw_tokens(type = "sync", object = list_strings), "Object parameter not pointing to valid response object.")

  realjson <- testthat::test_path("examples", "sample_v1.json")

  expect_error(draw_tokens(type = "async", object = realjson, prefix = number_random), "Invalid prefix parameter.")
  expect_error(draw_tokens(type = "async", object = realjson, prefix = df), "Invalid prefix parameter.")
  expect_error(draw_tokens(type = "async", object = realjson, prefix = matrix), "Invalid prefix parameter.")
  expect_error(draw_tokens(type = "async", object = realjson, prefix = vector_strings), "Invalid prefix parameter.")
  expect_error(draw_tokens(type = "async", object = realjson, prefix = list_strings), "Invalid prefix parameter.")

  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "test", dir = null),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "test", dir = na),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "test", dir = boolean),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "test", dir = number_random),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = df),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = matrix),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = vector_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = list_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )

  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = null),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = na),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = boolean),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = string_random),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = vector_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = list_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = df),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = matrix),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = null),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = na),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = boolean),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = string_random),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = vector_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = list_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = df),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = matrix),
    "Invalid linewd parameter. Must be a single number."
  )

  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = null),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = na),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = boolean),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = string_random),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = vector_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = list_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = df),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = matrix),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = null),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = na),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = boolean),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = string_random),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = vector_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = list_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = df),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_tokens(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = matrix),
    "Invalid fontsize parameter. Must be a single number."
  )
})

test_that("draw_tokens() produces a correctly named new png file", {
  skip_on_cran()
  skip_on_ci()
  skip_if_not_installed("grDevices")
  skip_if_not_installed("magick")
  realjson <- testthat::test_path("examples", "sample_v1.json")
  draw_tokens(type = "async", object = realjson, dir = tempdir())
  parsed <- jsonlite::fromJSON(realjson)
  pages <- parsed[["pages"]][["pageNumber"]]
  expected_filepaths <- character()
  stem <- substr(basename(realjson), 1, nchar(basename(realjson)) - 5)
  for (i in pages) {
    fname <- glue::glue("{stem}_page{i}_tokens.png")
    fpath <- file.path(tempdir(), fname)
    expected_filepaths <- c(expected_filepaths, fpath)
  }
  for (j in 1:length(expected_filepaths)) {
    expect_true(file.exists(expected_filepaths[j]))
  }
  unlink(expected_filepaths, force = TRUE)
})

## DRAW ENTITIES ----------------------------------------

test_that("draw_entities() warns of input errors", {
  expect_error(draw_entities(type = null), "Invalid type parameter.")
  expect_error(draw_entities(type = na), "Invalid type parameter.")
  expect_error(draw_entities(type = boolean), "Invalid type parameter.")
  expect_error(draw_entities(type = number_random), "Invalid type parameter.")
  expect_error(draw_entities(type = string_random), "Invalid type parameter.")
  expect_error(draw_entities(type = df), "Invalid type parameter.")
  expect_error(draw_entities(type = matrix), "Invalid type parameter.")
  expect_error(draw_entities(type = vector_strings), "Invalid type parameter.")
  expect_error(draw_entities(type = list_strings), "Invalid type parameter.")

  expect_error(draw_entities(type = "sync", object = null), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = na), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = boolean), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = number_random), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = df), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = matrix), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = vector_strings), "Object parameter not pointing to valid response object.")
  expect_error(draw_entities(type = "sync", object = list_strings), "Object parameter not pointing to valid response object.")

  realjson <- testthat::test_path("examples", "sample_v1.json")

  expect_error(draw_entities(type = "async", object = realjson, prefix = number_random), "Invalid prefix parameter.")
  expect_error(draw_entities(type = "async", object = realjson, prefix = df), "Invalid prefix parameter.")
  expect_error(draw_entities(type = "async", object = realjson, prefix = matrix), "Invalid prefix parameter.")
  expect_error(draw_entities(type = "async", object = realjson, prefix = vector_strings), "Invalid prefix parameter.")
  expect_error(draw_entities(type = "async", object = realjson, prefix = list_strings), "Invalid prefix parameter.")

  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "test", dir = null),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "test", dir = na),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "test", dir = boolean),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "test", dir = number_random),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = df),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = matrix),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = vector_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = list_strings),
    "Invalid dir parameter. Must be a valid folder path."
  )

  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = null),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = na),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = boolean),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = string_random),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = vector_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = list_strings),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = df),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linecol = matrix),
    "Invalid linecol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = null),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = na),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = boolean),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = string_random),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = vector_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = list_strings),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = df),
    "Invalid linewd parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", linewd = matrix),
    "Invalid linewd parameter. Must be a single number."
  )

  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = null),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = na),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = boolean),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = string_random),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = vector_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = list_strings),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = df),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontcol = matrix),
    "Invalid fontcol parameter. Must be a single valid colour representation."
  )

  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = null),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = na),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = boolean),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = string_random),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = vector_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = list_strings),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = df),
    "Invalid fontsize parameter. Must be a single number."
  )
  expect_error(
    draw_entities(type = "async", object = realjson, prefix = "myfile", dir = ".", fontsize = matrix),
    "Invalid fontsize parameter. Must be a single number."
  )
})

test_that("draw_entities() produces a correctly named new png file", {
  skip_on_cran()
  skip_on_ci()
  skip_if_not_installed("grDevices")
  skip_if_not_installed("magick")
  realjson <- testthat::test_path("examples", "table_form_parsed.json")
  draw_entities(type = "async", object = realjson, dir = tempdir())
  parsed <- jsonlite::fromJSON(realjson)
  pages <- parsed[["pages"]][["pageNumber"]]
  expected_filepaths <- character()
  stem <- substr(basename(realjson), 1, nchar(basename(realjson)) - 5)
  for (i in pages) {
    fname <- glue::glue("{stem}_page{i}_entities.png")
    fpath <- file.path(tempdir(), fname)
    expected_filepaths <- c(expected_filepaths, fpath)
  }
  for (j in 1:length(expected_filepaths)) {
    expect_true(file.exists(expected_filepaths[j]))
  }
  unlink(expected_filepaths, force = TRUE)
})

unlink(madeup_json_file, force = TRUE)
