% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core_estimate_wrapper.R
\name{estimate_model_ids}
\alias{estimate_model_ids}
\title{Fit Multiple Individuals and Save Results}
\usage{
estimate_model_ids(
  drift_dm_obj,
  obs_data_ids,
  lower,
  upper,
  fit_procedure_name,
  fit_path,
  fit_dir = "drift_dm_fits",
  folder_name = fit_procedure_name,
  seed = NULL,
  force_refit = FALSE,
  progress = 2,
  start_vals = NULL,
  ...
)
}
\arguments{
\item{drift_dm_obj}{an object inheriting from \link{drift_dm} that will
be estimated for each individual in \code{obs_data_ids}.}

\item{obs_data_ids}{data.frame, see
\link{obs_data}. An additional column \code{ID} necessary, to
identify a single individual.}

\item{lower, upper}{numeric vectors or lists, providing the parameter space,
see \link{estimate_model}.}

\item{fit_procedure_name}{character, providing a name of the fitting
procedure. This name will be stored in \code{drift_dm_fit_info.rds} to identify
the fitting procedure, see also \link{load_fits_ids}.}

\item{fit_path}{character, a path, pointing to the location where all fits
shall be stored (i.e., \code{fit_dir} will be created in this location). From the
user perspective, the path will likely be identical to the current working
directory.}

\item{fit_dir}{character, a directory where (multiple) fitting procedures
can be stored. If the directory does not exist yet, it will be created
via \code{base::create.dir(fit_dir, recursive = TRUE)} in the location
provided by \code{fit_path}. Default is \code{"drift_dm_fits"}.}

\item{folder_name}{character, a folder name for storing all the individual
model fits. This variable should just state the name, and should not be
a path. Per default \code{folder_name} is identical to \code{fit_procedure_name}.}

\item{seed}{numeric, a seed to make the fitting procedure reproducable
(only relevant for differential evolution, see \link{estimate_model}).
Default is \code{NULL} which means no seed.}

\item{force_refit}{logical, if \code{TRUE} each individual of a fitting routine will
be fitted once more. Default is \code{FALSE} which indicates that saved files}

\item{progress}{numerical, indicating if and how progress shall be displayed.
If 0, no progress is shown. If 1, the currently fitted individual is printed
out. If 2, a progressbar is shown. Default is 2.}

\item{start_vals}{optional data.frame, providing values to be set
before calling \link{estimate_model}. Can be used to control the
starting values for each individual when calling Nelder-Mead. Note that this
will only have an effect if DEoptim is not used (i.e., when setting
\code{use_de_optim = FALSE}; see \link{estimate_model}). The data.frame
must provide a column \code{ID} whose entries match the \code{ID} column
in \code{obs_data_ids}, as well as a column for each parameter of the model
matching with \code{coef(drift_dm_obj, select_unique = TRUE)}.}

\item{...}{additional arguments passed down to \link{estimate_model}.}
}
\value{
nothing (\code{NULL}; invisibly)
}
\description{
Provides a wrapper around \link{estimate_model} to fit multiple
individuals. Each individual will be stored in a folder. This folder will
also contain a file \code{drift_dm_fit_info.rds}, containing the main arguments
of the function call. One call to this function is considered a
"fit procedure". Fit procedures can be loaded via \link{load_fits_ids}.
}
\details{
Examples and more information can be found here
\code{vignette("use_ddm_models", "dRiftDM")}.

When developing the fitting routine we had three levels of files/folders
in mind:
\itemize{
\item In a directory/folder named \code{fit_dir} multiple fitting routines can be
stored (default is "drift_dm_fits")
\item Each fitting routine has its own folder with a name as given by
\code{folder_name} (e.g., "ulrich_flanker", "ulrich_simon", ...)
\item Within each folder, a file called
\code{drift_dm_fit_info.rds} contains the main information about the function
call. That is, the time when last modifying/calling a fitting routine, the
\code{lower} and \code{upper} parameter boundaries, the \code{drift_dm_object} that was
fitted to each individual, the original data set \code{obs_data_ids}, and
the identifier \code{fit_procedure_name}. In the same folder each individual
has its own \verb{<individual>.rds} file containing the modified
\code{drift_dm_object}.
}
}
\examples{
# We'll provide a somewhat unrealistic example, trimmed for speed.
# In practice, users likely employ more complex models and more individuals.
# However, a more realistic example would take minutes (and maybe even hours)
# and is therefore not suitable for an example.

# Fit the Ratcliff model to synthetic data --------------------------------
# get the model (pre-built by dRiftDM)
model <- ratcliff_dm(t_max = 2.0, dx = .005, dt = .005)

# define an upper and lower boundary for the parameter space
lower <- c(muc = 1, b = 0.2, non_dec = 0.1)
upper <- c(muc = 7, b = 1.0, non_dec = 0.6)

# simulate synthetic data for demonstration purpose
synth_data_prms <- simulate_data(
  model,
  n = 100, k = 2, lower = lower, upper = upper, seed = 1
)
synth_data <- synth_data_prms$synth_data

# finally, call the fit procedure. To increase speed, we'll use the
# Nelder-Mead minimization routine. Note: We'll save the fits in tempdir()
# to avoid writing to a user's file directory without explicit permission.
estimate_model_ids(
  drift_dm_obj = model, # which model (the Ratcliff model)
  obs_data_ids = synth_data, # which data (the synthetic data set)
  lower = lower, # the lower and upper parameter/search space
  upper = upper,
  fit_procedure_name = "example", # a label for the fit procedure
  fit_path = tempdir(), # temporary directory (replace, e.g., with getwd())
  use_nmkb = TRUE, # use Nelder-Mead (fast, but less robust)
  use_de_optim = FALSE # and not differential evolution
)

\dontshow{
unlink(file.path(tempdir(), "drift_dm_fits"), recursive = TRUE)
}

}
\seealso{
\link{load_fits_ids}
}
