#' Scoring
#'
#' \code{score_transfer} is  for transfer woe to score.
#' @param model A data frame with x and target.
#' @param tbl_woe a data.frame with woe variables.
#' @param a  Base line of score.
#' @param b  Numeric.Increased scores from doubling Odds.
#' @param save_data Logical, save results in locally specified folder. Default is TRUE
#' @param file_name The name for periodically saved score file. Default is "dat_score".
#' @param dir_path  The path for periodically saved score file.  Default is "./data"
#' @return  A data.frame with variables which values transfered to score.
#' @examples
#' # dataset spliting
#' sub = cv_split(UCICreditCard, k = 30)[[1]]
#' dat = UCICreditCard[sub,]
#' #rename the target variable
#' dat = re_name(dat, "default.payment.next.month", "target")
#' dat = cleaning_data(dat, target = "target", obs_id = "ID", 
#' occur_time = "apply_date", miss_values =  list("", -1))
#' #train_ test pliting
#' train_test <- train_test_split(dat, split_type = "OOT", prop = 0.7,
#'                                 occur_time = "apply_date")
#' dat_train = train_test$train
#' dat_test = train_test$test
#' #get breaks of all predictive variables
#' x_list = c("PAY_0", "LIMIT_BAL", "PAY_AMT5", "EDUCATION", "PAY_3", "PAY_2")
#' breaks_list <- get_breaks_all(dat = dat_train, target = "target",
#'                               x_list = x_list, occur_time = "apply_date", ex_cols = "ID", 
#' save_data = FALSE, note = FALSE)
#' #woe transforming
#' train_woe = woe_trans_all(dat = dat_train,
#'                           target = "target",
#'                           breaks_list = breaks_list,
#'                           woe_name = FALSE)
#' test_woe = woe_trans_all(dat = dat_test,
#'                        target = "target",
#'                          breaks_list = breaks_list,
#'                          note = FALSE)
#' Formula = as.formula(paste("target", paste(x_list, collapse = ' + '), sep = ' ~ '))
#' set.seed(46)
#' lr_model = glm(Formula, data = train_woe[, c("target", x_list)], family = binomial(logit))
#' #get LR coefficient
#' dt_imp_LR = get_logistic_coef(lg_model = lr_model, save_data = FALSE)
#' bins_table = get_bins_table_all(dat = dat_train, target = "target",
#'                                occur_time = "apply_date",
#'                                 x_list = x_list,
#'                                breaks_list = breaks_list, note = FALSE)
#' #score card
#' LR_score_card <- get_score_card(lg_model = lr_model, bins_table, target = "target")
#' #scoring
#' train_pred = dat_train[, c("ID", "apply_date", "target")]
#' test_pred = dat_test[, c("ID", "apply_date", "target")]
#' train_pred$pred_LR = score_transfer(model = lr_model,
#'                                                     tbl_woe = train_woe,
#'                                                     save_data = FALSE)[, "score"]
#' 
#' test_pred$pred_LR = score_transfer(model = lr_model,
#' tbl_woe = test_woe, save_data = FALSE)[, "score"]
#' @export

score_transfer <- function(model, tbl_woe, a = 600, b = 50,
                           file_name =NULL, dir_path = tempdir(),
                           save_data = FALSE) {
    coef = model$coefficients
    glm_vars <- names(coef)[-1]
    A = a
    B = b / log(2)
    base_score = A - B * coef[1]

    tbl_woe = tbl_woe[c( glm_vars)]
    score_name = c()
    for (i in glm_vars) {
        tbl_woe[i] = (-1) * B * tbl_woe[i] * coef[i]
        score_name[i] = gsub("_woe", "_score", i)
    }
    names(tbl_woe) = score_name
    tbl_woe$score <- apply(tbl_woe[1:length(tbl_woe)], MARGIN = 1, function(x) sum(x))
    tbl_woe$score <- round(tbl_woe$score + base_score, 2)
    if (save_data) {
        dir_path = ifelse(is.null(dir_path) || !is.character(dir_path) || !grepl('.|/|:', dir_path),
                      tempdir(), dir_path)
        if (!dir.exists(dir_path)) dir.create(dir_path)
        if (!is.character(file_name)) file_name = NULL

        save_dt(tbl_woe, file_name = ifelse(is.null(file_name), "dat.score" ,paste(file_name, "dat.score", sep = ".")), dir_path = dir_path, note = FALSE)
    }
    tbl_woe
}

#' Score Card
#'
#' \code{get_score_card} is  for generating a stardard scorecard
#' @param lg_model An object of glm model.
#' @param target The name of target variable.
#' @param bins_table a data.frame generated by \code{\link{get_bins_table}}
#' @param a  Base line of score.
#' @param b  Numeric.Increased scores from doubling Odds.
#' @param save_data Logical, save results in locally specified folder. Default is TRUE
#' @param file_name  The name for periodically saved scorecard file. Default is "LR_Score_Card".
#' @param dir_path  The path for periodically saved scorecard file. Default is "./model"
#' @return  scorecard
#' @export
#' @examples
#' # dataset spliting
#' sub = cv_split(UCICreditCard, k = 30)[[1]]
#' dat = UCICreditCard[sub,]
#' #rename the target variable
#' dat = re_name(dat, "default.payment.next.month", "target")
#' dat = cleaning_data(dat, target = "target", obs_id = "ID", 
#' occur_time = "apply_date", miss_values =  list("", -1))
#' #train_ test pliting
#' train_test <- train_test_split(dat, split_type = "OOT", prop = 0.7,
#'                                 occur_time = "apply_date")
#' dat_train = train_test$train
#' dat_test = train_test$test
#' #get breaks of all predictive variables
#' x_list = c("PAY_0", "LIMIT_BAL", "PAY_AMT5", "EDUCATION", "PAY_3", "PAY_2")
#' breaks_list <- get_breaks_all(dat = dat_train, target = "target",
#'                               x_list = x_list, occur_time = "apply_date", ex_cols = "ID", 
#' save_data = FALSE, note = FALSE)
#' #woe transforming
#' train_woe = woe_trans_all(dat = dat_train,
#'                           target = "target",
#'                           breaks_list = breaks_list,
#'                           woe_name = FALSE)
#' test_woe = woe_trans_all(dat = dat_test,
#'                        target = "target",
#'                          breaks_list = breaks_list,
#'                          note = FALSE)
#' Formula = as.formula(paste("target", paste(x_list, collapse = ' + '), sep = ' ~ '))
#' set.seed(46)
#' lr_model = glm(Formula, data = train_woe[, c("target", x_list)], family = binomial(logit))
#' #get LR coefficient
#' dt_imp_LR = get_logistic_coef(lg_model = lr_model, save_data = FALSE)
#' bins_table = get_bins_table_all(dat = dat_train, target = "target",
#'                                occur_time = "apply_date",
#'                                 x_list = x_list,
#'                                breaks_list = breaks_list, note = FALSE)
#' #score card
#' LR_score_card <- get_score_card(lg_model = lr_model, bins_table, target = "target")
#' #scoring
#' train_pred = dat_train[, c("ID", "apply_date", "target")]
#' test_pred = dat_test[, c("ID", "apply_date", "target")]
#' train_pred$pred_LR = score_transfer(model = lr_model,
#'                                                     tbl_woe = train_woe,
#'                                                     save_data = FALSE)[, "score"]
#' 
#' test_pred$pred_LR = score_transfer(model = lr_model,
#' tbl_woe = test_woe, save_data = FALSE)[, "score"]

get_score_card <- function(lg_model, target, bins_table, a = 600, b = 50,
                           file_name = NULL, dir_path = tempdir(),
                           save_data = FALSE) {
    coef = lg_model$coefficients
    glm_vars <- gsub("_woe", "", names(coef))
    names(coef) <- glm_vars
    A = a
    B = b / log(2)
    base_score = A - B * coef[1]
    dt_score_card <- bins_table[which(as.character(bins_table[, "Feature"]) %in% glm_vars),
                                c("Feature", "cuts", "bins", "woe")]

    for (i in glm_vars) {
        dt_score_card[which(as.character(dt_score_card[, "Feature"]) == i), "coefficient"] = round( coef[i], 5)
    }

    for (i in glm_vars) {
        dt_score_card[which(as.character(dt_score_card[, "Feature"]) == i), "score"] =
        round((-1) * B * as.numeric(dt_score_card[which(as.character(dt_score_card[, "Feature"]) == i), "woe"] )* coef[i], 2)
    }
    Intercept = c("Intercept", "", "", "", round(coef[1], 5), paste("Base:", round(base_score)))
    dt_score_card = rbind(Intercept, dt_score_card)
    if (save_data) {
        dir_path = ifelse(is.null(dir_path) || !is.character(dir_path) || !grepl('.|/|:', dir_path),
                      tempdir(), dir_path)
        if (!dir.exists(dir_path)) dir.create(dir_path)
        if (!is.character(file_name)) file_name = NULL
        save_dt(dt_score_card, file_name = ifelse(is.null(file_name), "scorecard", paste(file_name, "scorecard", sep = ".")), dir_path = dir_path, note = FALSE)
    }
    dt_score_card
}



#' get logistic coef
#'
#' \code{get_logistic_coef} is  for geting logistic coefficient.
#' @param lg_model  An object of logistic model.
#' @param save_data Logical, save the result or not. Default is TRUE.
#' @param file_name  The name for periodically saved coefficient file.  Default is "LR_coef".
#' @param dir_path  The Path for periodically saved coefficient file. Default is "./model".
#' @return  A data.frame with logistic coefficients.
#' @examples
#' # dataset spliting
#' sub = cv_split(UCICreditCard, k = 30)[[1]]
#' dat = UCICreditCard[sub,]
#' #rename the target variable
#' dat = re_name(dat, "default.payment.next.month", "target")
#' dat = cleaning_data(dat, target = "target", obs_id = "ID", 
#' occur_time = "apply_date", miss_values =  list("", -1))
#' #train_ test pliting
#' train_test <- train_test_split(dat, split_type = "OOT", prop = 0.7,
#'                                 occur_time = "apply_date")
#' dat_train = train_test$train
#' dat_test = train_test$test
#' #get breaks of all predictive variables
#' x_list = c("PAY_0", "LIMIT_BAL", "PAY_AMT5", "EDUCATION", "PAY_3", "PAY_2")
#' breaks_list <- get_breaks_all(dat = dat_train, target = "target",
#'                               x_list = x_list, occur_time = "apply_date", ex_cols = "ID", 
#' save_data = FALSE, note = FALSE)
#' #woe transforming
#' train_woe = woe_trans_all(dat = dat_train,
#'                           target = "target",
#'                           breaks_list = breaks_list,
#'                           woe_name = FALSE)
#' test_woe = woe_trans_all(dat = dat_test,
#'                        target = "target",
#'                          breaks_list = breaks_list,
#'                          note = FALSE)
#' Formula = as.formula(paste("target", paste(x_list, collapse = ' + '), sep = ' ~ '))
#' set.seed(46)
#' lr_model = glm(Formula, data = train_woe[, c("target", x_list)], family = binomial(logit))
#' #get LR coefficient
#' dt_imp_LR = get_logistic_coef(lg_model = lr_model, save_data = FALSE)
#' bins_table = get_bins_table_all(dat = dat_train, target = "target",
#'                                occur_time = "apply_date",
#'                                 x_list = x_list,
#'                                breaks_list = breaks_list, note = FALSE)
#' #score card
#' LR_score_card <- get_score_card(lg_model = lr_model, bins_table, target = "target")
#' #scoring
#' train_pred = dat_train[, c("ID", "apply_date", "target")]
#' test_pred = dat_test[, c("ID", "apply_date", "target")]
#' train_pred$pred_LR = score_transfer(model = lr_model,
#'                                                     tbl_woe = train_woe,
#'                                                     save_data = FALSE)[, "score"]
#' 
#' test_pred$pred_LR = score_transfer(model = lr_model,
#' tbl_woe = test_woe, save_data = FALSE)[, "score"]
#' @importFrom car vif
#' @export
get_logistic_coef = function(lg_model, file_name = NULL,
                             dir_path = tempdir(), save_data = FALSE) {
    lg_coef = data.frame(summary(lg_model)$coefficients)
    lg_coef[4] = round(lg_coef[4],5)
    lg_coef[, "Feature"] = row.names(lg_coef)
    if (length(row.names(lg_coef)) > 2) {
        lg_coef[-1, "vif"] = car :: vif(lg_model)
    } else {
        lg_coef[-1, "vif"] = 0
    }
    names(lg_coef) <- c("estimate", "std.error", "Z_value", "P_value", "Feature", "vif")
    lg_coef = lg_coef[c("Feature", "estimate", "std.error", "Z_value", "P_value", "vif")]
    if (save_data) {
        dir_path = ifelse(is.null(dir_path) || !is.character(dir_path) || !grepl('.|/|:', dir_path),
                      tempdir(), dir_path)
        if (!dir.exists(dir_path)) dir.create(dir_path)
        if (!is.character(file_name)) file_name = NULL
        save_dt(lg_coef, file_name = ifelse(is.null(file_name), "logistic.coef", paste(file_name, "logistic.coef", sep = ".")), dir_path = dir_path, note = FALSE)
    }
    return(lg_coef)
}
