% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxphMIC.R
\name{coxphMIC}
\alias{coxphMIC}
\alias{coxph.MIC}
\title{Sparse Estimation for a Cox PH model via Approximated Information Criterion}
\usage{
coxphMIC(formula = Surv(time, status) ~ ., data, method.beta0 = "MPLE",
  beta0 = NULL, theta0 = 1, method = "BIC", lambda0 = 2, a0 = NULL,
  scale.x = TRUE, maxit.global = 300, maxit.local = 100,
  rounding.digits = 4, zero = sqrt(.Machine$double.eps),
  compute.se.gamma = TRUE, compute.se.beta = TRUE, CI.gamma = TRUE,
  conf.level = 0.95, details = FALSE)
}
\arguments{
\item{formula}{A formula object, with the response on the left of a \code{~} operator, and the terms on the right.
The response must be a survival object as returned by the \code{Surv} function.}

\item{data}{A data.frame in which to interpret the variables named in the \code{formula} argument.}

\item{method.beta0}{A method to supply the starting point for beta with choices: \code{"MPLE"} and "\code{ridge}".
By default, the maximum partial likelihood estimator (MPLE) is used with "\code{MPLE}". The option "\code{ridge}"
asks for a ridge estimator with penalty parameter specified by \code{theta0}. You may supply a set of values for beta0 of
your choice. If \code{NULL}, then beta0 is set as 0.}

\item{beta0}{User-supplied beta0 value, the starting point for optimization.}

\item{theta0}{Specified the penalty parameter for the ridge estimator when \code{method.beta0="ridge"}.}

\item{method}{Specifies the model selection criterion used. If \code{"AIC"}, the complexity penalty parameter (lambda)
equals 2; if \code{"BIC"}, lambda equals ln(n0), where n0 denotes the number of uncensored events. You may specify
the penalty parameter of your choice by setting \code{lambda0}.}

\item{lambda0}{User-supplied penalty parameter for model complexity. If \code{method="AIC"} or \code{"BIC"}, the value
of \code{lambda0} will be ignored.}

\item{a0}{The scale (or sharpness) parameter used in the hyperbolic tangent penalty. By default, \code{a0} is set as
n0, where n0 is again the number of uncensored events.}

\item{scale.x}{Logical value: should the predictors X be normalized? Default to \code{TRUE}.}

\item{maxit.global}{Maximum number of iterations allowed for the global optimization algorithm -- \code{SANN}. Default value is 300.}

\item{maxit.local}{Maximum number of iterations allowed for the local optimizaiton algorithm -- \code{BFGS}. Default value is 100.}

\item{rounding.digits}{Number of digits after the decimal point for rounding-up estiamtes. Default value is 4.}

\item{zero}{Tolerance level for convergence. Default is \code{sqrt(.Machine$double.eps)}.}

\item{compute.se.gamma}{Logical value indicating whether to compute the standard errors for gamma in the
reparameterization. Default is \code{TRUE}.}

\item{compute.se.beta}{Logical value indicating whether to compute the standard errors for nonzero beta estimates.
Default is \code{TRUE}. Note that this result is subject to post-selection inference.}

\item{CI.gamma}{Logical indicator of whether the confidence inverval for \code{gamma} is outputed. Default is \code{TRUE}.}

\item{conf.level}{Specifies the confidence level for \code{CI.gamma}. Defaulted as 0.95.}

\item{details}{Logical value: if \code{TRUE}, detailed results will be printed out when running \code{coxphMIC}.}
}
\value{
A list containing the following component is returned.
\describe{
\item{opt.global}{Results from the preliminary run of a global optimization procedure (\code{SANN} as default).}
\item{opt.local}{Results from the second run of a local optimization procedure (\code{BFGS} as default).}
\item{min.Q}{Value of the minimized objective function.}
\item{gamma}{Estimated gamma;}
\item{beta}{Estimated beta;}
\item{VCOV.gamma}{The estimated variance-covariance matrix for the gamma estimate;}
\item{se.gamma}{Standard errors for the gamma estimate;}
\item{se.beta}{Standard errors for the beta estimate (post-selection);}
\item{BIC}{The BIC value for the \emph{selected} model;}
\item{result}{A summary table of the fitting results.}
\item{call}{the matched call.}
}
}
\description{
Sparse Estimation for a Cox PH model via Approximated Information Criterion
}
\details{
The main idea of MIC involves approximation of the l0 norm with a continuous or smooth
unit dent function. This method bridges the best subset selection and regularization by
borrowing strength from both. It mimics the best subset selection using a penalized likelihood
approach yet with no need of a tuning parameter.

The problem is further reformulated with a reparameterization step by relating \code{beta}
to \code{gamma}. There are two benefits of doing so:  first, it reduces the optimization to
one unconstrained nonconvex yet smooth programming problem, which can be solved efficiently
as in computing the maximum partial likelihood estimator (MPLE); furthermore, the
reparameterization tactic yields an additional advantage in terms of circumventing post-selection inference.
Significance testing on \code{beta} can be done through \code{gamma}.

The solve the smooth yet nonconvex optimization, a simulated annealing (\code{method="SANN"} option
in \code{\link[stats]{optim}}) global optimization algorithm is first applied. The resultant estimator is then used
as the starting point for another local optimization algorithm. The quasi-Newton BFGS method (\code{method="BFGS"}
in \code{\link{optim}}) is used.

In its current version, some appropriate data preparation might be needed. For example, nomincal
variables (especially character-valued ones) needed to be coded with dummy variables; missing values would cause
errors too and hence need prehanlding too.
}
\examples{
  # PREPARE THE PBC DATA
  library(survival); data(pbc);
  dat <- pbc; dim(dat);
  dat$status <- ifelse(pbc$status==2, 1, 0)
  # HANDLE CATEGORICAL VARIABLES
  dat$sex <- ifelse(pbc$sex=="f", 1, 0)
  # LISTWISE DELETION USED TO HANDLE MISSING VALUES
  dat <- stats::na.omit(dat);
  dim(dat); utils::head(dat)

  fit.mic <- coxphMIC(formula=Surv(time, status)~.-id, data=dat, method="BIC", scale.x=TRUE)
  names(fit.mic)
  print(fit.mic)
  plot(fit.mic)
}
\references{
\itemize{
\item Abdolyousefi, R. N. and Su, X. (2016). \bold{coxphMIC}: An R package for sparse estimation of Cox PH Models via approximated information criterion. Tentatively accepted, \emph{The R Journal}.
\item Su, X. (2015). Variable selection via subtle uprooting.
\emph{Journal of Computational and Graphical Statistics}, \bold{24}(4): 1092--1113.
URL \url{http://www.tandfonline.com/doi/pdf/10.1080/10618600.2014.955176}
\item Su, X., Wijayasinghe, C. S., Fan, J., and Zhang, Y. (2015). Sparse estimation of Cox proportional
hazards models via approximated information criteria. \emph{Biometrics}, \bold{72}(3): 751--759.
URL \url{http://onlinelibrary.wiley.com/doi/10.1111/biom.12484/epdf}
}
}
\seealso{
\code{\link{coxph}}
}
