% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table-getClinDT.R
\name{getClinDT}
\alias{getClinDT}
\title{Create an interactive table to display clinical data}
\usage{
getClinDT(
  data,
  nonVisibleVar = NULL,
  nonVisible = NULL,
  percVar = NULL,
  barVar = NULL,
  barColorThr = NULL,
  barRange = NULL,
  filter = "top",
  searchBox = FALSE,
  pageLength,
  fixedColumns = NULL,
  columnsWidth = NULL,
  options = list(),
  expandVar = NULL,
  expandIdx = NULL,
  escape = TRUE,
  rowGroup = NULL,
  rowGroupVar = NULL,
  vAlign = "top",
  callback = NULL,
  buttons = getClinDTButtons(),
  scrollX = TRUE,
  file = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{Data.frame, matrix or \code{\link[crosstalk]{SharedData}}
object with input data for the table.}

\item{nonVisibleVar}{Character vector with column(s) in \code{data} to 
hide in the output table (column is hidden).\cr
The column(s) also get the extra attribute: \code{className = 'noVis'},
to ensure they are not displayed in the
button to show/hide column(s).}

\item{nonVisible}{This parameter is deprecated, use the new interface
with the \code{nonVisibleVar} parameter. 
Numeric vector with column(s)
in \code{data} to not display in the output table (column is hidden),
in \strong{Javascript unit: first column is 0}, second column is 1, ...}

\item{percVar}{Character vector with percentage columns.
These columns should contain the percentage from 0 to 1.
The content of these colunms will be rounded to 2 digits.}

\item{barVar}{Character vector with numeric variable of \code{data}
which should be represented as bar in the table.}

\item{barColorThr}{Numeric vector with threshold to 
consider to color the bar, either:
\itemize{
\item{a numeric vector of length 1, same threshold for all bars}
\item{named vector with threshold for each bar, named with the variable
in \code{barVar}}
}}

\item{barRange}{(optional) range for the bars, either:
\itemize{
\item{a numeric vector of length 2, same range for all bars}
\item{list with range for each bar, named with the variable
in \code{barVar}}
}
If not specified, the range of each \code{barVar} variable
in \code{data} is used.}

\item{filter}{String with position of the filter boxes
(\code{filter} parameter of the \code{\link[DT]{datatable}} function), 
'top' by default. Set to 'none' to not included any filtering boxes.}

\item{searchBox}{Logical, if TRUE (FALSE by default)
a general search box is included.}

\item{pageLength}{Numeric with number of records to include in one page,
by default set to 10.
Set to Inf to include all records.}

\item{fixedColumns}{List with fixed columns, see corresponding
parameter in the \code{options} parameter of the \code{\link[DT]{datatable}} function.}

\item{columnsWidth}{Character vector with column width,
of length 1 (used for all columns) or of length: \code{ncol(data)}}

\item{options}{List with additional \code{\link[DT]{datatable}} options.
This parameter overwrites the default options set internally
in the function (an indicative message mentions it if that is the case).}

\item{expandVar}{Character vector with expandable variables of \code{data}.
These columns won't be included in the table, but displayed
for each row when the '+' icon in the first column 
of the table will be clicked on.}

\item{expandIdx}{Matrix named with: 'row'/'column'
containing row/column indices to expand.}

\item{escape}{Column(s) to escape in the table
(e.g. containing raw HTML code), either character, numeric or logical of length 1.
See corresponding parameter in the \code{\link[DT]{datatable}} function.}

\item{rowGroup}{This parameter is deprecated, please use \code{rowGroup} instead.}

\item{rowGroupVar}{Character vector with colname(s) of \code{data}
containing variables to group rows by.
This creates row header containing this column.
Please note that the original row order in \code{data} is respected,
so you might want to order rows based on the grouping variables upfront.}

\item{vAlign}{String with vertical alignment for the cells,
'top' by default.}

\item{callback}{String with custom Javascript callback function.}

\item{buttons}{DataTable buttons
(passed to the 'buttons' element of the \code{options} parameter of \code{\link[DT]{datatable}}).
See \code{\link{getClinDTButtons}} for the default options.
To remove all buttons, set this parameter to NULL.}

\item{scrollX}{Logical, if TRUE (by default) a horizontal scrolling bar
is included. 
Note: this differs from the \code{\link[DT]{datatable}} default (FALSE),
because required for \code{bookdown::gitbook} output if table is too wide.}

\item{file}{(optional) String with name of html file to which the
created DT should be exported.}

\item{verbose}{Logical, if TRUE (by default) informative messages
are displayed, e.g. if specified \code{options} overwrite the
internal default.}

\item{...}{Additional parameters for the \code{\link[DT]{datatable}} function,
e.g table width.}
}
\value{
A \code{\link[DT]{datatable}} object.
}
\description{
This function converts a \code{data.frame} from R into 
a \code{\link[DT]{datatable}} object
with sensitive defaults.\cr
Extra functionalities are available to:
\itemize{
\item{have columns or cells of interest that are collapsible/expandable
(see \code{expandVar}/\code{expandIdx})}
\item{group rows based on a variable (see \code{rowGroupVar})}
\item{display a variable as barplot (with specified range of threshold)
(see \code{barVar})}
\item{hide variable(s) (see \code{nonVisibleVar})}
}
}
\examples{
data(dataADaMCDISCP01)
labelVars <- attr(dataADaMCDISCP01, "labelVars")

# example of simple adverse event table
dataAE <- dataADaMCDISCP01$ADAE
subjectsSafety <- subset(dataADaMCDISCP01$ADSL, SAFFL == "Y")$USUBJID

# compute counts of subjects presenting each AE
tableAE <- stats::aggregate(
	USUBJID ~ AESOC:AEDECOD,
	data = dataAE, 
	FUN = function(usubjid)	length(unique(usubjid))
)
colnames(tableAE)[colnames(tableAE) == "USUBJID"] <- "N"
# and percentages
tableAE$perc <- round(tableAE$N/length(subjectsSafety)*100, 3)
# sort records in decreasing percentage
tableAE <- tableAE[order(tableAE$perc, decreasing = TRUE), ]

# extract new variables labels
tableAELabels <- getLabelVar(
	var = colnames(tableAE),
	labelVars = labelVars,
	label = c(N = '# subjects', perc = "\% subjects")
)
# 'colnames' for DT should be specified as c('new name' = 'old name', ...)
tableAELabelsDT <- setNames(names(tableAELabels), tableAELabels)

## create table with bar

# default:
getClinDT(
	data = tableAE,
	barVar = "perc",
	colnames = tableAELabelsDT
)

# specify range for the bar
getClinDT(
	data = tableAE,
	filter = "none",
	barVar = "perc",
	barRange = c(0, 100),
	colnames = tableAELabelsDT
)

# change color according to threshold
getClinDT(
	data = tableAE,
	filter = "none",
	barVar = "perc",
	barColorThr = seq(from = 0, to = 100, by = 25),
	colnames = tableAELabelsDT
)

## group per system organ class (and decreasing N):
tableAESOC <- aggregate(N ~ AESOC, data = tableAE, FUN = sum)
tableAE$AESOC <- factor(tableAE$AESOC,
	levels = tableAESOC[order(tableAESOC$N, decreasing = FALSE), "AESOC"]
)
tableAE <- tableAE[order(tableAE$AESOC, tableAE$perc, decreasing = TRUE), ]
	
getClinDT(
	data = tableAE,
	filter = "none",
	barVar = "perc",
	barRange = c(0, 100),
	colnames = tableAELabelsDT,
	rowGroupVar = "AESOC",
	pageLength = Inf
)

# expand the subject ID column, will
# be accessible when clicking on the '+' button
# Format URL correctly with: 'escape',
# please note that indexing starts at 0!
getClinDT(
	data = tableAE,
	barVar = "perc",
	colnames = tableAELabelsDT,
	expandVar = "USUBJID", 
	escape = grep("USUBJID", colnames(tableAE))-1
)

# fix size for columns
getClinDT(
	data = tableAE,
	colnames = tableAELabelsDT,
	fixedColumns = list(leftColumns = 1),
	columnsWidth = c(0.1, 0.7, 0.1, 0.1),
	width = "350px" # change dimension table
)

\dontrun{# due to time constraint in CRAN

# change default buttons
getClinDT(
	data = tableAE,
	colnames = tableAELabelsDT,
	# remove general filter
	filter = "none",
	# custom set of buttons
	buttons = getClinDTButtons(type = c("csv", "excel", "pdf"))
)
# add button to select columns
getClinDT(
	data = tableAE,
	colnames = tableAELabelsDT,
	# custom set of buttons
	buttons = getClinDTButtons(typeExtra = "colvis")
)
# export pdf in landscape format
buttons <- getClinDTButtons(
	opts = list(pdf = list(orientation = "landscape"))
)
getClinDT(
	data = tableAE,
	colnames = tableAELabelsDT,
	# custom set of buttons
	buttons = buttons
)

# hide the first column:
getClinDT(
	data = tableAE,
	nonVisibleVar = "AESOC"
)

# with specific caption
library(htmltools)
caption <- tags$caption(
	"Number of subjects with adverse events grouped by system organ class.",
	br(), 
	paste(
		"Percentages are based on the total number of patients having",
		"received a first study treatment."
	)
)
getClinDT(
	data = tableAE,
	filter = "none",
	barVar = "perc",
	barRange = c(0, 100),
	pageLength = Inf,
	colnames = tableAELabelsDT,
	rowGroupVar = "AESOC",
	caption = caption
)

}
}
\author{
Laure Cougnaud
}
