% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analIntervals.r
\name{deltaInverse}
\alias{deltaInverse}
\title{Calculate inverse (dose-finding) intervals, using local inversion and the Delta method}
\usage{
deltaInverse(
  y,
  x = NULL,
  wt = NULL,
  target = NULL,
  estfun = cirPAVA,
  intfun = morrisCI,
  conf = 0.9,
  adaptiveShrink = FALSE,
  starget = target[1],
  parabola = FALSE,
  ...
)
}
\arguments{
\item{y}{can be either of the following: y values (response rates), a 2-column matrix with positive/negative response counts by dose, a \code{\link{DRtrace}} object or a \code{\link{doseResponse}} object.}

\item{x}{dose levels (if not included in y).}

\item{wt}{weights (if not included in y).}

\item{target}{A vector of target response rate(s), for which the interval is needed. If \code{NULL} (default), interval will be returned for the point estimates at design points (e.g., if the forward point estimate at $x_1$ is 0.2, then the first returned interval is for the 20th percentile).}

\item{estfun}{the function to be used for point estimation. Default \code{\link{cirPAVA}}.}

\item{intfun}{the function to be used for initial (forward) interval estimation. Default \code{\link{morrisCI}} (see help on that function for additional options).}

\item{conf}{numeric, the interval's confidence level as a fraction in (0,1). Default 0.9.}

\item{adaptiveShrink}{logical, should the y-values be pre-shrunk towards an experiment's target? Recommended if data were obtained via an adaptive dose-finding design. See \code{\link{DRshrink}}.}

\item{starget}{The shrinkage target. Defaults to \code{target[1]}.}

\item{parabola}{logical, should the interpolation between design points follow a parabola (\code{TRUE}) or a straight line (\code{FALSE}, default)? See details.}

\item{...}{additional arguments passed on to \code{\link{quickIsotone}}}
}
\value{
two-column matrix with the left and right bounds, respectively
}
\description{
Calculate left-bound to right-bound intervals for the dose point estimates, using local slopes at design points (places where observations exist) to invert the forward lower-upper bounds.
}
\details{
The Delta method in this application boils down to dividing the distance to the forward (vertical) bounds, by the slope, to get the left/right interval width. Slope estimates are performed by \code{\link{slope}}. An alternative method (dubbed "global") is hard-coded into \code{\link{quickInverse}}.
}
\examples{
# Interesting run (#664) from a simulated up-and-down ensemble:
# (x will be auto-generated as dose levels 1:5)
dat=doseResponse(y=c(1/7,1/8,1/2,1/4,4/17),wt=c(7,24,20,12,17))
# The experiment's goal is to find the 30th percentile
quick1=quickIsotone(dat)
invDelta=deltaInverse(dat)

### Showing the data and the estimates
par(mar=c(3,3,4,1),mgp=c(2,.5,0),tcl=-0.25)
# Following command uses plot.doseResponse()
plot(dat,ylim=c(0.05,0.55),refsize=4,las=1,xlim=c(-1,6),main="Inverse-Estimation CIs") 

# The true response function; true target is where it crosses the y=0.3 line
lines(seq(0,7,0.1),pweibull(seq(0,7,0.1),shape=1.1615,scale=8.4839),col=4)
abline(h=0.3,col=2,lty=3) ### The experiment's official target

# Forward CIs; the "global" inverse interval just draws horizontal lines between them
# To get "global" values calculated for you at specific targets, choose 'delta=FALSE' 
# when calling quickInverse()
lines(quick1$lower90conf,lty=2,col=3) 
lines(quick1$upper90conf,lty=2,col=3) 
# Note how for y=0.3, both bounds are infinite (i.e., no intersection with the horizontal line)
# unless one dares to extrapolate outside range of observations.

# Now, the default "local" inverse interval, which is finite for the range of estimated y values.
# In particular, it is finite (albeit very wide) for y=0.3.
lines(invDelta[,1],quick1$y,lty=2)
lines(invDelta[,2],quick1$y,lty=2)

legend('topleft',pch=c(NA,'X',NA,NA),lty=c(1,NA,2,2),col=c(4,1,1,3),legend=
	c('True Curve','Observations','Local Interval (default)','Forward/Global Interval'),bty='n')

}
\seealso{
\code{\link{quickIsotone}},\code{\link{quickInverse}},\code{\link{isotInterval}},
\code{\link{slope}}; \code{\link{DRshrink}} for the shrinkage fix.
}
