% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/custom_callbacks.R
\name{trace_plotter}
\alias{trace_plotter}
\title{Plot simulation covers}
\usage{
trace_plotter(
  mod,
  initmat,
  fun = NULL,
  col = NULL,
  max_samples = 256,
  fps_cap = 24,
  burn_in = 0,
  new_window = TRUE,
  ...
)
}
\arguments{
\item{mod}{The model being used (created by \code{link{camodel}})}

\item{initmat}{The initial landscape given to \code{\link{run_camodel}}}

\item{fun}{The function used to summarise the landscape into summary metrics. By
default, gloal covers of each state are computed. It must return a vector of
numeric values.}

\item{col}{a set of colors (character vector) of length equal to the number of
values returned by fun.}

\item{max_samples}{The maximum number of samples to display in the plot}

\item{fps_cap}{The maximum number of frame displayed per seconds. Simulation
will be slowed down if necessary so that plot updates will not be
more frequent than this value}

\item{burn_in}{Do not display anything before this time step has been
reached}

\item{new_window}{Controls whether the plots are displayed in a new window,
or in the default device (typically the plot panel in Rstudio)}

\item{...}{other arguments passed to \code{\link[graphics]{matplot}}, which is used to
display the trends.}
}
\value{
This function returns another function, which will be called internally 
   when simulating the model using \code{\link{run_camodel}}, and has probably 
   not much use outside of this context. The return function will display the 
   simulation and returns NULL.
}
\description{
This function creates an internal function to plot the model
  landscape during the simulation of a stochastic cellular automaton.
}
\details{
This function creates another function that is suitable for use with
  \code{\link{run_camodel}}. It can plot any quantity computed on the
  landscape as it is being simulated, using the base function
  \code{\link[graphics]{matplot}}. The resulting function must by passed to
  \code{\link{run_camodel}} as the control argument \code{custom_output_fun}.
  Typically, this function is not used by itself, but is being used when
  specifying simulation options before calling \code{\link{run_camodel}},
  see examples below.

  By default, the global covers of each state in the landscape will be displayed, but
  you can pass any function as argument \code{fun} to compute something else, as long
  as \code{fun} returns a numeric vector of length at least 1.

  \code{\link[graphics]{matplot}} is used internally, and tends to be quite slow at
  displaying results, but if it is still too fast for your taste, you can cap the
  refresh rate at a value given by the argument \code{fps_cap}.

  It is important to note that this function will probably massively slow down a
  simulation, so it is mostly here for exploratory analyses, or just to
  have a good look of what is happening in a model.
}
\examples{

\donttest{ 

# Display covers of the rock/paper/scissor model as it is running
mod <- ca_library("rock-paper-scissor")
init <- generate_initmat(mod, rep(1, 3)/3, nrow = 100, ncol = 178)
ctrl <- list(custom_output_every = 1,
             custom_output_fun = trace_plotter(mod, init))
run_camodel(mod, init, times = seq(0, 256), control = ctrl)

# Adjust colors of the previous example and increase speed
colors <- c("#fb8500", "#023047", "#8ecae6")
ctrl <- list(custom_output_every = 1,
             custom_output_fun = trace_plotter(mod, init, fps_cap = 60, col = colors))
run_camodel(mod, init, times = seq(0, 600), control = ctrl)

# Display the vegetated to degraded cover ratio for the arid
# vegetation model. 
mod <- ca_library("aridvege")
init <- generate_initmat(mod, rep(1, 3)/3, nrow = 100, ncol = 178)
ratio <- function(mat) {
  mean(mat == "VEGE") / mean(mat == "DEGR")
}
ctrl <- list(custom_output_every = 1,
             custom_output_fun = trace_plotter(mod, init,
                                               fun = ratio))
run_camodel(mod, init, times = seq(0, 512), control = ctrl)

# Display ratios of cell pairs in the rock-paper-scissor model
mod <- ca_library("rock-paper-scissor")
init <- generate_initmat(mod, rep(1, 3)/3, nrow = 100, ncol = 178)
ratio <- function(mat) {
  c(mean(mat == "r") / mean(mat == "p"), 
    mean(mat == "p") / mean(mat == "c"), 
    mean(mat == "c") / mean(mat == "r"))
}
ctrl <- list(custom_output_every = 1,
             custom_output_fun = trace_plotter(mod, init,
                                               fun = ratio))
run_camodel(mod, init, times = seq(0, 512), control = ctrl)

} 

}
\seealso{
landscape_plotter, run_camodel
}
