% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apes.R
\name{apes}
\alias{apes}
\title{Compute average partial effects after fitting binary choice models
with a 1,2,3-way error component}
\usage{
apes(
  object = NULL,
  n_pop = NULL,
  panel_structure = c("classic", "network"),
  sampling_fe = c("independence", "unrestricted"),
  weak_exo = FALSE
)
}
\arguments{
\item{object}{an object of class \code{"bias_corr"} or \code{"feglm"};
currently restricted to \code{\link[stats]{binomial}}.}

\item{n_pop}{unsigned integer indicating a finite population correction for
the estimation of the covariance matrix of the average partial effects
proposed by Cruz-Gonzalez, Fernández-Val, and Weidner (2017). The correction
factor is computed as follows:
\eqn{(n^{\ast} - n) / (n^{\ast} - 1)}{(n_pop - n) / (n_pop - 1)},
where \eqn{n^{\ast}}{n_pop} and \eqn{n}{n} are the sizes of the entire
population and the full sample size. Default is \code{NULL}, which refers to
a factor of zero and a covariance obtained by the delta method.}

\item{panel_structure}{a string equal to \code{"classic"} or \code{"network"}
which determines the structure of the panel used. \code{"classic"} denotes
panel structures where for example the same cross-sectional units are
observed several times (this includes pseudo panels). \code{"network"}
denotes panel structures where for example bilateral trade flows are
observed for several time periods. Default is \code{"classic"}.}

\item{sampling_fe}{a string equal to \code{"independence"} or
\code{"unrestricted"} which imposes sampling assumptions about the
unobserved effects. \code{"independence"} imposes that all unobserved
effects are independent sequences. \code{"unrestricted"} does not impose any
sampling assumptions. Note that this option only affects the optional finite
population correction. Default is \code{"independence"}.}

\item{weak_exo}{logical indicating if some of the regressors are assumed to
be weakly exogenous (e.g. predetermined). If object is of class
\code{"bias_corr"}, the option will be automatically set to \code{TRUE} if
the chosen bandwidth parameter is larger than zero. Note that this option
only affects the estimation of the covariance matrix. Default is
\code{FALSE}, which assumes that all regressors are strictly exogenous.}
}
\value{
The function \code{\link{apes}} returns a named list of class
\code{"apes"}.
}
\description{
\code{\link{apes}} is a post-estimation routine that can be used
to estimate average partial effects with respect to all covariates in the
model and the corresponding covariance matrix. The estimation of the
covariance is based on a linear approximation (delta method) plus an
optional finite population correction. Note that the command automatically
determines which of the regressors are binary or non-binary.

\strong{Remark:} The routine currently does not allow to compute average
partial effects based on functional forms like interactions and polynomials.
}
\examples{
# subset trade flows to avoid fitting time warnings during check
set.seed(123)
trade_2006 <- trade_panel[trade_panel$year == 2006, ]
trade_2006 <- trade_2006[sample(nrow(trade_2006), 500), ]

trade_2006$trade <- ifelse(trade_2006$trade > 100, 1L, 0L)

# Fit 'feglm()'
mod <- feglm(trade ~ lang | year, trade_2006, family = binomial())

# Compute average partial effects
mod_ape <- apes(mod)
summary(mod_ape)

# Apply analytical bias correction
mod_bc <- bias_corr(mod)
summary(mod_bc)

# Compute bias-corrected average partial effects
mod_ape_bc <- apes(mod_bc)
summary(mod_ape_bc)

}
\references{
Cruz-Gonzalez, M., I. Fernández-Val, and M. Weidner (2017). "Bias
corrections for probit and logit models with two-way fixed effects". The
Stata Journal, 17(3), 517-545.

Czarnowske, D. and A. Stammann (2020). "Fixed Effects Binary
Choice Models: Estimation and Inference with Long Panels". ArXiv e-prints.

Fernández-Val, I. and M. Weidner (2016). "Individual and time
effects in nonlinear panel models with large N, T". Journal of Econometrics,
192(1), 291-312.

Fernández-Val, I. and M. Weidner (2018). "Fixed effects
estimation of large-t panel data models". Annual Review of Economics, 10,
109-138.

Hinz, J., A. Stammann, and J. Wanner (2020). "State Dependence
and Unobserved Heterogeneity in the Extensive Margin of Trade". ArXiv
e-prints.

Neyman, J. and E. L. Scott (1948). "Consistent estimates based on
partially consistent observations". Econometrica, 16(1), 1-32.
}
\seealso{
\code{\link{bias_corr}}, \code{\link{feglm}}
}
