% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cNORM.R
\name{cnorm}
\alias{cnorm}
\title{Continuous Norming}
\usage{
cnorm(
  raw = NULL,
  group = NULL,
  age = NULL,
  width = NA,
  weights = NULL,
  scale = "T",
  method = 4,
  descend = FALSE,
  k = NULL,
  t = NULL,
  terms = 0,
  R2 = NULL,
  plot = TRUE,
  extensive = TRUE,
  subsampling = TRUE
)
}
\arguments{
\item{raw}{Numeric vector of raw scores}

\item{group}{Numeric vector of grouping variable, e. g. grade. If no group
or age variable is provided, conventional norming is applied}

\item{age}{Numeric vector with chronological age, please additionally specify
width of window}

\item{width}{Size of the sliding window in case an age vector is used}

\item{weights}{Vector or variable name in the dataset with weights for each
individual case. It can be used to compensate for moderate imbalances due to
insufficient norm data stratification. Weights should be numerical and positive.}

\item{scale}{type of norm scale, either T (default), IQ, z or percentile (= no
transformation); a double vector with the mean and standard deviation can as
well, be provided f. e. c(10, 3) for Wechsler scale index points}

\item{method}{Ranking method in case of bindings, please provide an index,
choosing from the following methods: 1 = Blom (1958), 2 = Tukey (1949),
3 = Van der Warden (1952), 4 = Rankit (default), 5 = Levenbach (1953),
6 = Filliben (1975), 7 = Yu & Huang (2001)}

\item{descend}{ranking order (default descent = FALSE): inverses the
ranking order with higher raw scores getting lower norm scores; relevant
for example when norming error scores, where lower scores mean higher
performance}

\item{k}{The power constant. Higher values result in more detailed approximations
but have the danger of over-fit (max = 6). If not set, it uses t and if both
parameters are NULL, k is set to 5.}

\item{t}{The age power parameter (max = 6). If not set, it uses k and if both
parameters are NULL, k is set to 3, since age trajectories are most often well
captured by cubic polynomials.}

\item{terms}{Selection criterion for model building. The best fitting model with
this number of terms is used}

\item{R2}{Adjusted R square as a stopping criterion for the model building
(default R2 = 0.99)}

\item{plot}{Default TRUE; plots the regression model and prints report}

\item{extensive}{If TRUE, screen models for consistency and - if possible, exclude inconsistent ones}

\item{subsampling}{If TRUE (default), model coefficients are calculated using 10-folds and averaged across the folds.
This produces more robust estimates with a slight increase in bias.}
}
\value{
cnorm object including the ranked raw data and the regression model
}
\description{
Conducts continuous norming in one step and returns an object including ranked
raw data and the continuous norming model. Please consult the function
description ' of 'rankByGroup', 'rankBySlidingWindow' and 'bestModel' for specifics
of the steps in the data preparation and modeling process. In addition to the
raw scores, either provide
\itemize{
 \item{a numeric vector for the grouping information (group)}
 \item{a numeric age vector and the width of the sliding window (age, width)}
}
for the ranking of the raw scores. You can
adjust the grade of smoothing of the regression model by setting the k and terms
parameter. In general, increasing k to more than 4 and the number of terms lead
to a higher fit, while lower values lead to more smoothing. The power parameter
for the age trajectory can be specified independently by 't'. If both parameters
are missing, cnorm uses k = 5 and t = 3 by default.
}
\examples{
\dontrun{
# Using this function with the example dataset 'elfe'

# Conventional norming (no modelling over age)
cnorm(raw=elfe$raw)

# Continuous norming
# You can use the 'getGroups()' function to set up grouping variable in case,
# you have a continuous age variable.
cnorm.elfe <- cnorm(raw = elfe$raw, group = elfe$group)

# return norm tables including 90\% confidence intervals for a
# test with a reliability of r = .85; table are set to mean of quartal
# in grade 3 (children completed 2 years of schooling)
normTable(c(2.125, 2.375, 2.625), cnorm.elfe, CI = .90, reliability = .95)

# ... or instead of raw scores for norm scores, the other way round
rawTable(c(2.125, 2.375, 2.625), cnorm.elfe, CI = .90, reliability = .95)


# Using a continuous age variable instead of distinct groups, using a sliding
# window for percentile estimation. Please specify continuos variable for age
# and the sliding window size.
cnorm.ppvt.continuous <- cnorm(raw = ppvt$raw, age = ppvt$age, width=1)


# In case of unbalanced datasets, deviating from the census, the norm data
# can be weighted by the means of raking / post stratification. Please generate
# the weights with the computeWeights() function and pass them as the weights
# parameter. For computing the weights, please specify a data.frame with the
# population margins (further information is available in the computeWeights
# function). A demonstration based on sex and migration status in vocabulary
# development (ppvt dataset):
margins <- data.frame(variables = c("sex", "sex",
                                    "migration", "migration"),
                      levels = c(1, 2, 0, 1),
                      share = c(.52, .48, .7, .3))
weights <- computeWeights(ppvt, margins)
model <- cnorm(raw = ppvt$raw, group=ppvt$group, weights = weights)
}
}
\references{
\enumerate{
  \item Gary, S. & Lenhard, W. (2021). In norming we trust. Diagnostica.
  \item Gary, S., Lenhard, W. & Lenhard, A. (2021). Modelling Norm Scores with the cNORM Package in R. Psych, 3(3), 501-521. https://doi.org/10.3390/psych3030033
  \item Lenhard, A., Lenhard, W., Suggate, S. & Segerer, R. (2016). A continuous solution to the norming problem. Assessment, Online first, 1-14. doi:10.1177/1073191116656437
  \item Lenhard, A., Lenhard, W., Gary, S. (2018). Continuous Norming (cNORM). The Comprehensive R Network, Package cNORM, available: https://CRAN.R-project.org/package=cNORM
  \item Lenhard, A., Lenhard, W., Gary, S. (2019). Continuous norming of psychometric tests: A simulation study of parametric and semi-parametric approaches. PLoS ONE, 14(9),  e0222279. doi:10.1371/journal.pone.0222279
  \item Lenhard, W., & Lenhard, A. (2020). Improvement of Norm Score Quality via Regression-Based Continuous Norming. Educational and Psychological Measurement(Online First), 1-33. https://doi.org/10.1177/0013164420928457

}
}
\seealso{
rankByGroup, rankBySlidingWindow, computePowers, bestModel
}
