% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factblocks.r
\name{factblocks}
\alias{factblocks}
\title{Block designs for factorial treatment sets}
\usage{
factblocks(treatments, replicates = 1, rows = NULL, columns = NULL,
  model = NULL, searches = NULL, seed = sample(10000, 1), jumps = 1)
}
\arguments{
\item{treatments}{a data frame with columns for individual treatment factors and rows for individual treatment factor combinations.}

\item{replicates}{a single replication number, not necessarily integral.}

\item{rows}{the number of rows nested in each preceding block for each level of nesting from the top-level block downwards. The top-level block is a
single super-block which need not be defined explicitly.}

\item{columns}{the number of columns nested in each preceding block for each level of nesting from the top-level block downwards. The \code{rows} and 
\code{columns} parameters must be of equal length unless the \code{columns} parameter is null, in which case the design has a
single column block for each level of nesting and the design becomes a simple nested row blocks design.}

\item{model}{a model equation for the treatment factors in the design where the equation is defined using the model.matrix notation
in the {\link[stats]{model.matrix}} package. If undefined, the model is a full factorial treatment design.}

\item{searches}{the maximum number of local optima searched for a design optimization. The default is 1 plus the floor of 10000 divided by the number of plots.}

\item{seed}{an integer initializing the random number generator. The default is a random seed.}

\item{jumps}{the number of pairwise random treatment swaps used to escape a local maximum. The default is a single swap.}
}
\value{
\item{Treatments}{The treatment factors defined by the \code{treatments} inputs in standard factorial order.}
\item{model.matrix}{The model.matrix used to define the \code{treatments} design.}
\item{Design}{Data frame giving the optimized block and treatment factors in plot order.}
\item{BlocksEfficiency}{The D-efficiencies of the blocks in each stratum of the design.}
\item{DesignEfficiency}{The generalized variance of the complete factorial design divided by the generalized variance of the fractional factorial design.}
\item{seed}{Numerical seed for random number generator.}
\item{searches}{Maximum number of searches in each stratum.}
\item{jumps}{Number of random treatment swaps to escape a local maxima.}
}
\description{
Constructs randomized nested block designs for factorial or fractional factorial treatment designs with any
feasible depth of nesting and up to two crossed block structures in each level of nesting.
}
\details{
\code{factblocks} generates blocked factorial designs for general factorial treatment structures possibly including
mixtures of qualitative and quantitative level factors. Qualitative level factors are
modelled factorially while quantitative level factors are modelled by polynomials of the required degree.  
Designs can be based on any multiple, not necessarily integral, of the complete factorial 
treatment design where the fractional part of the design, if any, is chosen by optimizing a 
D-optimal fraction of that size for that treatment design.

The \code{treatments} parameter defines the treatment factors of the design and must be a data frame with
a column for each factor and a row for each factorial combination (see examples). The treatment factors 
can be any mixture of qualitative or quantitative level factors and the treatment model can be any feasible model defined 
by the \code{models} formula of the \code{\link[stats]{model.matrix}} package (see examples). 

Quantitative factors can be modelled either by raw or by orthogonal polynomials. Orthogonal polynomials are numerically more stable
than raw polynomials and are usually the best choice at the design stage. Polynomial models can be fitted at the analysis stage either by raw or
by orthogonal polynomials regardless of the type of polynomial fitted at the design stage.
  
The \code{replicates} parameter defines the required replication for the treatments design and should be a single number, not necessarily integral, 
representing a required multiple or a required fraction of the \code{treatments} data frame. The algorithm will find a
D-optimal or near D-optimal fraction of the required size for any fractional part of replication number, assuming the required design is non-singular.

The \code{rows} parameter, if any, defines the nested row blocks for each level of nesting taken in order from the highest to the lowest. The
first number, if any, is the number of nested row blocks in the first-level of nesting, the second number, if any, is the number of nested row blocks in
the second-level of nesting and so on for any required feasible depth of nesting.

The \code{columns} parameter, if any, defines the nested column blocks for each level of nesting taken in order from the highest to the lowest.
The first number, if any, is the number of nested column blocks in the first-level of nesting, the second, if any, is the number of nested column blocks in
the second-level of nesting and so on for the same required depth of nesting as in the \code{rows} parameter.

The \code{rows} and \code{columns} parameters, if defined, must be of equal length. If a simple set of nested blocks is required for
any particular level of nesting, the number of columns for that level should be set to unity. Any required combination of simple or
crossed blocks can be obtained by appropriate choice of the levels of the \code{rows} and \code{columns} parameters.
If the \code{rows} parameter is defined but the \code{columns} parameter is null, the design will be a simple nested
blocks design with numbers of block levels defined by the \code{rows} parameter. If both the \code{rows} parameter and the \code{columns} parameter are null, 
the default block design will be a set of orthogonal main blocks equal in number to the highest common factor of the replication numbers. 

Block sizes are always as nearly equal as possible and will never differ by more than a single plot for any particular block classification. 
Row blocks and column blocks must always contain at least two plots per block and this restriction will constrain the permitted numbers of 
rows and columns in the various nested levels of a block design.

For any particular level of nesting, the algorithm first optimizes the row blocks conditional on any higher-level blocks
and then optimizes the columns blocks, if any, conditional on the rows blocks.

The efficiency factor of a fractional factorial design is the generalized variance of the complete factorial design divided by the generalized variance of
the fractional factorial design where the generalized variance of a design is the (1/p)th power of the determinant of the crossed-product of the p-dimensional
model matrix divided by the number of observations in the design. 

Comment:

Row-and-column designs may contain useful treatment information in the individual row-by-column intersection blocks but \code{blocksdesign} does not currently
optimize the efficiency of these blocks.

Row-and-column design with 2 complete treatment replicates, 2 complete rows and 2 complete columns will always confound one treatment contrast in the
rows-by-columns interaction. For these designs, it is impossible to nest a non-singular block design in the rows-by-columns intersections and instead
we suggest a randomized nested blocks design with four incomplete main blocks.

Outputs:

The principle design outputs comprise:
\itemize{
 \item  A data frame showing the allocation of treatments to blocks with successive nested strata arranged in standard block order. \cr
 \item  A table showing the replication number of each treatment in the design. \cr
 \item  An efficiency factor for fractional factorial treatment designs. \cr
 \item  A table showing the block levels and the achieved D-efficiency factors for each stratum. \cr
}
}
\examples{

## The number of searches in the examples have been limited for fast execution. 
## For optimum results, the number of searches may need to be increased in practice.
## Designs should be rebuilt repeatedly to check that a near-optimum design has been found.  


## Factorial designs defined by a treatments data frame and a factorial model equation.

# Main effects of five 2-level factors in a half-fraction of a 4 x 4 row-and column design.
GF = expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2))
factblocks(treatments=GF,model="~ F1+F2+F3+F4+F5",replicates=.5,rows=4,columns=4,searches=20)

# Quadratic regression for one 6-level numeric factor in 2 randomized blocks assuming 10/6 fraction
factblocks(treatments=expand.grid(X=1:6),model=" ~ poly(X,2)",rows=2,searches=5,replicates=10/6) 

# Second-order model for five qualitative 2-level factors in 4 randomized blocks
GF=expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2))
factblocks(treatments=GF,model=" ~ (F1+F2+F3+F4+F5)^2",rows=4,searches=5)

# First-order model for 1/3rd fraction of four qualitative 3-level factors in 3  blocks
GF=expand.grid(F1=factor(1:3),F2=factor(1:3),F3=factor(1:3),F4=factor(1:3))
factblocks(treatments=GF,model=" ~ F1+F2+F3+F4",replicates=(1/3),rows=3,searches=5)

# Second-order model for a 1/3rd fraction of five qualitative 3-level factors in 3 blocks
GF=expand.grid( F1=factor(1:3), F2=factor(1:3), F3=factor(1:3), F4=factor(1:3), F5=factor(1:3) )
factblocks(treatments=GF,model=" ~ (F1+F2+F3+F4+F5)^2",rows=3,replicates=(1/3),searches=5)

# Second-order model for two qualitative and two quantitative level factors in 4 randomized blocks
GF=expand.grid(F1=factor(1:2),F2=factor(1:3),V1=1:3,V2=1:4)
modelform=" ~ F1 + F2 + poly(V1,2) +  poly(V2,2) + (poly(V1,1)+F1+F2):(poly(V2,1)+F1+F2) "
factblocks(treatments=GF,model=modelform,rows=4,searches=5)

# Plackett and Burman design for eleven 2-level factors in 12 runs (needs large number of searches)
GF=expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2),
F6=factor(1:2),F7=factor(1:2),F8=factor(1:2),F9=factor(1:2),F10=factor(1:2),F11=factor(1:2))
\dontrun{factblocks(GF,model="~ F1+F2+F3+F4+F5+F6+F7+F8+F9+F10+F11",replicates=(12/2048))}


}
\references{
Sailer, M. O. (2013). crossdes: Construction of Crossover Designs. R package version 1.1-1. https://CRAN.R-project.org/package=crossdes

Edmondson R. N. (1998). Trojan square and incomplete Trojan square designs for crop research. Journal of Agricultural Science, Cambridge, 131, pp.135-142

Cochran, W.G., and G.M. Cox. 1957. Experimental Designs, 2nd ed., Wiley, New York.
}
