% Generated by roxygen2 (4.0.2): do not edit by hand
\name{blocks}
\alias{blocks}
\title{Block designs}
\usage{
blocks(treatments, replicates, blocklevels = hcf, searches = min(64,
  floor(4096/nunits)), seed = NULL)
}
\arguments{
\item{treatments}{A list of the number of treatments for each treatment set in the design. Each treatment number must have a matching
replication number in the \code{replicates} list}

\item{replicates}{A list of the number of replicates for each treatment set in the design. Each replication number must
have a matching treatment number in the \code{treatments} list}

\item{blocklevels}{A hierarchical list of nested blocks where the first number is the number of main blocks and the  remaining numbers, if any,
are the numbers of blocks nested in each preceding block. The default is the hcf of the replication numbers.}

\item{searches}{The number of local optima searched during
a design optimization. The default is the minimum of 64 or the integer quotient of 4096 divided by the number of plots.}

\item{seed}{An integer seed for initializing the random number generator where a design must be reproducible. The default is a random seed.}
}
\value{
\item{Design}{Data frame showing the block and treatment factors for each plot}
\item{Plan}{Data frame showing the allocation of treatments to plots for each block in the design}
\item{Incidences}{List of blocks-by-treatments incidence matrices, one for each stratum in the design}
\item{Efficiencies}{Data frame showing the A-efficiency factor for each stratum in the design together with an upper bound, where available}
\item{seed}{Numerical seed for random number generator}
}
\description{
\code{blocks} function constructs nested blocks for unstructured treatments with arbitrary levels of replication and arbitrary depth of nesting
}
\details{
\code{blocks} constructs nested block designs for unstructured treatment sets where treatments can have any arbitrary replication, not necessarily all equal,
and blocks can have any feasible depth of nesting.

Treatment and replication numbers are defined by the \code{treatments} and \code{replicates} parameter lists. These lists must be of equal length
and each matching pair of numbers in the two lists represents a treatment set where the \code{treatments} list gives the number of treatments in the set and
the \code{replicates} list gives the replication of the set.

Any number of treatment sets is allowed and the treatments are numbered consecutively according to the ordering of the treatment sets in the parameter lists
(see the examples).

Blocks are defined by the \code{blocklevels} list which is a hierarchical list of nested blocks. The first number is the number of main blocks
and the succesive numbers, if any, are the numbers of blocks nested in each preceding block. The cumulative product of the levels for any stratum
is the total number of blocks in that stratum. The default value for the \code{blocklevels} list is a single number equal to the highest common factor (hcf) of
the replication numbers, which gives an orthogonal blocks design with the maximum possible number of othogonal blocks.

Block sizes in any given stratum will be equal if the cumulative number of blocks exactly divides the number of plots otherwise
they will be as near equal as possible and will never differ by more than a single unit.

Lattice designs where the number of treatments is the square of the block size v and the number of replicates is k+2 or less and k mutually
orthogonal latin squares (MOLS) of size v*v exist are constructed algebraically. Lattice designs exist for any
v if k = 1 and for any prime or prime power v if k < v and for v = 10 and k = 2. Prime-power MOLS are constructed by using the MOLS
function of the \code{crossdes} package (Sailer 2013).

All other designs are constructed algorithmically by a swapping algorithm that maximizes the determinant of the information matrix (D-optimality).

Designs are optimized hierarchically with the blocks of each new set optimized within the blocks of the preceding set.

Designs are fully randomized with treatments randomized within blocks and each set of nested blocks randomized within the
preceding set of blocks.
}
\examples{
# 3 treatments with 2 reps, 2 treatments with 4 reps, 4 treatments with 3 reps
# the replication hcf is 1 and the default design is a completely randomized design
blocks(treatments=c(3,2,4),replicates=c(2,4,3))

# 50 treatments with 4 reps in 4 complete randomized blocks
blocks(treatments=50,replicates=4)

# as above but with 4 main blocks and 5 nested blocks in each main block
blocks(treatments=50,replicates=4,blocklevels=c(4,5))

# as above but with 20 additional single replicate treatments, one to each block
blocks(treatments=c(50,20),replicates=c(4,1),blocklevels=c(4,5))

# 64 treatments with 2 reps and 2 main blocks with five 2-level nested factors
blocks(treatments=64,replicates=2,blocklevels=c(2,2,2,2,2,2),searches=12)

# concurrence matrices of 36 treatments with 3 reps and 3 main blocks with 6 nested blocks
crossprod(blocks(treatments=36,replicates=3,blocklevels=c(3,6))$Incidences[[2]])

# concurrence matrix for 13 treatments with 4 reps and 13 treatments with one rep in 13 blocks
crossprod(blocks(c(13,13),c(4,1),13,searches=100)$Incidences[[1]])
}
\references{
Sailer, M. O. (2013). crossdes: Construction of Crossover Designs. R package version 1.1-1. http://CRAN.R-project.org/package=crossdes
}

