% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integrate_to_ppi.R
\name{integrate_to_ppi}
\alias{integrate_to_ppi}
\title{Calculate a plan position indicator (\code{ppi}) of vertically integrated density
adjusted for range effects}
\usage{
integrate_to_ppi(
  pvol,
  vp,
  nx = 100,
  ny = 100,
  xlim,
  ylim,
  zlim = c(0, 4000),
  res,
  quantity = "eta",
  param = "DBZH",
  raster = NA,
  lat,
  lon,
  antenna,
  beam_angle = 1,
  crs,
  param_ppi = c("VIR", "VID", "R", "overlap", "eta_sum", "eta_sum_expected"),
  k = 4/3,
  re = 6378,
  rp = 6357
)
}
\arguments{
\item{pvol}{A \code{pvol} object.}

\item{vp}{A \code{vp} object}

\item{nx}{number of raster pixels in the x (longitude) dimension}

\item{ny}{number of raster pixels in the y (latitude) dimension}

\item{xlim}{x (longitude) range}

\item{ylim}{y (latitude) range}

\item{zlim}{Numeric vector of length two. Altitude range, in m}

\item{res}{numeric vector of length 1 or 2 to set the resolution of the raster (see \link[raster:resolution]{res}).
If this argument is used, arguments \code{nx} and \code{ny} are ignored. Unit is identical to \code{xlim} and \code{ylim}.}

\item{quantity}{Character. Profile quantity on which to base range
corrections, either \code{eta} or \code{dens}.}

\item{param}{reflectivity Character. Scan parameter on which to base range
corrections. Typically the same parameter from which animal densities are
estimated in \code{vp}. Either \code{DBZH}, \code{DBZV}, \code{DBZ}, \code{TH}, or \code{TV}.}

\item{raster}{(optional) RasterLayer with a CRS. When specified this raster topology is used for the output, and nx, ny, res
arguments are ignored.}

\item{lat}{Latitude of the radar, in degrees. If missing taken from \code{pvol}.}

\item{lon}{Latitude of the radar, in degrees. If missing taken from \code{pvol}.}

\item{antenna}{Numeric. Radar antenna height, in m. Default to antenna height
in \code{vp}.}

\item{beam_angle}{Numeric. Beam opening angle in degrees, typically the
angle between the half-power (-3 dB) points of the main lobe.}

\item{crs}{character or object of class CRS. PROJ.4 type description of a Coordinate Reference System (map projection).
When 'NA' (default), an azimuthal equidistant projection with origin at the radar location is used.
To use a WSG84 (lat,lon) projection, use crs="+proj=longlat +datum=WGS84"}

\item{param_ppi}{Character (vector). One or multiple of \code{VIR}, \code{VID}, \code{R},
\code{overlap}, \code{eta_sum} or \code{eta_sum_expected}.}

\item{k}{Numeric. Standard refraction coefficient.}

\item{re}{Numeric. Earth equatorial radius, in km.}

\item{rp}{Numeric. Earth polar radius, in km.}
}
\value{
A \code{ppi} object.
}
\description{
Estimates a spatial image of vertically integrated density (\code{vid}) based on
all elevation scans of the radar, while accounting for the changing overlap
between the radar beams as a function of range. The resulting \code{ppi} is a
vertical integration over the layer of biological scatterers based on all
available elevation scans, corrected for range effects due to partial beam
overlap with the layer of biological echoes (overshooting) at larger
distances from the radar. The methodology is described in detail in
Kranstauber et al. (2020).
}
\details{
The function requires:
\itemize{
\item A polar volume, containing one or multiple scans (\code{pvol}).
\item A vertical profile (of birds) calculated for that same polar volume (\code{vp}).
\item A grid defined on the earth's surface, on which we will calculate the range
corrected image (defined by \code{raster}, or a combination of \code{nx}, \code{ny},\code{res}
arguments).
}

The pixel locations on the ground are easily translated into a corresponding
azimuth and range of the various scans (see \code{\link[=beam_range]{beam_range()}}).

For each scan within the polar volume, the function calculates:
\itemize{
\item the vertical radiation profile for each ground surface pixel for that
particular scan, using \link{beam_profile}.
\item the reflectivity expected for each ground surface pixel
(\eqn{\eta_{expected}}), given the vertical profile (of biological
scatterers) and the part of the profile radiated by the beam.
This \eqn{\eta_{expected}} is simply the average of (linear) \code{eta} in the
profile, weighted by the vertical radiation profile.
\item the observed eta at each pixel \eqn{\eta_{observed}}, which is converted
form \code{DBZH} using function \link{dbz_to_eta}, with \code{DBZH} the reflectivity
factor measured at the pixel's distance from the radar.
\item The vertical radiation profile for each ground surface pixel for that
particular scan, using \code{\link[=beam_profile]{beam_profile()}}.
\item The reflectivity expected for each ground surface pixel
(\eqn{\eta_{expected}}), given the vertical profile (of biological
scatterers) and the part of the profile radiated by the beam. This
\eqn{\eta_{expected}} is simply the average of (linear) \code{eta} in the profile,
weighted by the vertical radiation profile.
\item The observed \code{eta} at each pixel \eqn{\eta_{observed}},
which is converted form \code{DBZH} using \code{\link[=dbz_to_eta]{dbz_to_eta()}}, with \code{DBZH} the
reflectivity factor measured at the pixel's distance from the radar.
}

If one of \code{lat} or \code{lon} is missing, the extent of the \code{ppi} is taken equal
to the extent of the data in the first scan of the polar volume.

To arrive at the final PPI image, the function calculates
\itemize{
\item the vertically integrated density (\code{vid}) and vertically integrated
reflectivity (\code{vir}) for the profile, using the function
\link{integrate_profile}.
\item the spatial range-corrected PPI for \code{VID}, defined as the adjustment
factor image (\code{R}), multiplied by the \code{vid} calculated for the profile
\item the spatial range-corrected PPI for \code{VIR}, defined as the adjustment
factor \code{R}, multiplied by the \code{vir} calculated for the profile.
}

Scans at 90 degree beam elevation (e.g. birdbath scans) are ignored.

#' @seealso
\itemize{
\item \code{\link[=summary.ppi]{summary.ppi()}}
\item \code{\link[=beam_profile]{beam_profile()}}
\item \code{\link[=beam_range]{beam_range()}}
\item \code{\link[=integrate_profile]{integrate_profile()}}
}
}
\examples{
\donttest{
# Locate and read the polar volume example file
pvolfile <- system.file("extdata", "volume.h5", package = "bioRad")

# load polar volume
pvol <- read_pvolfile(pvolfile)

# Read the corresponding vertical profile example
data(example_vp)

# Calculate the range-corrected ppi on a 50x50 pixel raster
ppi <- integrate_to_ppi(pvol, example_vp, nx = 50, ny = 50)

# Plot the vertically integrated reflectivity (VIR) using a
# 0-2000 cm^2/km^2 color scale
plot(ppi, zlim = c(0, 2000))

# Calculate the range-corrected ppi on finer 2000m x 2000m pixel raster
ppi <- integrate_to_ppi(pvol, example_vp, res = 2000)

# Plot the vertically integrated density (VID) using a
# 0-200 birds/km^2 color scale
plot(ppi, param = "VID", zlim = c(0, 200))

# Download a basemap and map the ppi
bm <- download_basemap(ppi)
map(ppi, bm)

# The ppi can also be projected on a user-defined raster, as follows:

# First define the raster
template_raster <- raster::raster(
  raster::extent(12, 13, 56, 57),
  crs = sp::CRS("+proj=longlat")
)

# Project the ppi on the defined raster
ppi <- integrate_to_ppi(pvol, example_vp, raster = template_raster)

# Extract the raster data from the ppi object
raster::brick(ppi$data)

# Calculate the range-corrected ppi on an even finer 500m x 500m pixel raster,
# cropping the area up to 50000 meter from the radar
ppi <- integrate_to_ppi(
  pvol, example_vp, res = 500,
  xlim = c(-50000, 50000), ylim = c(-50000, 50000)
)
plot(ppi, param = "VID", zlim = c(0, 200))
}
}
\references{
\itemize{
\item Kranstauber B, Bouten W, Leijnse H, Wijers B, Verlinden L, Shamoun-Baranes
J, Dokter AM (2020) High-Resolution Spatial Distribution of Bird Movements
Estimated from a Weather Radar Network. Remote Sensing 12 (4), 635.
\doi{https://doi.org/10.3390/rs12040635}
\item Buler JJ & Diehl RH (2009) Quantifying bird density during migratory
stopover using weather surveillance radar. IEEE Transactions on Geoscience
and Remote Sensing 47: 2741-2751.
\doi{https://doi.org/10.1109/TGRS.2009.2014463}
}

\itemize{
\item Kranstauber B, Bouten W, Leijnse H, Wijers B, Verlinden L, Shamoun-Baranes
J, Dokter AM (2020) High-Resolution Spatial Distribution of Bird
Movements Estimated from a Weather Radar Network. Remote Sensing 12 (4),
635. \doi{10.3390/rs12040635}
\item Buler JJ & Diehl RH (2009) Quantifying bird density during migratory
stopover using weather surveillance radar. IEEE Transactions on Geoscience
and Remote Sensing 47: 2741-2751.
\doi{10.1109/TGRS.2009.2014463}
}
}
