% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/operatingCharacteristics1.R
\name{operatingCharacteristics1}
\alias{operatingCharacteristics1}
\alias{opChar1}
\title{Calculate operating characteristics for group testing algorithms 
that use a single-disease assay}
\usage{
operatingCharacteristics1(
  algorithm,
  p = NULL,
  probabilities = NULL,
  Se = 0.99,
  Sp = 0.99,
  hier.config = NULL,
  rowcol.sz = NULL,
  alpha = 2,
  a = NULL,
  print.time = TRUE,
  ...
)

opChar1(
  algorithm,
  p = NULL,
  probabilities = NULL,
  Se = 0.99,
  Sp = 0.99,
  hier.config = NULL,
  rowcol.sz = NULL,
  alpha = 2,
  a = NULL,
  print.time = TRUE,
  ...
)
}
\arguments{
\item{algorithm}{character string defining the group testing algorithm to be 
used. Non-informative testing options include two-stage hierarchical ("\kbd{D2}"),
three-stage hierarchical ("\kbd{D3}"), four-stage hierarchical ("\kbd{D4}"), 
square array testing without master pooling ("\kbd{A2}"), and square array 
testing with master pooling ("\kbd{A2M}"). Informative testing options 
include two-stage hierarchical ("\kbd{ID2}"), three-stage hierarchical 
("\kbd{ID3}"), four-stage hierarchical ("\kbd{ID4}"), and square array testing 
without master pooling ("\kbd{IA2}").}

\item{p}{overall probability of disease that will be used to generate a
vector/matrix of individual probabilities. For non-informative algorithms, a
homogeneous set of probabilities will be used. For informative algorithms, the
\code{\link{expectOrderBeta}} function will be used to generate a heterogeneous
set of probabilities. Further details are given under 'Details'. Either 
\kbd{p} or \kbd{probabilities} should be specified, but not both.}

\item{probabilities}{a vector of individual probabilities, which is homogeneous
for non-informative testing algorithms and heterogeneous for informative
testing algorithms. Either  \kbd{p} or \kbd{probabilities} should be specified,
but not both.}

\item{Se}{a vector of sensitivity values, where one value is given for each 
stage of testing (in order). If a single value is provided, sensitivity 
values are assumed to be equal to this value for all stages of testing. 
Further details are given under 'Details'.}

\item{Sp}{a vector of specificity values, where one value is given for each 
stage of testing (in order). If a single value is provided, specificity 
values are assumed to be equal to this value for all stages of testing. 
Further details are given under 'Details'.}

\item{hier.config}{a matrix specifying the configuration for a hierarchical 
testing algorithm. The rows correspond to the stages of testing, the columns 
correspond to each individual to be tested, and the cell values 
specify the group number of each individual at each stage. Further details 
are given under 'Details'. For array testing algorithms, this argument will be 
ignored.}

\item{rowcol.sz}{the row/column size for array testing algorithms. For hierarchical 
testing algorithms, this argument will be ignored.}

\item{alpha}{a shape parameter for the beta distribution that specifies the degree of
heterogeneity for the generated probability vector (for informative testing only).}

\item{a}{a vector containing indices indicating which individuals to 
calculate individual accuracy measures for. If \kbd{NULL}, individual accuracy 
measures will be displayed for all individuals in the algorithm.}

\item{print.time}{a logical value indicating whether the length of time 
for calculations should be printed. The default is \kbd{TRUE}.}

\item{...}{arguments to be passed to the \code{\link{expectOrderBeta}} function, which
generates a vector of probabilities for informative testing algorithms.
Further details are given under 'Details'.}
}
\value{
A list containing:
\item{algorithm}{the group testing algorithm used for calculations.}
\item{prob}{the probability of disease or the vector of individual 
probabilities, as specified by the user.}
\item{alpha}{level of heterogeneity for the generated probability vector
(for informative testing only).}
\item{Se}{the vector of sensitivity values for each stage of testing.}
\item{Sp}{the vector of specificity values for each stage of testing.}
\item{Config}{a list specifying elements of the specified testing configuration, 
which may include:
\describe{
\item{Stage1}{group size for the first stage of hierarchical testing, if applicable.}
\item{Stage2}{group sizes for the second stage of hierarchical testing, if applicable.}
\item{Stage3}{group sizes for the third stage of hierarchical testing, if applicable.}
\item{Block.sz}{the block size/initial group size for informative Dorfman testing,
which is not tested.}
\item{pool.szs}{group sizes for the first stage of testing for informative Dorfman
testing.}
\item{Array.dim}{the row/column size for array testing.}
\item{Array.sz}{the overall array size for array testing (the square of the 
row/column size).}}}
\item{p.vec}{the sorted vector of individual probabilities, if applicable.}
\item{p.mat}{the sorted matrix of individual probabilities in gradient arrangement,
if applicable. Further details are given under 'Details'.}
\item{ET}{the expected testing expenditure to decode all individuals in the algorithm; 
this includes all individuals in all groups for hierarchical algorithms or in the
entire array for array testing.}
\item{value}{the value of the expected number of tests per individual.}
\item{Accuracy}{a list containing:
\describe{
\item{Individual}{a matrix of accuracy measures for each individual 
specified in \kbd{a}. The rows correspond to each unique set of accuracy 
measures in the algorithm. Individuals with the same set of accuracy 
measures are displayed together in a single row of the matrix. The columns 
correspond to the pooling sensitivity, pooling specificity, pooling positive 
predictive value, pooling negative predictive value, and the indices for the 
individuals in each row of the matrix.}
\item{Overall}{a matrix of overall accuracy measures for the algorithm. 
The columns correspond to the pooling sensitivity, pooling specificity, 
pooling positive predictive value, and pooling negative predictive value 
for the overall algorithm. Further details are given under 'Details'.}}}
}
\description{
Calculate operating characteristics, such as the expected 
number of tests, for a specified testing configuration using 
non-informative and informative hierarchical and array-based group testing 
algorithms. Single-disease assays are used at each stage of the algorithms.
}
\details{
This function computes the operating characteristics for 
group testing algorithms with an assay that tests for one disease, as 
described in Hitt et al. (2019).

Available algorithms include two-, three-, and four-stage hierarchical testing and
array testing with and without master pooling. Both non-informative and informative
group testing settings are allowed for each algorithm, except informative
array testing with master pooling is unavailable because this method has not
appeared in the group testing literature. Operating characteristics calculated are
expected number of tests, pooling sensitivity, pooling specificity, pooling
positive predictive value, and pooling negative predictive value for each individual.

For informative algorithms where the \kbd{p} argument is specified, the 
expected value of order statistics from a beta distribution are found. 
These values are used to represent disease risk probabilities for each 
individual to be tested. The beta distribution has two parameters: a mean 
parameter \kbd{p} (overall disease prevalence) and a shape parameter 
\kbd{alpha} (heterogeneity level). Depending on the specified \kbd{p}, 
\kbd{alpha}, and overall group size, simulation may be necessary to 
generate the vector of individual probabilities. This is done using 
\code{\link{expectOrderBeta}} and requires the user to set a seed to 
reproduce results. 
 
Informative two-stage hierarchical (Dorfman) testing is implemented via 
the pool-specific optimal Dorfman (PSOD) method described in McMahan et al. 
(2012a), where the greedy algorithm proposed for PSOD is replaced by 
considering all possible testing configurations. Informative array testing 
is implemented via the gradient method (the most efficient array design), 
where higher-risk individuals are grouped in the left-most columns of the 
array. For additional details on the gradient arrangement method for 
informative array testing, see McMahan et al. (2012b).

The sensitivity/specificity values are allowed to vary across stages of 
testing. For hierarchical testing, a different sensitivity/specificity value 
may be used for each stage of testing. For array testing, a different 
sensitivity/specificity value may be used for master pool testing (if included), 
row/column testing, and individual testing. The values must be specified 
in order of the testing performed. For example, values are specified 
as (stage 1, stage 2, stage 3) for three-stage hierarchical testing or 
(master pool testing, row/column testing, individual testing) for array 
testing with master pooling. A single sensitivity/specificity value may be 
specified instead. In this situation, sensitivity/specificity values for all 
stages are assumed to be equal.
 
The matrix specified  by \kbd{hier.config} defines the hierarchical group testing 
algorithm for \eqn{I} individuals. The rows of the matrix correspond to the stages 
\eqn{s=1,...,S} in the testing algorithm, and the columns correspond to individuals 
\eqn{i=1,...I}. The cell values within the matrix represent the group number of 
individual \eqn{i} at stage \eqn{s}. For three-stage, four-stage, and non-informative 
two-stage hierarchical testing, the first row of the matrix consists of all ones. 
This indicates that all individuals in the algorithm are tested together in a single 
group in the first stage of testing. For informative two-stage hierarchical testing, 
the initial group (block) is not tested. Thus, the first row of the matrix 
consists of the group numbers for each individual in the first stage of testing. For 
all hierarchical algorithms, the final row of the matrix denotes individual 
testing. Individuals who are not tested in a particular stage are represented 
by "NA" (e.g., an individual tested in a group of size 1 in the second stage of testing 
would not be tested again in a third stage of testing). It is important to note that 
this matrix represents the testing that could be performed if each group tests positively 
at each stage prior to the last. For more details on this matrix (called a group membership 
matrix), see Bilder et al. (2019). 

For array testing without master pooling, the \kbd{rowcol.sz} specified
represents the row/column size for initial (stage 1) testing. For array testing
with master pooling, the \kbd{rowcol.sz} specified represents the row/column size
for stage 2 testing. This is because the master pool size is the overall array 
size, given by the square of the row/column size.

The displayed overall pooling sensitivity, pooling specificity, pooling positive
predictive value, and pooling negative predictive value are weighted
averages of the corresponding individual accuracy measures for all
individuals within the initial group (or block) for a hierarchical algorithm, or
within the entire array for an array-based algorithm.
Expressions for these averages are provided in the Supplementary
Material for Hitt et al. (2019). These expressions are based on accuracy
definitions given by Altman and Bland (1994a, 1994b).

The \kbd{operatingCharacteristics1} function accepts additional arguments, 
namely \kbd{num.sim}, to be passed to the \code{\link{expectOrderBeta}} function, 
which generates a vector of probabilities for informative group testing algorithms. 
The \kbd{num.sim} argument specifies the number of simulations from the beta 
distribution when simulation is used. By default, 10,000 simulations are used.
}
\section{Note}{
 This function returns the pooling positive and negative 
predictive values for all individuals even though these measures are 
diagnostic specific; e.g., the pooling positive predictive value should 
only be considered for those individuals who have tested positive.

Additionally, only stage dependent sensitivity and specificity values are 
allowed within the program (no group within stage dependent values are 
allowed). See Bilder et al. (2019) for additional information.
}

\examples{
# Calculate the operating characteristics for non-informative
#   two-stage hierarchical (Dorfman) testing.
config.mat <- matrix(data = c(rep(1, 10), 1:10), 
                     nrow = 2, ncol = 10, byrow = TRUE)
opChar1(algorithm="D2", p=0.05, Se=0.99, Sp=0.99, 
        hier.config=config.mat)
opChar1(algorithm="D2", p=0.05, Se=0.99, Sp=0.99, 
        hier.config=config.mat, a=c(1,4), print.time=FALSE)

# Calculate the operating characteristics for informative
#   two-stage hierarchical (Dorfman) testing.
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta
#   distribution with p = 0.01 and a heterogeneity level
#   of alpha = 0.5.
config.mat <- matrix(data = c(rep(1:3, each = 10), 1:30), 
                     nrow = 2, ncol = 30, byrow = TRUE)
set.seed(52613)
opChar1(algorithm="ID2", p=0.01, Se=0.95, Sp=0.95, 
        hier.config=config.mat, alpha=0.5, num.sim=10000)
# Equivalent code using a heterogeneous vector of 
#   probabilities
set.seed(52613)
probs <- expectOrderBeta(p=0.01, alpha=0.5, grp.sz=30)
opChar1(algorithm="ID2", probabilities=probs, Se=0.95, Sp=0.95, 
        hier.config=config.mat)

# Calculate the operating characteristics for
#   non-informative three-stage hierarchical testing.
config.mat <- matrix(data = c(rep(1, 18), rep(1:3, each = 5), 
                              rep(4, 3), 1:18), 
                    nrow = 3, ncol = 18, byrow = TRUE)
opChar1(algorithm="D3", p=0.001, Se=0.95, Sp=0.95, 
        hier.config=config.mat)
opChar1(algorithm="D3", p=0.001, Se=c(0.95, 0.95, 0.99), 
        Sp=c(0.96, 0.96, 0.98), hier.config=config.mat)

# Calculate the operating characteristics for 
#   informative three-stage hierarchical testing, 
#   given a heterogeneous vector of probabilities.
config.mat <- matrix(data = c(rep(1, 6), rep(1:2, each = 3), 
                              1:6), nrow = 3, ncol = 6, 
                     byrow = TRUE)
set.seed(52613)
opChar1(algorithm="ID3", 
         probabilities=c(0.012, 0.014, 0.011, 0.012, 0.010, 0.015), 
         Se=0.99, Sp=0.99, hier.config=config.mat, 
         alpha=0.5, num.sim=5000)

# Calculate the operating characteristics for 
#   non-informative four-stage hierarchical testing.
config.mat <- matrix(data = c(rep(1, 12), rep(1, 8), 
                              rep(2, 2), 3, 4, rep(1, 5), 
                              rep(2, 3), 3, 4, rep(NA, 2), 
                              1:8, rep(NA, 4)), nrow = 4, 
                     ncol = 12, byrow = TRUE)
opChar1(algorithm="D4", p=0.041, Se=0.99, Sp=0.90, 
        hier.config=config.mat)
        
# Calculate the operating characteristics for 
#   informative four-stage hierarchical testing. 
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta
#   distribution with p = 0.041 and a heterogeneity level
#   of alpha = 0.5.
config.mat <- matrix(data = c(rep(1, 12), rep(1, 8), 
                              rep(2, 2), 3, 4, rep(1, 5), 
                              rep(2, 3), 3, 4, rep(NA, 2), 
                              1:8, rep(NA, 4)), nrow = 4, 
                     ncol = 12, byrow = TRUE)
set.seed(5678)
opChar1(algorithm="ID4", p=0.041, Se=0.99, Sp=0.90,
        hier.config=config.mat, alpha=0.5)

# Calculate the operating characteristics for
#   non-informative array testing without master pooling.
opChar1(algorithm="A2", p=0.005, Se=c(0.95, 0.99), 
        Sp=c(0.95, 0.99), rowcol.sz=8, a=1)

# Calculate the operating characteristics for 
#   informative array testing without master pooling.
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta
#   distribution with p = 0.03 and a heterogeneity level
#   of alpha = 2.
set.seed(1002)
opChar1(algorithm="IA2", p=0.03, Se=0.95, Sp=0.95,
         rowcol.sz=8, alpha=2, a=1:10)

# Calculate the operating characteristics for 
#   non-informative array testing with master pooling.
opChar1(algorithm="A2M", p=0.02, Se=c(0.95,0.95,0.99), 
        Sp=c(0.98,0.98,0.99), rowcol.sz=5)
}
\references{
\insertRef{Altman1994a}{binGroup2}

\insertRef{Altman1994b}{binGroup2}

\insertRef{Bilder2019}{binGroup2}

\insertRef{Hitt2019}{binGroup2}

\insertRef{McMahan2012a}{binGroup2}

\insertRef{McMahan2012b}{binGroup2}
}
\seealso{
Other operating characteristic functions: 
\code{\link{Sterrett}()},
\code{\link{halving}()},
\code{\link{operatingCharacteristics2}()}
}
\author{
Brianna D. Hitt
}
\concept{operating characteristic functions}
